/**********************************************************************************
 *
 * @file    irq.c
 * @brief   Interrupt handler
 *
 * @date    26 Dec. 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          26 Dec. 2022    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
uint8_t g_tx_buf[20];

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/**
  * @brief  格式化用户数据，使之符合传输协议
  * @param  srcdata-保存用户数据的首地址；
  *         desdata-保存处理好的数据的内存首地址(其内存大小至少大于前者5字节)；
  *         len-用户数据长度(单位:字节)
  * @retval 返回数据帧的总长度
  */
uint32_t data_initialize(uint8_t *desdata, uint8_t *srcdata, uint32_t len)
{
    uint32_t i;

    /* 帧头 */
    desdata[0] = 0xAA;
    desdata[1] = 0xAA;

    /* 功能码 */
    desdata[2] = 0xF1;

    /* 数据长度 */
    desdata[3] = len;

    /* 用户数据 */
    memcpy(desdata + 4, srcdata, len);

    /* 累加校验和 */
    desdata[len + 4] = 0U;

    for (i = 0; i < len + 4; i++)
    {
        /* 累加校验和, 高位舍去, 只保留低8位 */
        desdata[len + 4] += desdata[i];
    }

    return (len + 5);
}

/**
  * @brief  CLINT IRQ handler
  * @retval None
  */
void __attribute__((interrupt)) CLINT_Handler(void)
{
    /* Nested IRQ start : Save CSR and enable global interrupt. */
    MD_NEST_INT_START();

    csi_coret_clr(md_cmu_get_clock() / 1000, CLINT_IRQn);
    md_inc_tick();

    /* Nested IRQ end : Restore CSR and disable global interrupt. */
    MD_NEST_INT_END();

    return;
}

/**
  * @brief  ADC IRQ handler
  * @retval None
  */
void __attribute__((interrupt)) ADC_Handler(void)
{
    uint16_t cnt;
	uint32_t length, i;

    /* Nested IRQ start : Save CSR and enable global interrupt. */
    MD_NEST_INT_START();

    if (md_adc_is_conv_end_interrupt_enable(ADC) && md_adc_get_stat_nche(ADC))
    {
        md_adc_set_clr_nche(ADC);
        g_adc_result[0] = md_adc_get_normal_channel_val(ADC) * VDD / 4095;

        length = data_initialize(g_tx_buf, (uint8_t *)g_adc_result, 2);

        for (i = 0; i < length; i++)
        {
            md_uart_set_send_data8(UARTX, g_tx_buf[i]);

            cnt = 4000U;
            while (!md_uart_is_active_it_tbc(UARTX) && (--cnt));

            md_uart_clear_it_tbc(UARTX);
        }
    }

    /* Nested IRQ end : Restore CSR and disable global interrupt. */
    MD_NEST_INT_END();

    return;
}

/**
  * @}
  */

/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
