/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    16 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          16 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
ald_uart_handle_t g_h_uart;
ald_pis_modulate_config_t g_modu_conf;
ald_pis_handle_t g_h_pis;

uint8_t g_tx_buf[32] = "essemi UART DEMO: from MCU\r\n";
uint8_t g_rx_buf[32];

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void init_uart_pin(void)
{
    ald_gpio_init_t gpio_init;
    memset(&gpio_init, 0x00, sizeof(ald_gpio_init_t));

    /* Initialize tx pin: PA2 */
    gpio_init.mode = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od   = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd = ALD_GPIO_PUSH_UP;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_2;
    ald_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    /* Initialize rx pin: PA3 */
    gpio_init.mode = ALD_GPIO_MODE_INPUT;
    gpio_init.od   = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd = ALD_GPIO_PUSH_UP;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_2;
    ald_gpio_init(UART_RX_PORT, UART_RX_PIN, &gpio_init);

    return;
}

/**
  * @brief  Initialize TIMER.
  * @param  None
  * @retval None
  */
void timer_init(void)
{
    ald_timer_handle_t h_tim;
    ald_timer_oc_init_t tim_ocinit;
    memset(&h_tim, 0x0, sizeof(ald_timer_handle_t));
    memset(&tim_ocinit, 0x0, sizeof(ald_timer_oc_init_t));

    /* Initialize Timer */
    h_tim.perh           = AD16C4T;
    h_tim.init.prescaler = 0;
    h_tim.init.mode      = ALD_TIMER_CNT_MODE_UP;
    h_tim.init.period    = 1262;
    h_tim.init.clk_div   = ALD_TIMER_CLOCK_DIV1;
    h_tim.init.re_cnt    = 0;
    ald_timer_pwm_init(&h_tim);

    /* Common configuration for channel 1 */
    tim_ocinit.oc_mode      = ALD_TIMER_OC_MODE_PWM1;
    tim_ocinit.oc_polarity  = ALD_TIMER_OC_POLARITY_HIGH;
    tim_ocinit.oc_fast_en   = DISABLE;
    tim_ocinit.ocn_polarity = ALD_TIMER_OCN_POLARITY_HIGH;
    tim_ocinit.ocn_idle     = ALD_TIMER_OCN_IDLE_RESET;
    tim_ocinit.pulse = 631;
    ald_timer_oc_config_channel(&h_tim, &tim_ocinit, ALD_TIMER_CHANNEL_1);

    /* Start input pwm from tim0 channel 1 */
    ald_timer_oc_start(&h_tim, ALD_TIMER_CHANNEL_1);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    uint8_t tx_len = strlen((char *) g_tx_buf);
    uint8_t rx_len = sizeof(g_rx_buf);

    /* Initialize ALD */
    ald_cmu_init();

    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_48M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 48000000);

    /* Enable all peripherals */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    ald_mcu_irq_config(CUART1_IRQn, 4, ENABLE);
    __enable_irq();

    memset(&g_modu_conf, 0x0, sizeof(ald_pis_modulate_config_t));
    memset(&g_h_uart, 0x0, sizeof(ald_uart_handle_t));

    g_h_pis.perh        = PIS;
    g_modu_conf.target  = ALD_PIS_UART3_TX;
    g_modu_conf.src     = ALD_PIS_SRC_TIMER0;
    g_modu_conf.level   = ALD_PIS_LOW_LEVEL;
    g_modu_conf.channel = ALD_PIS_TIMER_CH1;
    ald_pis_modu_config(&g_h_pis, &g_modu_conf);

    timer_init();

    /* Initialize pin */
    init_uart_pin();

    /* Initialize uart */
    g_h_uart.perh             = UARTX;
    g_h_uart.init.baud        = 1200;
    g_h_uart.init.word_length = ALD_UART_WORD_LENGTH_8B;
    g_h_uart.init.stop_bits   = ALD_UART_STOP_BITS_1;
    g_h_uart.init.parity      = ALD_UART_PARITY_NONE;
    g_h_uart.init.mode        = ALD_UART_MODE_UART;
    g_h_uart.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    ald_uart_init(&g_h_uart);

    ald_uart_send_by_it(&g_h_uart, g_tx_buf, tx_len);
    ald_uart_recv_by_it(&g_h_uart, g_rx_buf, rx_len);

    while (1)
    {
        ald_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
