/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    14 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          14 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
ald_uart_handle_t g_uart_init;

uint8_t g_rx_buf[256];
uint32_t g_rx_len = sizeof(g_rx_buf);

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void init_uart_pin(void)
{
    ald_gpio_init_t gpio_init;
    memset(&gpio_init, 0x00, sizeof(ald_gpio_init_t));

    /* Initialize tx pin: PB5 */
    gpio_init.mode = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od   = ALD_GPIO_OPEN_DRAIN;
    gpio_init.pupd = ALD_GPIO_PUSH_UP;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_5;
    ald_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    return;
}

/**
  * @brief  Send message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_send_complete(ald_uart_handle_t *arg)
{
    uint16_t cnt = 4000U;

    while (ald_uart_get_status(&g_uart_init, ALD_UART_STATUS_TSBUSY) && (--cnt));

    ALD_UART_TX_DISABLE(&g_uart_init);
    ALD_UART_RX_ENABLE(&g_uart_init);
    ald_uart_recv_by_it(arg, g_rx_buf, 1);

    return;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_recv_complete(ald_uart_handle_t *arg)
{
    uint16_t cnt;
    static uint32_t s_i;
    s_i++;

    if ((s_i > 1) && (*(arg->rx_buf - 2) == 0x0D) && (*(arg->rx_buf - 1) == 0x0A))
    {
        cnt = 4000;

        while (ald_uart_get_status(&g_uart_init, ALD_UART_STATUS_RSBUSY) && (--cnt));

        ald_delay_1us(70);

        ALD_UART_RX_DISABLE(&g_uart_init);
        ALD_UART_TX_ENABLE(&g_uart_init);
        ald_uart_send_by_it(&g_uart_init, g_rx_buf, s_i);
        s_i = 0;
    }
    else
    {
        if (s_i >= g_rx_len)
            s_i = 0;

        ald_uart_recv_by_it(arg, g_rx_buf + s_i, 1);
    }

    return;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_error(ald_uart_handle_t *arg)
{
    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();

    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable all peripherals */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    ald_mcu_irq_config(EUART0_IRQn, 4, ENABLE);
    __enable_irq();

    /* Initialize pin */
    init_uart_pin();

    memset(&g_uart_init, 0x00, sizeof(ald_uart_handle_t));
    /* Initialize uart */
    g_uart_init.perh             = UARTX;
    g_uart_init.init.baud        = 115200;
    g_uart_init.init.word_length = ALD_UART_WORD_LENGTH_8B;
    g_uart_init.init.stop_bits   = ALD_UART_STOP_BITS_1;
    g_uart_init.init.parity      = ALD_UART_PARITY_NONE;
    g_uart_init.init.mode        = ALD_UART_MODE_HDSEL;
    g_uart_init.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    g_uart_init.rx_cplt_cbk      = uart_recv_complete;
    g_uart_init.tx_cplt_cbk      = uart_send_complete;
    ald_uart_init(&g_uart_init);

    ald_uart_recv_by_it(&g_uart_init, g_rx_buf, 1);

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
