/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    15 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          15 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
ald_timer_handle_t g_ad16c4t_init;
ald_gpio_init_t g_gpio_init;
ald_timer_clock_config_t g_ad16c4t_clock_config;

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/**
  * @brief  ald timer compare elapsed callback
  * @param  arg: Pointer to timer_handle_t structure.
  * @retval None.
  */
void ald_timer_compare_callback(struct ald_timer_handle_s *arg)
{
    switch (arg->ch)
    {
        case ALD_TIMER_ACTIVE_CHANNEL_1:
            __NOP();
            break;

        case ALD_TIMER_ACTIVE_CHANNEL_2:
            __NOP();
            break;

        case ALD_TIMER_ACTIVE_CHANNEL_3:
            __NOP();
            break;

        case ALD_TIMER_ACTIVE_CHANNEL_4:
            __NOP();
            break;

        default:
            break;
    }
}

/**
  * @brief  Initializate pin of module.
  * @retval None
  */
static void init_gpio(void)
{
    ald_gpio_init_t gpio_init;

    memset(&gpio_init, 0x0, sizeof(gpio_init));
    gpio_init.mode = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od   = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd = ALD_GPIO_FLOATING;
    gpio_init.func = ALD_GPIO_FUNC_5;
    ald_gpio_init(CH3_PORT, CH3_PIN, &gpio_init);
    ald_gpio_init(CH1N_PORT, CH1N_PIN, &gpio_init);
    ald_gpio_init(CH2N_PORT, CH2N_PIN, &gpio_init);
    ald_gpio_init(CH3N_PORT, CH3N_PIN, &gpio_init);

    gpio_init.func = ALD_GPIO_FUNC_3;
    ald_gpio_init(CH1_PORT, CH1_PIN, &gpio_init);
    ald_gpio_init(CH2_PORT, CH2_PIN, &gpio_init);

    gpio_init.mode = ALD_GPIO_MODE_INPUT;
    gpio_init.pupd = ALD_GPIO_PUSH_DOWN;
    gpio_init.func = ALD_GPIO_FUNC_4;
    ald_gpio_init(AD16C4T_ET_PORT, AD16C4T_ET_PIN, &gpio_init);

    return;
}

/**
  * @brief:  Initialize AD16C4T to 6 PWM funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    ald_timer_oc_init_t oc_init;
    ald_timer_clear_input_config_t timer_clear_input_config;

    /* Initialize AD16C4T */
    memset(&g_ad16c4t_init, 0x0, sizeof(g_ad16c4t_init));
    g_ad16c4t_init.perh = AD16C4T;
    g_ad16c4t_init.init.prescaler    = 72 - 1;                 /* clk_count: 72 */
    g_ad16c4t_init.init.mode         = ALD_TIMER_CNT_MODE_CENTER1; /* count mode is center1 */
    g_ad16c4t_init.init.period       = 5000 - 1;              /* period is 5000 count clock */
    g_ad16c4t_init.init.clk_div      = ALD_TIMER_CLOCK_DIV1;       /* working clock of dead time and filter */
    g_ad16c4t_init.init.re_cnt       = 0;                      /* no repeat count */
    g_ad16c4t_init.delay_elapse_cbk = ald_timer_compare_callback; /* compare elapsed callback */
    ald_timer_oc_init(&g_ad16c4t_init);

    /* Initialize clock source */
    g_ad16c4t_clock_config.source = ALD_TIMER_SRC_INTER;   /* clock sources is internal */
    ald_timer_config_clock_source(&g_ad16c4t_init, &g_ad16c4t_clock_config);

    /* Preload register on CCVAL1/CCVAL2/CCVAL3 enabled */
	MODIFY_REG(g_ad16c4t_init.perh->CHMR1, TIMER_CHMR1_CH1OPREN_MSK, ENABLE << TIMER_CHMR1_CH1OPREN_POS);
	MODIFY_REG(g_ad16c4t_init.perh->CHMR1, TIMER_CHMR1_CH2OPEN_MSK, ENABLE << TIMER_CHMR1_CH2OPEN_POS);
	MODIFY_REG(g_ad16c4t_init.perh->CHMR2, TIMER_CHMR2_CH3OPEN_MSK, ENABLE << TIMER_CHMR2_CH3OPEN_POS);

    /* Compare output configuration for all channels */
    memset(&oc_init, 0x0, sizeof(oc_init));
    oc_init.oc_mode      = ALD_TIMER_OC_MODE_PWM2;      /* compare output mode is PWM2 */
    oc_init.oc_polarity  = ALD_TIMER_OC_POLARITY_HIGH;  /* CHxO compare output polarity is high */
    oc_init.ocn_polarity = ALD_TIMER_OCN_POLARITY_HIGH; /* CHxON compare output polarity is high */
    oc_init.ocn_idle     = ALD_TIMER_OCN_IDLE_RESET;    /* CHxO idle is low */
    oc_init.oc_idle      = ALD_TIMER_OC_IDLE_RESET;     /* CHxON idle is low */

    /* Set the compare value for channel 1 */
    oc_init.pulse = 1500 - 1;
    ald_timer_oc_config_channel(&g_ad16c4t_init, &oc_init, ALD_TIMER_CHANNEL_1);
    /* Set the compare value for channel 2 */
    oc_init.pulse = 2500 - 1;
    ald_timer_oc_config_channel(&g_ad16c4t_init, &oc_init, ALD_TIMER_CHANNEL_2);
    /* Set the compare value for channel 3 */
    oc_init.pulse = 3500 - 1;
    ald_timer_oc_config_channel(&g_ad16c4t_init, &oc_init, ALD_TIMER_CHANNEL_3);

    /* Select OC clear source: ETFP */
    /* Select ET polarity: high level or rise edge */
    /* Enable OC1 clear */
    memset(&timer_clear_input_config, 0x0, sizeof(ald_timer_clear_input_config_t));
    timer_clear_input_config.clrsel   = ALD_TIMER_CHNREF_CLR_ETF;
    timer_clear_input_config.polarity = ALD_TIMER_POLARITY_NO_INV;
    timer_clear_input_config.state    = ENABLE;
    ald_timer_config_oc_ref_clear(&g_ad16c4t_init, &timer_clear_input_config, ALD_TIMER_CHANNEL_1);

    /* Start compare output from channel 1 */
    ald_timer_oc_start_by_it(&g_ad16c4t_init, ALD_TIMER_CHANNEL_1);
    /* Start compare output from channel 2 */
    ald_timer_oc_start_by_it(&g_ad16c4t_init, ALD_TIMER_CHANNEL_2);
    /* Start compare output from channel 3 */
    ald_timer_oc_start_by_it(&g_ad16c4t_init, ALD_TIMER_CHANNEL_3);
    /* Start compare output from channel 1N */
    ald_timer_ocn_start_by_it(&g_ad16c4t_init, ALD_TIMER_CHANNEL_1);
    /* Start compare output from channel 2N */
    ald_timer_ocn_start_by_it(&g_ad16c4t_init, ALD_TIMER_CHANNEL_2);
    /* Start compare output from channel 3N */
    ald_timer_ocn_start_by_it(&g_ad16c4t_init, ALD_TIMER_CHANNEL_3);

    /* Enable AD16C4T CC IRQ handler */
    ald_mcu_irq_config(AD16C4T_CC_IRQn, 4, ENABLE);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable ALL peripheral */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    __enable_irq();

    /* Initialize pin */
    init_gpio();

    /* Initialize timer */
    init_timer();

    while (1)
    {
        ald_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
