/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    15 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          15 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
#define BUF_LEN 10

/* Private Variables --------------------------------------------------------- */
ald_timer_handle_t g_ad16c4t_init, g_ad16c4t_init_2;
ald_timer_ic_init_t g_ic_init;
ald_timer_clock_config_t g_ad16c4t_clock_config;
ald_gpio_init_t g_gpio_init;

/* duty/cycle data buffer */
uint16_t g_duty_buffer[BUF_LEN] = {0U};
uint16_t g_cycle_buffer[BUF_LEN] = {0U};

/* Public Variables ---------------------------------------------------------- */
uint8_t g_cplt_cnt = 0U;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/**
  * @brief  DMA callback function
  * @retval None
  */
void ald_dma_cplt_callback(ald_timer_handle_t *tim)
{
    if (ald_dma_get_flag_status(ALD_DMA_CH_0, ALD_DMA_IT_FLAG_TC))
    {
        ALD_TIMER_CCx_DISABLE(tim, ALD_TIMER_CHANNEL_1);
        ald_timer_clear_flag_status(tim, ALD_TIMER_FLAG_CC1);
        g_cplt_cnt++;
    }

    if (ald_dma_get_flag_status(ALD_DMA_CH_1, ALD_DMA_IT_FLAG_TC))
    {
        ALD_TIMER_CCx_DISABLE(tim, ALD_TIMER_CHANNEL_2);
        ald_timer_clear_flag_status(tim, ALD_TIMER_FLAG_CC2);
        g_cplt_cnt++;
    }

    return;
}

/**
  * @brief  Initialize TOGGLE IO.
  * @param  None.
  * @retval None.
  */
void init_gpio(void)
{
    /* Initialize capture io */
    memset(&g_gpio_init, 0x0, sizeof(g_gpio_init));
    g_gpio_init.mode = ALD_GPIO_MODE_INPUT;
	g_gpio_init.od   = ALD_GPIO_PUSH_PULL;
	g_gpio_init.pupd = ALD_GPIO_FLOATING;
	g_gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    g_gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    g_gpio_init.type = ALD_GPIO_TYPE_CMOS;
    g_gpio_init.func = ALD_GPIO_FUNC_3;
    ald_gpio_init(CAPTURE_CH1_PORT, CAPTURE_CH1_PIN, &g_gpio_init);
}

/**
  * @brief:  Initialize AD16C4T to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    /* Initialize AD16C4T */
    memset(&g_ad16c4t_init, 0x0, sizeof(g_ad16c4t_init));   /* initialize the g_ad16c4t_init */
    g_ad16c4t_init.perh = AD16C4T;
    g_ad16c4t_init.init.prescaler    = 72 - 1;                  /* clk_count: 1MHz */
    g_ad16c4t_init.init.mode         = ALD_TIMER_CNT_MODE_UP;   /* count up */
    g_ad16c4t_init.init.period       = 0xFFFF;                  /* period is max count clock */
    g_ad16c4t_init.init.clk_div      = ALD_TIMER_CLOCK_DIV1;    /* working clock of dead time and filter */
    g_ad16c4t_init.init.re_cnt       = 0;                       /* 1 repeat count */
    g_ad16c4t_init.capture_cbk       = ald_dma_cplt_callback;
    ald_timer_base_init(&g_ad16c4t_init);
    /* Initialize clock source */
    g_ad16c4t_clock_config.source = ALD_TIMER_SRC_INTER;        /**< TIMER clock sources */
    ald_timer_config_clock_source(&g_ad16c4t_init, &g_ad16c4t_clock_config);

    /* Initialize input capture channel 1 */
    g_ic_init.polarity = ALD_TIMER_IC_POLARITY_RISE;
    g_ic_init.sel      = ALD_TIMER_IC_SEL_DIRECT;
    g_ic_init.psc      = ALD_TIMER_IC_PSC_DIV1;
    g_ic_init.filter   = 0;
    ald_timer_ic_config_channel(&g_ad16c4t_init, &g_ic_init, ALD_TIMER_CHANNEL_1);

    /* Initialize input capture channel 2 */
    g_ic_init.polarity = ALD_TIMER_IC_POLARITY_FALL;
    g_ic_init.sel      = ALD_TIMER_IC_SEL_INDIRECT;
    g_ic_init.psc      = ALD_TIMER_IC_PSC_DIV1;
    g_ic_init.filter   = 0;
    ald_timer_ic_config_channel(&g_ad16c4t_init, &g_ic_init, ALD_TIMER_CHANNEL_2);

    ald_timer_pwm_set_input(&g_ad16c4t_init, ALD_TIMER_CHANNEL_1);
    MODIFY_REG(g_ad16c4t_init.perh->CCEP, TIMER_CCEP_CC1POL_MSK, ALD_TIMER_IC_POLARITY_RISE << TIMER_CCEP_CC1POL_POS);
    MODIFY_REG(g_ad16c4t_init.perh->CCEP, TIMER_CCEP_CC1NPOL_MSK, ALD_TIMER_IC_POLARITY_RISE << TIMER_CCEP_CC1NPOL_POS);
    MODIFY_REG(g_ad16c4t_init.perh->CCEP, TIMER_CCEP_CC2POL_MSK, ALD_TIMER_IC_POLARITY_FALL << TIMER_CCEP_CC2POL_POS);
    MODIFY_REG(g_ad16c4t_init.perh->CCEP, TIMER_CCEP_CC2NPOL_MSK, ALD_TIMER_IC_POLARITY_RISE << TIMER_CCEP_CC2NPOL_POS);

    if (g_ad16c4t_init.hdma1.perh == NULL)
        g_ad16c4t_init.hdma1.perh = DMA;

    g_ad16c4t_init_2 = g_ad16c4t_init;
    g_ad16c4t_init_2.state = ALD_TIMER_STATE_READY;

    ald_timer_ic_start(&g_ad16c4t_init_2, ALD_TIMER_CHANNEL_2);
    ald_timer_ic_start_by_dma(&g_ad16c4t_init_2, ALD_TIMER_CHANNEL_2, g_duty_buffer, BUF_LEN, ALD_DMA_CH_1);

    ald_timer_ic_start(&g_ad16c4t_init, ALD_TIMER_CHANNEL_1);
    ald_timer_ic_start_by_dma(&g_ad16c4t_init, ALD_TIMER_CHANNEL_1, g_cycle_buffer, BUF_LEN, ALD_DMA_CH_0);

    ald_timer_clear_flag_status(&g_ad16c4t_init_2, ALD_TIMER_FLAG_CC1);
    ald_timer_clear_flag_status(&g_ad16c4t_init_2, ALD_TIMER_FLAG_CC2);

    return;
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable ALL peripheral */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    __enable_irq();

    /* Initialize pin */
    init_gpio();

    /* Initialize timer */
    init_timer();

    while (1)
    {
        if (g_cplt_cnt == 2)
        {
            g_cplt_cnt = 0;
            init_timer();
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
