/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    15 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          15 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
ald_timer_handle_t g_ad16c4t_init;
ald_gpio_init_t g_gpio_init;
ald_timer_clock_config_t g_ad16c4t_clock_config;

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/**
  * @brief  ald timer period elapsed callback
  * @param  arg: Pointer to timer_handle_t structure.
  * @retval None.
  */
void ald_timer_period_elapsed_callback(struct ald_timer_handle_s *arg)
{
    ald_gpio_toggle_pin(TOGGLE_PORT, TOGGLE_PIN);          /* toggle io */
}

/**
  * @brief  Initializate pin of module.
  * @retval None
  */
static void init_gpio(void)
{
    ald_gpio_init_t gpio_init;

    /* initialize the gpio_init */
    memset(&gpio_init, 0x0, sizeof(gpio_init));
    /* Initialize LED pin */
    gpio_init.mode  = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od    = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd  = ALD_GPIO_FLOATING;
    gpio_init.odrv  = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type  = ALD_GPIO_TYPE_CMOS;
    gpio_init.func  = ALD_GPIO_FUNC_1;
    ald_gpio_init(TOGGLE_PORT, TOGGLE_PIN, &gpio_init);

    /* Initialize ET IO */
    memset(&g_gpio_init, 0x0, sizeof(g_gpio_init)); /* initialize the g_gpio_init */
    g_gpio_init.mode = ALD_GPIO_MODE_INPUT;             /* input */
    g_gpio_init.func = ALD_GPIO_FUNC_4;                 /* GPIO function 4 */
    ald_gpio_init(ET_PORT, ET_PIN, &g_gpio_init);

    return;
}

/**
  * @brief:  Initialize AD16C4T to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    /* Initialize AD16C4T1 */
    memset(&g_ad16c4t_init, 0x0, sizeof(g_ad16c4t_init));
    g_ad16c4t_init.perh = AD16C4T;
    g_ad16c4t_init.init.prescaler    = 1 - 1;
    g_ad16c4t_init.init.mode         = ALD_TIMER_CNT_MODE_UP;
    g_ad16c4t_init.init.period       = 100 - 1;
    g_ad16c4t_init.init.clk_div      = ALD_TIMER_CLOCK_DIV1;
    g_ad16c4t_init.init.re_cnt       = 1 - 1;
    g_ad16c4t_init.period_elapse_cbk = ald_timer_period_elapsed_callback;
    ald_timer_base_init(&g_ad16c4t_init);

    /* Initialize clock source */
    g_ad16c4t_clock_config.source   = ALD_TIMER_SRC_ETRMODE2;      /**< TIMER clock sources */
    g_ad16c4t_clock_config.polarity = ALD_TIMER_CLK_POLARITY_RISE; /**< TIMER clock polarity */
    g_ad16c4t_clock_config.psc      = ALD_TIMER_ETR_PSC_DIV1;      /**< TIMER clock prescaler */
    g_ad16c4t_clock_config.filter   = 0;                       /**< TIMER clock filter */
    ald_timer_config_clock_source(&g_ad16c4t_init, &g_ad16c4t_clock_config);

    /* Enable AD16C4T1 UPDATE interrupt */
    ald_mcu_irq_config(AD16C4T_UP_TRIG_COM_IRQn, 4, ENABLE);
    /* Start by UPDATE interrupt */
    ald_timer_base_start_by_it(&g_ad16c4t_init);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable ALL peripheral */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    __enable_irq();

    /* Initialize pin */
    init_gpio();

    /* Initialize timer */
    init_timer();

    while (1)
    {
        ald_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
