/**********************************************************************************
 *
 * @file    wm_i2s.c
 * @brief   I2S driver for WM8978
 *
 * @date    10 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          10 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */
#include "main.h"
#include "wm8978.h"
#include "wm_i2s.h"

/* Private Macros ----------------------------------------------------------- */
#define LOG_E
#define LOG_I
#define DMA_MSEL_SPIX ALD_DMA_MSEL_SPI0
/* Private Variables--------------------------------------------------------- */
static ald_i2s_handle_t s_h_i2s;
static void *s_i2s_buffers[2];
static uint16_t s_buffer_length;
static func_I2S_callback s_feed_data;
static func_I2S_callback s_recv_data;
static uint32_t s_i2s_send_dma_ch, s_i2s_recv_dma_ch;
static uint32_t s_status = 0;

/* Public Variables -------------------------------------------------------- */
/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */
void i2s_dma_config(uint32_t is_recv, uint32_t ch);
/* Private Function---------------------------------------------------------- */
void wm_set_vol(uint32_t type, uint8_t vol)
{
    if (type & 0x1)
        WM8978_HPvol_Set(vol, vol);   /* Headphone volume */

    if (type & 0x2)
        WM8978_SPKvol_Set(vol);      /* Speaker volume */

    if (type & 0x4)
        WM8978_MIC_Gain(vol); /* Microphone gain */

    if (type & 0x8)
        WM8978_LINEIN_Gain(vol); /* Linein gain (not used) */
}

void wm_set_mode(uint32_t mode)
{
    if (mode == WM_SEND)
    {
        WM8978_ADDA_Cfg(1, 0);                              /*Open the DAC*/
        WM8978_Input_Cfg(0, 0, 0);                          /*Close the input channel*/
        WM8978_Output_Cfg(1, 0);                            /*Open the DAC output*/

    }
    else if (mode == WM_RECV)
    {
        WM8978_ADDA_Cfg(0, 1);
        WM8978_Input_Cfg(1, 1, 0);
        WM8978_Output_Cfg(0, 1);
    }
    else if (mode == WM_SENDRECV)
    {
        WM8978_ADDA_Cfg(1, 1);
        WM8978_Input_Cfg(1, 1, 0);
        WM8978_Output_Cfg(1, 0);
    }
	else
	{

	}
}

void wm_set_callback(uint32_t is_recv, func_I2S_callback f)
{
    if (is_recv)
        s_recv_data = f;
    else
        s_feed_data = f;
}

void wm_start(void)
{
    SET_BIT(SPI0->I2SCFG, SPI_I2SCFG_I2SE_MSK);
    s_status = 1;
}

void wm_stop(void)
{
    CLEAR_BIT(SPI0->I2SCFG, SPI_I2SCFG_I2SE_MSK);
    s_status = 0;
}

void wm_pause(void)
{
    if (s_status)
        CLEAR_BIT(SPI0->I2SCFG, SPI_I2SCFG_I2SE_MSK);
}

void wm_resume(void)
{
    if (s_status)
        SET_BIT(SPI0->I2SCFG, SPI_I2SCFG_I2SE_MSK);
}

uint32_t wm_status(void)
{
    return s_status;
}

int wm_set_sample_rate(uint32_t rate)
{
    if (rate != s_h_i2s.init.sampling)
    {
        CLEAR_BIT(SPI0->I2SCFG, SPI_I2SCFG_I2SE_MSK);
        s_h_i2s.init.sampling = rate;

        if (ald_i2s_init(&s_h_i2s))
            return 1;
    }

    return 0;
}
/**
  * @brief  Init I2S for WM8978
  * @retval None
  */
int wm_i2s_init(struct wm_i2s_info *wm_info)
{
    s_h_i2s.perh            = wm_info->spi_x;
    s_h_i2s.init.ch_len     = ALD_I2S_WIDE_16;
    s_h_i2s.init.data_len   = ALD_I2S_LEN_16;
    s_h_i2s.init.polarity   = ALD_I2S_INACTIVE_LOW;
    s_h_i2s.init.standard   = ALD_I2S_STD_PHI;
    s_h_i2s.init.ext_clk_en = DISABLE;
    s_h_i2s.init.ext_clk    = 0;
    s_h_i2s.init.mck_en     = ENABLE;
    s_h_i2s.init.sampling   = wm_info->sample_rate;

    if (ald_i2s_init(&s_h_i2s))
        return 1;

    s_i2s_buffers[0] = wm_info->buffers[0];
    s_i2s_buffers[1] = wm_info->buffers[1];
    s_buffer_length = wm_info->buffer_length > 1024 ? 1024 : wm_info->buffer_length;
    s_feed_data = wm_info->feed_data;
    s_recv_data = wm_info->recv_data;

    s_i2s_send_dma_ch = wm_info->dma_tx_ch;
    s_i2s_recv_dma_ch = wm_info->dma_rx_ch;

    CLEAR_BIT(SPI0->I2SCFG, SPI_I2SCFG_I2SE_MSK);
    MODIFY_REG(SPI0->I2SCFG, SPI_I2SCFG_I2SCFG_MSK, ALD_I2S_MASTER_DUPLEX << SPI_I2SCFG_I2SCFG_POSS);
    SET_BIT(SPI0->I2SPR, SPI_I2SPR_MCKOE_MSK);

    i2s_dma_config(0, s_i2s_send_dma_ch);
    ald_dma_interrupt_config(s_i2s_send_dma_ch, ALD_DMA_IT_FLAG_TC, ENABLE);
    ald_dma_interrupt_config(s_i2s_send_dma_ch, ALD_DMA_IT_FLAG_HT, ENABLE);
    SET_BIT(SPI0->CON2, SPI_CON2_TXDMA_MSK);
    SET_BIT(DMA->CHANNEL[s_i2s_send_dma_ch].CON, DMA_CON_CHEN_MSK);

    i2s_dma_config(1, s_i2s_recv_dma_ch);
    ald_dma_interrupt_config(s_i2s_recv_dma_ch, ALD_DMA_IT_FLAG_TC, ENABLE);
    ald_dma_interrupt_config(s_i2s_recv_dma_ch, ALD_DMA_IT_FLAG_HT, ENABLE);
    SET_BIT(SPI0->CON2, SPI_CON2_RXDMA_MSK);
    SET_BIT(DMA->CHANNEL[s_i2s_recv_dma_ch].CON, DMA_CON_CHEN_MSK);
    return 0;
}
/**
  * @brief  callback to set I2Ssend data
  * @retval None
  */
void i2s_dma_send_callback(void *arg)
{
    s_feed_data(arg, s_buffer_length / 2);
}
/**
  * @brief  callback to receive i2s data
  * @retval None
  */
void i2s_dma_recv_callback(void *arg)
{
    s_recv_data(arg, s_buffer_length / 2);
}
/**
  * @brief  setup dma for i2s
  * @retval None
  */
void i2s_dma_config(uint32_t is_recv, uint32_t ch)
{
    ald_dma_config_t dma_config;

    if (is_recv)
    {
        g_dma_tc_callbacks[ch] = i2s_dma_recv_callback;
        g_dma_ht_callbacks[ch] = i2s_dma_recv_callback;
        g_dma_tc_callback_datas[ch] = s_i2s_buffers[1] + s_buffer_length / 2 * 2;
        g_dma_ht_callback_datas[ch] = s_i2s_buffers[1];
        dma_config.src            = (void *)&SPI0->DATA;
        dma_config.dst            = s_i2s_buffers[1];
        dma_config.msigsel        = ALD_DMA_MSIGSEL_SPI_RNR;
        dma_config.src_inc        = ALD_DMA_DATA_INC_DISABLE;
        dma_config.dst_inc        = ALD_DMA_DATA_INC_ENABLE;
    }
    else
    {
        g_dma_tc_callbacks[ch] = i2s_dma_send_callback;
        g_dma_ht_callbacks[ch] = i2s_dma_send_callback;
        g_dma_tc_callback_datas[ch] = s_i2s_buffers[0] + s_buffer_length / 2 * 2;
        g_dma_ht_callback_datas[ch] = s_i2s_buffers[0];
        dma_config.src            = s_i2s_buffers[0];
        dma_config.dst            = (void *)&SPI0->DATA;
        dma_config.msigsel        = ALD_DMA_MSIGSEL_SPI_TXEMPTY;
        dma_config.src_inc        = ALD_DMA_DATA_INC_ENABLE;
        dma_config.dst_inc        = ALD_DMA_DATA_INC_DISABLE;
    }

    dma_config.channel        = ch;
    dma_config.size           = s_buffer_length;
    dma_config.src_data_width = ALD_DMA_DATA_SIZE_HALFWORD;
    dma_config.dst_data_width = ALD_DMA_DATA_SIZE_HALFWORD;
    dma_config.R_power        = ALD_DMA_R_POWER_1;
    dma_config.priority       = ALD_DMA_HIGHEST_PRIORITY;
    dma_config.mem_to_mem     = DISABLE;
    dma_config.circle_mode    = ENABLE;
    dma_config.msel           = DMA_MSEL_SPIX;

    ald_dma_clear_flag_status(ch, ALD_DMA_IT_FLAG_TC);
    ald_dma_clear_flag_status(ch, ALD_DMA_IT_FLAG_HT);
    ald_dma_config_base(&dma_config);
}
