/**
  *********************************************************************************
  *
  * @file    uart.c
  * @brief   UART for debug printf
  *
  * @version V1.0
  * @date    09 Mar. 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          09 Mar. 2023    Lisq            The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */


/* Includes ----------------------------------------------------------------- */
#include <string.h>
#include "uart.h"

/* Private Macros ----------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */
static uint32_t s_uart_enabled = 0;
ald_uart_handle_t g_h_uart;
static uint32_t s_rx_num = 0;
static uint32_t s_rx_cplted = 0;
uint8_t g_rbuf[4 * 64];
uint32_t g_rlen[4];
/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

/* Private Function---------------------------------------------------------- */
void bsp_uart_recv(uint8_t *data, uint16_t len, uint16_t timeout)
{
    ald_uart_recv_frame_by_it(&g_h_uart, data, len, timeout);
}

uint32_t bsp_uart_recved(char *buf, int length)
{
    uint32_t len = length > g_rlen[s_rx_num] ? g_rlen[s_rx_num] : length;

    if (len)
    {
        memcpy(buf, g_rbuf + s_rx_num * 64, len);
        g_rlen[s_rx_num] = 0;
        s_rx_num++;

        if (s_rx_num >= 4)
            s_rx_num = 0;
    }

    return len;
}
void uart_rx_cplt(ald_uart_handle_t *uart)
{
    g_rlen[s_rx_cplted++] = uart->rx_count;

    if (s_rx_cplted >= 4)
        s_rx_cplted = 0;

    bsp_uart_recv(g_rbuf + s_rx_cplted * 64, 64, 100);
}
/**
  * @brief  Initialize UART for debug output/input
  * @retval None
  */
void init_uart(void)
{
    ald_gpio_init_t x;

    memset(&x, 0, sizeof(x));
    /* Initialize tx pin*/
    x.mode  = ALD_GPIO_MODE_OUTPUT;
    x.od    = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.odrv  = ALD_GPIO_OUT_DRIVE_NORMAL;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_CMOS;
    x.func  = M_UART_TX_FUNC;
    ald_gpio_init(M_UART_TX_PORT, M_UART_TX_PIN, &x);

    /* Initialize rx pin*/
    x.mode  = ALD_GPIO_MODE_INPUT;
    x.od    = ALD_GPIO_PUSH_PULL;
    x.pupd  = ALD_GPIO_PUSH_UP;
    x.odrv  = ALD_GPIO_OUT_DRIVE_NORMAL;
    x.flt   = ALD_GPIO_FILTER_DISABLE;
    x.type  = ALD_GPIO_TYPE_CMOS;
    x.func  = M_UART_RX_FUNC;
    ald_gpio_init(M_UART_RX_PORT, M_UART_RX_PIN, &x);

    /* Initialize UART */
    memset(&g_h_uart, 0x00, sizeof(ald_uart_handle_t));
    /* Initialize uart */
    g_h_uart.perh             = M_UART;
    g_h_uart.init.baud        = M_UART_BAUD;
    g_h_uart.init.word_length = ALD_UART_WORD_LENGTH_8B;
    g_h_uart.init.stop_bits   = ALD_UART_STOP_BITS_1;
    g_h_uart.init.parity      = ALD_UART_PARITY_NONE;
    g_h_uart.init.mode        = ALD_UART_MODE_UART;
    g_h_uart.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    g_h_uart.rx_cplt_cbk      = uart_rx_cplt;
    ald_uart_init(&g_h_uart);
    ald_mcu_irq_config(M_UART_IRQ, 4, ENABLE);

    s_uart_enabled = 1;
    bsp_uart_recv(g_rbuf, 64, 100);
}

int _write(int file, const void *ptr, int len)
{
    uint16_t cnt;
    (void)file;
    uint8_t *ptr_char = (uint8_t *)ptr;

    int i = 0;

    for (i = 0 ; i < len ; i++)
    {
        cnt = 4000;

        while (!ald_uart_get_status(&g_h_uart, ALD_UART_STATUS_TFEMPTY) && (--cnt));

        g_h_uart.perh->TXBUF = *ptr_char++;
    }

    return len;
}

void __attribute__((interrupt)) M_UART_HANDLER(void)
{
    ald_uart_irq_handler(&g_h_uart);

    return;
}
