/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    23 Feb. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Feb. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

uint8_t g_recv_buf[20];
uint8_t g_recv_len = sizeof(g_recv_buf);
uint8_t g_send_buf[20];
uint8_t g_send_len = sizeof(g_send_buf);
ald_i2c_handle_t g_ald_i2c;

uint8_t g_tx_complete;
uint8_t g_rx_complete;

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/* The host send complete */
void master_tx_complete(ald_i2c_handle_t *arg)
{
    g_tx_complete = 1;
    return;
}

/* The host receive complete */
void master_rx_complete(ald_i2c_handle_t *arg)
{
    g_rx_complete = 1;
    return;
}

/**
  * @brief  Initializate pin of i2c module.
  * @retval None
  */
void init_i2c_pin(void)
{
    ald_gpio_init_t gpio_init;
    memset(&gpio_init, 0, sizeof(ald_gpio_init_t));

    /* Initialize SCL pin */
    gpio_init.mode  = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od    = ALD_GPIO_OPEN_DRAIN;
    gpio_init.pupd  = ALD_GPIO_PUSH_UP;
    gpio_init.odrv  = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type  = ALD_GPIO_TYPE_CMOS;
    gpio_init.func  = ALD_GPIO_FUNC_5;
    ald_gpio_init(I2C_SCL_PORT, I2C_SCL_PIN, &gpio_init);

    /* Initialize SDA pin */
    gpio_init.mode  = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od    = ALD_GPIO_OPEN_DRAIN;
    gpio_init.pupd  = ALD_GPIO_PUSH_UP;
    gpio_init.odrv  = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type  = ALD_GPIO_TYPE_CMOS;
    gpio_init.func  = ALD_GPIO_FUNC_5;
    ald_gpio_init(I2C_SDA_PORT, I2C_SDA_PIN, &gpio_init);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    uint16_t cnt = 40000U;

    /* Initialize ALD */
    ald_cmu_init();

    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable all peripherals */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    __enable_irq();

    /* Initialize pin */
    init_i2c_pin();

    /* clear i2c_handle_t structure */
    memset(&g_ald_i2c, 0, sizeof(ald_i2c_handle_t));
    /* Initialize i2c */
    g_ald_i2c.perh = I2C0;
    g_ald_i2c.init.module        = ALD_I2C_MODULE_MASTER;
    g_ald_i2c.init.addr_mode     = ALD_I2C_ADDR_7BIT;
    g_ald_i2c.init.clk_speed     = 400000;
    g_ald_i2c.init.dual_addr     = ALD_I2C_DUALADDR_DISABLE;
    g_ald_i2c.init.general_call  = ALD_I2C_GENERALCALL_DISABLE;
    g_ald_i2c.init.no_stretch    = ALD_I2C_NOSTRETCH_DISABLE;
    g_ald_i2c.init.own_addr1     = 0x2D << 1;
    g_ald_i2c.master_rx_cplt_cbk = master_rx_complete;
    g_ald_i2c.master_tx_cplt_cbk = master_tx_complete;
    ald_i2c_init(&g_ald_i2c);

    while (1)
    {
        /* master receive by dma */
        g_rx_complete = 0;
        ald_i2c_master_recv_by_dma(&g_ald_i2c, 0x2D << 1, g_recv_buf, g_recv_len, 1);

        while (!g_rx_complete && (--cnt));

        memcpy(g_send_buf, g_recv_buf, g_recv_len);

        /* master send by dma */
        g_tx_complete = 0;
        ald_i2c_master_send_by_dma(&g_ald_i2c, 0x2D << 1, g_send_buf, g_send_len, 0);

        cnt = 40000;

        while (!g_tx_complete && (--cnt));

        ald_delay_1ms(2000);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
