/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    16 Feb. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          16 Feb. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

uint32_t g_sysclk;
uint32_t g_pclk;
uint8_t g_clk_err_flag = 0U;

/* Private Constants --------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Configure HSCO pin
  * @retval None
  */
void init_cmu_pin(void)
{
    ald_gpio_init_t gpio_config;

    /* initialize the gpio_init */
    memset(&gpio_config, 0x0, sizeof(gpio_config));
    /* Initialize HSCO pin */
    gpio_config.mode  = ALD_GPIO_MODE_OUTPUT;
    gpio_config.od    = ALD_GPIO_PUSH_PULL;
    gpio_config.pupd  = ALD_GPIO_PUSH_UP;
    gpio_config.odrv  = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_config.flt   = ALD_GPIO_FILTER_DISABLE;
    gpio_config.type  = ALD_GPIO_TYPE_CMOS;
    gpio_config.func  = ALD_GPIO_FUNC_4;
    ald_gpio_init(HSCO_PORT, HSCO_PIN, &gpio_config);

    return;
}

/**
  * @brief  Main function
  * @retval Status.
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();

    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable all peripherals */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    /* get current clock */
    g_sysclk = ald_cmu_get_sys_clock();
    g_pclk = ald_cmu_get_pclk_clock();

    /* set the clock divider */
    ald_cmu_div_config(ALD_CMU_SYS, ALD_CMU_DIV_2);    /* sysclk = fclk / 2 */
    ald_cmu_div_config(ALD_CMU_PCLK, ALD_CMU_DIV_2);   /* pclk = sysclk/ 2 */

    /* get current clock */
    g_sysclk = ald_cmu_get_sys_clock();
    g_pclk = ald_cmu_get_pclk_clock();

    /* Configure high-speed clock output */
    ald_cmu_output_high_clock_config(ALD_CMU_OUTPUT_HIGH_SEL_SYSCLK, ALD_CMU_OUTPUT_DIV_128, ENABLE);
    /* Initialize clock output pin */
    init_cmu_pin();
    /* enalbe pll clock safe cecurity management */
    ald_cmu_pll_safe_config(ENABLE);
    /* enalbe hosc clock safe cecurity management */
    ald_cmu_hosc_safe_config(ALD_CMU_HOSC_4M_8M, ENABLE);
    __enable_irq();

    while (1)
    {
        /* if pll have unlocked and resumed */
        if (!ald_cmu_pulmcr_current_clock_source_get() && g_clk_err_flag)
            g_clk_err_flag = 0;
    }
}

/**
  * @brief  Interrupt callback function.
  * @retval None
  */
void ald_cmu_irq_cbk(ald_cmu_security_t se)
{
    if (se == ALD_CMU_HOSC_STOP)
    {
        /* system rescue code in hosc fault  */
    }

    if (se == ALD_CMU_PLL_UNLOCK)
    {
        g_clk_err_flag = 1;
        /* system rescue code in pll fault  */
    }

}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
