/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    16 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          16 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
uint16_t g_adc_result[4];
ald_adc_handle_t g_adc_config;
ald_adc_nch_conf_t g_nch_config;
ald_adc_ich_conf_t g_ich_config;

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/**
  * @brief  Adc insert convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
void insert_convert_complete(ald_adc_handle_t *arg)
{
    g_adc_result[0] = ald_adc_normal_get_value(&g_adc_config) * VDD / 4095;
    g_adc_result[1] = ald_adc_insert_get_value(&g_adc_config, ALD_ADC_ICH_IDX_1) * VDD / 4095;
    g_adc_result[2] = ald_adc_insert_get_value(&g_adc_config, ALD_ADC_ICH_IDX_2) * VDD / 4095;
    g_adc_result[3] = ald_adc_insert_get_value(&g_adc_config, ALD_ADC_ICH_IDX_3) * VDD / 4095;

    return;
}

/**
  * @brief  Initialize the ADC Pins.
  * @param  None
  * @retval None
  */
void init_adc_pin(void)
{
    ald_gpio_init_t gpio_init;
    memset(&gpio_init, 0x0, sizeof(ald_gpio_init_t));
    gpio_init.mode = ALD_GPIO_MODE_CLOSE;
    gpio_init.pupd = ALD_GPIO_FLOATING;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_0;
    ald_gpio_init(ADC_CHANNEL6_PORT, ADC_CHANNEL6_PIN, &gpio_init);

    return;
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable ALL peripheral */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    __enable_irq();

    /* Initialize ADC gpio */
    init_adc_pin();

    memset(&g_adc_config, 0x0, sizeof(ald_adc_handle_t));
    memset(&g_nch_config, 0x0, sizeof(ald_adc_nch_conf_t));

    /* Initialize adc */
    g_adc_config.perh            = ADC;
    g_adc_config.init.align      = ALD_ADC_DATAALIGN_RIGHT;
    g_adc_config.init.scan       = ENABLE;
    g_adc_config.init.cont       = ENABLE;
    g_adc_config.init.disc       = ALD_ADC_ALL_DISABLE;
    g_adc_config.init.disc_nr    = ALD_ADC_DISC_NR_1;
    g_adc_config.init.data_bit   = ALD_ADC_CONV_BIT_12;
    g_adc_config.init.div        = ALD_ADC_CKDIV_16;
    g_adc_config.init.nche_sel   = ALD_ADC_NCHESEL_MODE_ALL;
    g_adc_config.init.nch_nr     = ALD_ADC_NCH_NR_1;
    g_adc_config.init.n_ref      = ALD_ADC_NEG_REF_VSS;
    g_adc_config.init.p_ref      = ALD_ADC_POS_REF_VDD;
    g_adc_config.insert_cplt_cbk = insert_convert_complete;
    ALD_ADC_ENABLE(&g_adc_config);
    ALD_ADC_DISABLE(&g_adc_config);
    ald_adc_init(&g_adc_config);

    /* Initialize nromal convert channel */
    g_nch_config.ch   = ALD_ADC_CHANNEL_6;
    g_nch_config.idx  = ALD_ADC_NCH_IDX_1;
    g_nch_config.samp = ALD_ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(&g_adc_config, &g_nch_config);

    /* Initialize insert convert channel */
    g_ich_config.auto_m = ENABLE;
    g_ich_config.nr     = ALD_ADC_ICH_NR_3;
    g_ich_config.ch     = ALD_ADC_CHANNEL_6;
    g_ich_config.idx    = ALD_ADC_ICH_IDX_1;
    g_ich_config.samp   = ALD_ADC_SAMPLETIME_4;
    g_ich_config.offset = 0;
    ald_adc_insert_channel_config(&g_adc_config, &g_ich_config);

    g_ich_config.idx    = ALD_ADC_ICH_IDX_2;
    ald_adc_insert_channel_config(&g_adc_config, &g_ich_config);

    g_ich_config.idx    = ALD_ADC_ICH_IDX_3;
    ald_adc_insert_channel_config(&g_adc_config, &g_ich_config);

    ald_adc_interrupt_config(&g_adc_config, ALD_ADC_IT_ICH, ENABLE);
    ald_mcu_irq_config(ADC_IRQn, 4, ENABLE);

    /* Start normal convert*/
    ald_adc_normal_start(&g_adc_config);

    while (1)
    {
        ald_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
