/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    06 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          06 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
ald_uart_handle_t g_uart_init;
ald_adc_handle_t g_adc_config;
ald_adc_timer_config_t g_timer_config;
uint16_t g_adc_result[1];
uint8_t g_tx_buf[20];

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  格式化用户数据，使之符合传输协议
  * @param  srcdata-保存用户数据的首地址；
  *         desdata-保存处理好的数据的内存首地址(其内存大小至少大于前者5字节)；
  *         len-用户数据长度(单位:字节)
  * @retval 返回数据帧的总长度
  */
uint32_t data_initialize(uint8_t *desdata, uint8_t *srcdata, uint32_t len)
{
    uint32_t i;

    /* 帧头 */
    desdata[0] = 0xAA;
    desdata[1] = 0xAA;

    /* 功能码 */
    desdata[2] = 0xF1;

    /* 数据长度 */
    desdata[3] = len;

    /* 用户数据 */
    memcpy(desdata + 4, srcdata, len);

    /* 累加校验和 */
    desdata[len + 4] = 0U;

    for (i = 0; i < len + 4; i++)
    {
        /* 累加校验和, 高位舍去, 只保留低8位 */
        desdata[len + 4] += desdata[i];
    }

    return (len + 5);
}

/**
  * @brief  Adc insert convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
void insert_convert_complete(ald_adc_handle_t *arg)
{
    uint16_t cnt;
	uint32_t length, i;

    g_adc_result[0] = ald_adc_insert_get_value(&g_adc_config, ALD_ADC_ICH_IDX_1) * VDD / 4095;

	length = data_initialize(g_tx_buf, (uint8_t *)g_adc_result, 2);

	for (i = 0; i < length; i++)
	{
		g_uart_init.perh->TXBUF = g_tx_buf[i];

		cnt = 4000U;
		while (!ald_uart_get_flag_status(&g_uart_init, ALD_UART_IF_TBC) && (--cnt));

		ald_uart_clear_flag_status(&g_uart_init, ALD_UART_IF_TBC);
	}

    return;
}

/**
  * @brief  Initialize the ADC Pins.
  * @param  None
  * @retval None
  */
void init_adc_pin(void)
{
    /* Initialize adc gpio pin */
    ald_gpio_init_t gpio_init;
    memset(&gpio_init, 0x0, sizeof(ald_gpio_init_t));

    /* Initialize tx pin: PA2 */
    gpio_init.mode = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od   = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd = ALD_GPIO_PUSH_UP;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_2;
    ald_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    gpio_init.mode = ALD_GPIO_MODE_CLOSE;
    gpio_init.pupd = ALD_GPIO_FLOATING;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_0;
    ald_gpio_init(ADC_IN_PORT, ADC_IN_PIN, &gpio_init);

    return;
}

/**
  * @brief:  Initialize the uart.
  * @param:  None
  * @retval: None
  */
void uart_init(void)
{
    memset(&g_uart_init, 0x00, sizeof(ald_uart_handle_t));
    /* Initialize uart */
    g_uart_init.perh             = UARTX;
    g_uart_init.init.baud        = 115200U;
    g_uart_init.init.word_length = ALD_UART_WORD_LENGTH_8B;
    g_uart_init.init.stop_bits   = ALD_UART_STOP_BITS_1;
    g_uart_init.init.parity      = ALD_UART_PARITY_NONE;
    g_uart_init.init.mode        = ALD_UART_MODE_UART;
    g_uart_init.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    g_uart_init.rx_cplt_cbk      = NULL;
    g_uart_init.tx_cplt_cbk      = NULL;
    ald_uart_init(&g_uart_init);
}

/**
  * @brief  Configure the PIS.
  * @param  None
  * @retval None
  */
void config_timer_trig_adc(void)
{
    memset(&g_timer_config, 0x0, sizeof(ald_adc_timer_config_t));
    /* Config timer trigger adc function */
    g_timer_config.p_adc        = ADC;
    g_timer_config.adc_ch       = ALD_ADC_CHANNEL_6;
    g_timer_config.n_ref        = ALD_ADC_NEG_REF_VSS;
    g_timer_config.p_ref        = ALD_ADC_POS_REF_VDD;
    g_timer_config.p_timer      = AD16C4T;
    g_timer_config.time         = 200;
    ald_adc_timer_trigger_insert(&g_timer_config);

    return;
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable ALL peripheral */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    ald_mcu_irq_config(ADC_IRQn, 4, ENABLE);
    __enable_irq();

    /* Initialize ADC gpio */
    init_adc_pin();

	uart_init();

    /* Config PIS */
    config_timer_trig_adc();

    g_adc_config.perh = ADC;
    g_adc_config.insert_cplt_cbk = insert_convert_complete;
    ald_adc_interrupt_config(&g_adc_config, ALD_ADC_IT_ICH, ENABLE);

    while (1)
    {
        ald_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
