/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    15 Mar. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          15 Mar. 2023    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
ald_uart_handle_t g_uart_init;
ald_adc_handle_t g_adc_config;
ald_adc_nch_conf_t g_nch_config;
uint16_t g_adc_result[1];
uint8_t g_tx_buf[20];

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/**
  * @brief  格式化用户数据，使之符合传输协议
  * @param  srcdata-保存用户数据的首地址；
  *         desdata-保存处理好的数据的内存首地址(其内存大小至少大于前者5字节)；
  *         len-用户数据长度(单位:字节)
  * @retval 返回数据帧的总长度
  */
uint32_t data_initialize(uint8_t *desdata, uint8_t *srcdata, uint32_t len)
{
    uint32_t i;

    /* 帧头 */
    desdata[0] = 0xAA;
    desdata[1] = 0xAA;

    /* 功能码 */
    desdata[2] = 0xF1;

    /* 数据长度 */
    desdata[3] = len;

    /* 用户数据 */
    memcpy(desdata + 4, srcdata, len);

    /* 累加校验和 */
    desdata[len + 4] = 0U;

    for (i = 0; i < len + 4; i++)
    {
        /* 累加校验和, 高位舍去, 只保留低8位 */
        desdata[len + 4] += desdata[i];
    }

    return (len + 5);
}

/**
  * @brief  Adc normal convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
void normal_convert_complete(ald_adc_handle_t *arg)
{
    uint16_t cnt;
	uint32_t length, i;

    g_adc_result[0] = ald_adc_normal_get_value(&g_adc_config) * VDD / 4095;

	length = data_initialize(g_tx_buf, (uint8_t *)g_adc_result, 2);

	for (i = 0; i < length; i++)
	{
		g_uart_init.perh->TXBUF = g_tx_buf[i];

		cnt = 4000U;
		while (!ald_uart_get_flag_status(&g_uart_init, ALD_UART_IF_TBC) && (--cnt));

		ald_uart_clear_flag_status(&g_uart_init, ALD_UART_IF_TBC);
	}

    return;
}

/**
  * @brief  Initialize the ADC Pins.
  * @param  None
  * @retval None
  */
void init_adc_pin(void)
{
    /* Initialize adc gpio pin: PC0 */
    ald_gpio_init_t gpio_init;
    memset(&gpio_init, 0x0, sizeof(ald_gpio_init_t));

    /* Initialize tx pin: PA2 */
    gpio_init.mode = ALD_GPIO_MODE_OUTPUT;
    gpio_init.od   = ALD_GPIO_PUSH_PULL;
    gpio_init.pupd = ALD_GPIO_PUSH_UP;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_2;
    ald_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    gpio_init.mode = ALD_GPIO_MODE_CLOSE;
    gpio_init.pupd = ALD_GPIO_FLOATING;
    gpio_init.odrv = ALD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt  = ALD_GPIO_FILTER_DISABLE;
    gpio_init.type = ALD_GPIO_TYPE_CMOS;
    gpio_init.func = ALD_GPIO_FUNC_0;
    ald_gpio_init(ADC_CHANNEL6_PORT, ADC_CHANNEL6_PIN, &gpio_init);

    return;
}

/**
  * @brief:  Initialize the uart.
  * @param:  None
  * @retval: None
  */
void uart_init(void)
{
    memset(&g_uart_init, 0x00, sizeof(ald_uart_handle_t));
    /* Initialize uart */
    g_uart_init.perh             = UARTX;
    g_uart_init.init.baud        = 115200U;
    g_uart_init.init.word_length = ALD_UART_WORD_LENGTH_8B;
    g_uart_init.init.stop_bits   = ALD_UART_STOP_BITS_1;
    g_uart_init.init.parity      = ALD_UART_PARITY_NONE;
    g_uart_init.init.mode        = ALD_UART_MODE_UART;
    g_uart_init.init.fctl        = ALD_UART_HW_FLOW_CTL_DISABLE;
    g_uart_init.rx_cplt_cbk      = NULL;
    g_uart_init.tx_cplt_cbk      = NULL;
    ald_uart_init(&g_uart_init);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll_config(ALD_CMU_PLL_INPUT_HOSC8M, ALD_CMU_PLL_OUTPUT_72M);
    ald_cmu_clock_config(ALD_CMU_CLOCK_PLL, 72000000);

    /* Enable ALL peripheral */
    ald_cmu_perh_clock_config(ALD_CMU_PERH_ALL, ENABLE);

    ald_mcu_irq_config(ADC_IRQn, 4, ENABLE);
    __enable_irq();

    /* Initialize ADC gpio */
    init_adc_pin();

    uart_init();

    memset(&g_adc_config, 0x0, sizeof(ald_adc_handle_t));
    memset(&g_nch_config, 0x0, sizeof(ald_adc_nch_conf_t));

    /* Initialize adc */
    g_adc_config.perh            = ADC;
    g_adc_config.init.align      = ALD_ADC_DATAALIGN_RIGHT;
    g_adc_config.init.scan       = ENABLE;
    g_adc_config.init.cont       = DISABLE;
    g_adc_config.init.disc       = ALD_ADC_ALL_DISABLE;
    g_adc_config.init.disc_nr    = ALD_ADC_DISC_NR_1;
    g_adc_config.init.data_bit   = ALD_ADC_CONV_BIT_12;
    g_adc_config.init.div        = ALD_ADC_CKDIV_16;
    g_adc_config.init.nche_sel   = ALD_ADC_NCHESEL_MODE_ALL;
    g_adc_config.init.nch_nr     = ALD_ADC_NCH_NR_1;
    g_adc_config.init.n_ref      = ALD_ADC_NEG_REF_VSS;
    g_adc_config.init.p_ref      = ALD_ADC_POS_REF_VDD;
    g_adc_config.normal_cplt_cbk = normal_convert_complete;
    ALD_ADC_ENABLE(&g_adc_config);
    ALD_ADC_DISABLE(&g_adc_config);
    ald_adc_init(&g_adc_config);

    /* Initialize nromal convert channel */
    g_nch_config.ch   = ALD_ADC_CHANNEL_6;
    g_nch_config.idx  = ALD_ADC_NCH_IDX_1;
    g_nch_config.samp = ALD_ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(&g_adc_config, &g_nch_config);

    while (1)
    {
        /* Start nromal convert, enable interrupt */
        ald_adc_normal_start_by_it(&g_adc_config);

        ald_delay_1ms(10);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
