/*
 * Copyright (c) 2006-2021, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 * 2018-08-24     yangjie      the first version
 * 2024-01-17     shiwa        update comments
 */

/*
 * 程序清单：使用互斥量来防止优先级反转
 *
 * 这个例子将创建 3 个动态线程以检查持有互斥量时，持有的线程优先级是否
 * 被调整到等待线程优先级中的最高优先级。
 *
 * 线程 1，2，3 的优先级从高到低分别被创建，
 * 线程 3 先持有互斥量，而后线程 2 试图持有互斥量，此时线程 3 的优先级应该
 * 被提升为和线程 2 的优先级相同。线程 1 用于检查线程 3 的优先级是否被提升
 * 为与线程 2的优先级相同。
 */
/* Includes ------------------------------------------------------------------ */
#include <rtthread.h>

/* Private Macros ------------------------------------------------------------ */
#define THREAD_PRIORITY       4
#define THREAD_STACK_SIZE     256
#define THREAD_TIMESLICE      5

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */
/* 指向线程控制块的指针 */
static rt_thread_t s_tid1 = RT_NULL;
static rt_thread_t s_tid2 = RT_NULL;
static rt_thread_t s_tid3 = RT_NULL;
static rt_mutex_t s_mutex = RT_NULL;

/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
/* 线程 1 入口 */
static void thread1_entry(void *parameter)
{
    /* 先让低优先级线程运行 */
    rt_thread_mdelay(100);

    /* 此时 thread3 持有 mutex，并且 thread2 等待持有 s_mutex */

    /* 检查 thread2 与 thread3 的优先级情况 */
    if (s_tid2->current_priority != s_tid3->current_priority)
    {
        /* 优先级不相同，测试失败 */
        rt_kprintf("the priority of thread2 is: %d\n", s_tid2->current_priority);
        rt_kprintf("the priority of thread3 is: %d\n", s_tid3->current_priority);
        rt_kprintf("test failed.\n");
        return;
    }
    else
    {
        rt_kprintf("the priority of thread2 is: %d\n", s_tid2->current_priority);
        rt_kprintf("the priority of thread3 is: %d\n", s_tid3->current_priority);
        rt_kprintf("test OK.\n");
    }
}

/* 线程 2 入口 */
static void thread2_entry(void *parameter)
{
    rt_err_t result;

    rt_kprintf("the priority of thread2 is: %d\n", s_tid2->current_priority);

    /* 先让低优先级线程运行 */
    rt_thread_mdelay(50);

    /*
     * 试图持有互斥锁，此时 thread3 持有，应把 thread3 的优先级提升
     * 到 thread2 相同的优先级
     */
    result = rt_mutex_take(s_mutex, RT_WAITING_FOREVER);

    if (result == RT_EOK)
    {
        /* 释放互斥锁 */
        rt_mutex_release(s_mutex);
    }
}

/* 线程 3 入口 */
static void thread3_entry(void *parameter)
{
    rt_tick_t tick;
    rt_err_t result;

    rt_kprintf("the priority of thread3 is: %d\n", s_tid3->current_priority);

    result = rt_mutex_take(s_mutex, RT_WAITING_FOREVER);

    if (result != RT_EOK)
    {
        rt_kprintf("thread3 take a s_mutex, failed.\n");
    }

    /* 做一个长时间的循环，500ms */
    tick = rt_tick_get();

    while (rt_tick_get() - tick < (RT_TICK_PER_SECOND / 2)) ;

    rt_mutex_release(s_mutex);
}

int pri_inversion(void)
{
    /* 创建互斥锁 */
    s_mutex = rt_mutex_create("s_mutex", RT_IPC_FLAG_PRIO);

    if (s_mutex == RT_NULL)
    {
        rt_kprintf("create dynamic s_mutex failed.\n");
        return -1;
    }

    /* 创建线程 1 */
    s_tid1 = rt_thread_create("thread1",
                            thread1_entry,
                            RT_NULL,
                            THREAD_STACK_SIZE,
                            THREAD_PRIORITY - 1, THREAD_TIMESLICE);

    if (s_tid1 != RT_NULL)
        rt_thread_startup(s_tid1);

    /* 创建线程 2 */
    s_tid2 = rt_thread_create("thread2",
                            thread2_entry,
                            RT_NULL,
                            THREAD_STACK_SIZE,
                            THREAD_PRIORITY, THREAD_TIMESLICE);

    if (s_tid2 != RT_NULL)
        rt_thread_startup(s_tid2);

    /* 创建线程 3 */
    s_tid3 = rt_thread_create("thread3",
                            thread3_entry,
                            RT_NULL,
                            THREAD_STACK_SIZE,
                            THREAD_PRIORITY + 1, THREAD_TIMESLICE);

    if (s_tid3 != RT_NULL)
        rt_thread_startup(s_tid3);

    return 0;
}

