/*
 * Copyright (c) 2006-2021, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 * 2018-08-24     yangjie      the first version
 * 2024-01-17     shiwa        update comments
 */

/*
 * 程序清单：邮箱例程
 *
 * 这个程序会创建2个动态线程，一个静态的邮箱对象，其中一个线程往邮箱中发送邮件，
 * 一个线程往邮箱中收取邮件。
 */
/* Includes ------------------------------------------------------------------ */
#include <rtthread.h>

/* Private Macros ------------------------------------------------------------ */
#define THREAD_PRIORITY      5
#define THREAD_TIMESLICE     5

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */
/* 邮箱控制块 */
static struct rt_mailbox s_mailbox;
/* 用于放邮件的内存池 */
static char s_mb_pool[64];

static char s_mb_str1[] = "I'm a mail!";
static char s_mb_str2[] = "this is another mail!";
static char s_mb_str3[] = "over";

ALIGN(RT_ALIGN_SIZE)
static char s_thread1_stack[256];
static struct rt_thread s_thread1;

/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
/* 线程1入口 */
static void thread1_entry(void *parameter)
{
    char *str;

    while (1)
    {
        rt_kprintf("s_thread1: try to recv a mail\n");

        /* 从邮箱中收取邮件 */
        if (rt_mb_recv(&s_mailbox, (rt_ubase_t *)&str, RT_WAITING_FOREVER) == RT_EOK)
        {
            rt_kprintf("s_thread1: get a mail from mailbox, the content:%s\n", str);

            if (str == s_mb_str3)
                break;

            /* 延时100ms */
            rt_thread_mdelay(100);
        }
    }

    /* 执行邮箱对象脱离 */
    rt_mb_detach(&s_mailbox);
}

ALIGN(RT_ALIGN_SIZE)
static char s_thread2_stack[256];
static struct rt_thread s_thread2;

/* 线程2入口 */
static void thread2_entry(void *parameter)
{
    rt_uint8_t count;

    count = 0;

    while (count < 10)
    {
        count ++;

        if (count & 0x1)
        {
            /* 发送mb_str1地址到邮箱中 */
            rt_mb_send(&s_mailbox, (rt_uint32_t)&s_mb_str1);
        }
        else
        {
            /* 发送mb_str2地址到邮箱中 */
            rt_mb_send(&s_mailbox, (rt_uint32_t)&s_mb_str2);
        }

        /* 延时200ms */
        rt_thread_mdelay(200);
    }

    /* 发送邮件告诉线程1，线程2已经运行结束 */
    rt_mb_send(&s_mailbox, (rt_uint32_t)&s_mb_str3);
}

int mailbox_sample(void)
{
    rt_err_t result;

    /* 初始化一个mailbox */
    result = rt_mb_init(&s_mailbox,
                        "mbt",                      /* 名称是mbt */
                        &s_mb_pool[0],                /* 邮箱用到的内存池是mb_pool */
                        sizeof(s_mb_pool) / sizeof(rt_ubase_t), /* 邮箱中的邮件数目,sizeof(rt_ubase_t)表示指针大小 */
                        RT_IPC_FLAG_PRIO);          /* 采用PRIO方式进行线程等待 */

    if (result != RT_EOK)
    {
        rt_kprintf("init mailbox failed.\n");
        return -1;
    }

    rt_thread_init(&s_thread1,
                   "s_thread1",
                   thread1_entry,
                   RT_NULL,
                   &s_thread1_stack[0],
                   sizeof(s_thread1_stack),
                   THREAD_PRIORITY, THREAD_TIMESLICE);
    rt_thread_startup(&s_thread1);

    rt_thread_init(&s_thread2,
                   "s_thread2",
                   thread2_entry,
                   RT_NULL,
                   &s_thread2_stack[0],
                   sizeof(s_thread2_stack),
                   THREAD_PRIORITY, THREAD_TIMESLICE);
    rt_thread_startup(&s_thread2);
    return 0;
}

