/*
 * Copyright (c) 2006-2021, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 * 2018-08-24     yangjie      the first version
 * 2024-01-17     shiwa        update comments
 */

/*
 * 程序清单：事件例程
 *
 * 程序会初始化2个线程及初始化一个静态事件对象
 * 一个线程等待于事件对象上，以接收事件；
 * 一个线程发送事件 (事件3/事件5)
*/
/* Includes ------------------------------------------------------------------ */
#include <rtthread.h>

/* Private Macros ------------------------------------------------------------ */
#define THREAD_PRIORITY      5
#define THREAD_TIMESLICE     5

#define EVENT_FLAG3 (1 << 3)
#define EVENT_FLAG5 (1 << 5)

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */
/* 事件控制块 */
static struct rt_event s_event;

ALIGN(RT_ALIGN_SIZE)
static char s_thread1_stack[256];
static struct rt_thread s_thread1;

/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
/* 线程1入口函数 */
static void thread1_recv_event(void *param)
{
    rt_uint32_t e;

    /* 第一次接收事件，事件3或事件5任意一个可以触发线程1，接收完后清除事件标志 */
    if (rt_event_recv(&s_event, (EVENT_FLAG3 | EVENT_FLAG5),
                      RT_EVENT_FLAG_OR | RT_EVENT_FLAG_CLEAR,
                      RT_WAITING_FOREVER, &e) == RT_EOK)
    {
        rt_kprintf("s_thread1: OR recv s_event 0x%x\n", e);
    }

    rt_kprintf("s_thread1: delay 1s to prepare the second s_event\n");
    rt_thread_mdelay(1000);

    /* 第二次接收事件，事件3和事件5均发生时才可以触发线程1，接收完后清除事件标志 */
    if (rt_event_recv(&s_event, (EVENT_FLAG3 | EVENT_FLAG5),
                      RT_EVENT_FLAG_AND | RT_EVENT_FLAG_CLEAR,
                      RT_WAITING_FOREVER, &e) == RT_EOK)
    {
        rt_kprintf("s_thread1: AND recv s_event 0x%x\n", e);
    }

    rt_kprintf("s_thread1 leave.\n");
}

ALIGN(RT_ALIGN_SIZE)
static char s_thread2_stack[256];
static struct rt_thread s_thread2;

/* 线程2入口 */
static void thread2_send_event(void *param)
{
    rt_kprintf("s_thread2: send event3\n");
    rt_event_send(&s_event, EVENT_FLAG3);
    rt_thread_mdelay(200);

    rt_kprintf("s_thread2: send event5\n");
    rt_event_send(&s_event, EVENT_FLAG5);
    rt_thread_mdelay(200);

    rt_kprintf("s_thread2: send event3\n");
    rt_event_send(&s_event, EVENT_FLAG3);
    rt_kprintf("s_thread2 leave.\n");
}

int event_sample(void)
{
    rt_err_t result;

    /* 初始化事件对象 */
    result = rt_event_init(&s_event, "s_event", RT_IPC_FLAG_PRIO);

    if (result != RT_EOK)
    {
        rt_kprintf("init s_event failed.\n");
        return -1;
    }

    rt_thread_init(&s_thread1,
                   "s_thread1",
                   thread1_recv_event,
                   RT_NULL,
                   &s_thread1_stack[0],
                   sizeof(s_thread1_stack),
                   THREAD_PRIORITY - 1, THREAD_TIMESLICE);
    rt_thread_startup(&s_thread1);

    rt_thread_init(&s_thread2,
                   "s_thread2",
                   thread2_send_event,
                   RT_NULL,
                   &s_thread2_stack[0],
                   sizeof(s_thread2_stack),
                   THREAD_PRIORITY, THREAD_TIMESLICE);
    rt_thread_startup(&s_thread2);

    return 0;
}
