/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    9 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Jan. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "es_st7789.h"
#include "es_spi_flash.h"
#include "printf.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

uint32_t g_es_gui_ext_mem_base_addr = 0U;

extern uint16_t g_dma_used_color;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief spi function
  * @retval None.
  */
void es_init_extern_flash(void)
{
	md_dma_set_dest_addr(ES_EXTERN_FLASH_SPI_DMA_CH, (void*)(&(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA)));
}

void es_spiflash_wait_spi_over(void)
{
    uint8_t tmp8;

    while (READ_REG(DMA->CHANNEL[ES_LCD_SPI_DMA_CH].CON) & 0x1) {}

    while (READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_TXFLV_MSK);

    while (((READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_BUSY_MSK))) {}

    while (READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_RXFLV_MSK)
    {
        tmp8  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);
    }

    (void)tmp8;
}

void es_spiflash_read_data8_start(uint32_t addr)
{
    es_spi_datasize_8(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    md_gpio_set_pin_low(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN);

    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , (uint8_t)(FLASH_READ));
    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , (uint8_t)((addr >> 16) & 0xFF));
    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , (uint8_t)((addr >> 8) & 0xFF));
    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , (uint8_t)((addr) & 0xFF));

    es_spiflash_wait_spi_over();
}

void es_spiflash_read_data8_end(void)
{
    es_spiflash_wait_spi_over();

    md_gpio_set_pin_high(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN);
}

void es_spiflash_read_data16_start(uint32_t addr)
{
    es_spi_datasize_16(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    md_gpio_set_pin_low(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN);

    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , (uint16_t)((FLASH_READ << 8) | ((addr >> 16) & 0xFF)));
    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , (uint16_t)(addr & 0xFFFF));

    es_spiflash_wait_spi_over();
}

void es_spiflash_read_data16_end(void)
{
    es_spiflash_wait_spi_over();

    md_gpio_set_pin_high(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN);

    es_spi_datasize_8(ES_EXTERN_FLASH_SPI_PERIPHERAL);
}

void es_spiflash_base_addr_refresh(void)
{
    uint32_t buf32[4];

    ll_flash_sector_read((uint8_t *)buf32, 2047 * 4096, 16);

    if ((buf32[0] == 0xAEAE55AA) && (buf32[3] == 0xEAEAAA55) && (buf32[1] == (~buf32[2])))
    {
        g_es_gui_ext_mem_base_addr = buf32[1];
    }
    else
    {
        spiflash_data_err_handle();/*信息中没有该对象，更新spiflash*/
    }
}

/**
  * @brief  low-level flash sector read operation function.
  * @param  str_buf: pointer to the first item of the byte array to store read data.
  * @param  sec: start sector.
  * @param  sec_num: amount of sector to be read.
  * @retval Status, see @ref ald_status_t.
  */
uint8_t ll_flash_sector_read(uint8_t *str_buf, uint32_t addr, uint32_t num)
{
    uint32_t i;

    if (str_buf == NULL)
        return 1;

    if (num < 1)
    {
        return 1;
    }

    es_spiflash_read_data8_start(addr);

    num--;
    i = 0;

    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , 0xFF);

    while (i < num)
    {
        WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , 0xFF);

        while (ES_FLASH_WAIT_RXNE_FLAG);

        str_buf[i++]  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);
    }

    while (ES_FLASH_WAIT_RXNE_FLAG);

    str_buf[i]  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);

    es_spiflash_read_data8_end();

    return 0;
}

/*16字节读*/
uint8_t ll_flash_sector_read16(uint8_t *str_buf, uint32_t addr, uint32_t num)
{
    uint32_t i;
    uint16_t *str_buf16 = (uint16_t *)str_buf;

    if (num < 2)
    {
        return 1;
    }

    es_spiflash_read_data16_start(addr);

    num = (num / 2) - 1;
    i = 0;

    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , 0xFFFF);

    while (i < num)
    {
        WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , 0xFFFF);

        while (ES_FLASH_WAIT_RXNE_FLAG);

        str_buf16[i++]  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);
    }

    while (ES_FLASH_WAIT_RXNE_FLAG);

    str_buf16[i]  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);

    es_spiflash_read_data16_end();

    return 0;
}

void es_st7789_write_area_flash_spi_perh2perh(uint16_t x1, uint16_t x2, uint16_t y1, uint16_t y2,  uint32_t addr)
{
    while (READ_REG(DMA->CHANNEL[ES_LCD_SPI_DMA_CH].CON) & 0x1);

    md_gpio_set_pin_high(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN);
    md_gpio_set_pin_high(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN);

    uint32_t size = (x2 - x1 + 1) * (y2 - y1 + 1);
    uint8_t temp8;
    uint16_t temp16;
    uint32_t dma_transfer_num;

    es_spi_datasize_8(ES_LCD_SPI_PERIPHERAL);

    /*LCD写像素数据+SPI片选*/
    es_st7789_lcd_write_area_cmd(x1, x2, y1, y2);

    md_gpio_set_pin_high(ES_LCD_CD_PIN_PORT, ES_LCD_CD_PIN_MD_PIN);

    es_spiflash_read_data16_start(addr);

    md_gpio_write_pin(ES_EXTERN_FLASH_SPI_MOSI_PIN_PORT, ES_EXTERN_FLASH_SPI_MOSI_PIN_MD_PIN, 1);

    md_gpio_set_pin_function(ES_EXTERN_FLASH_SPI_MOSI_PIN_PORT, ES_EXTERN_FLASH_SPI_MOSI_PIN_MD_PIN, MD_GPIO_FUNC_1);
    md_gpio_set_pin_function(ES_LCD_SPI_SCK_PIN_PORT, ES_LCD_SPI_SCK_PIN_MD_PIN, MD_GPIO_FUNC_6);
    md_gpio_set_pin_function(ES_LCD_SPI_MOSI_PIN_PORT, ES_LCD_SPI_MOSI_PIN_MD_PIN, MD_GPIO_FUNC_6);

    md_gpio_set_pin_low(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN);

    while (size)
    {
        dma_transfer_num = (size > 65535) ? 65535 : size;
        /*SPI发送DMA开始*/

        md_dma_set_transfer_size(ES_EXTERN_FLASH_SPI_DMA_CH, dma_transfer_num);

        md_dma_enable_channel(ES_EXTERN_FLASH_SPI_DMA_CH);

        md_spi_enable_txdma(ES_EXTERN_FLASH_SPI_PERIPHERAL);

        size -= dma_transfer_num;

        /*等待DMA完成*/
        while (READ_REG(DMA->CHANNEL[ES_EXTERN_FLASH_SPI_DMA_CH].CON) & 0x1)
        {
        }

        md_spi_disable_txdma(ES_EXTERN_FLASH_SPI_PERIPHERAL);
    }

    es_spiflash_read_data16_end();

    md_gpio_set_pin_high(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN);

    md_gpio_set_pin_function(ES_EXTERN_FLASH_SPI_MOSI_PIN_PORT, ES_EXTERN_FLASH_SPI_MOSI_PIN_MD_PIN, ES_EXTERN_FLASH_SPI_MOSI_PIN_MD_FUNC);
    md_gpio_set_pin_function(ES_LCD_SPI_SCK_PIN_PORT, ES_LCD_SPI_SCK_PIN_MD_PIN, ES_LCD_SPI_SCK_PIN_MD_FUNC);
    md_gpio_set_pin_function(ES_LCD_SPI_MOSI_PIN_PORT, ES_LCD_SPI_MOSI_PIN_MD_PIN, ES_LCD_SPI_MOSI_PIN_MD_FUNC);

    (void)temp8;
    (void)temp16;
}

#pragma GCC push_options
#pragma GCC optimize("Ofast" )

void lv_rle16_decompress(uint16_t x1, uint16_t x2, uint16_t y1, uint16_t y2, uint32_t addr, uint32_t len)
{
    uint32_t i;
    uint16_t temp16;
    uint32_t ctrl_byte;
    uint32_t ctrl_byte2;
    uint32_t rd_len = 0;

    md_gpio_set_pin_high(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN);

    md_gpio_set_pin_high(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN);

    es_spi_datasize_8(ES_LCD_SPI_PERIPHERAL);

    es_spiflash_read_data16_start(addr);

    /*LCD写像素数据+SPI片选*/
    es_st7789_lcd_write_area_cmd(x1, x2, y1, y2);

    es_spi_datasize_16(ES_LCD_SPI_PERIPHERAL);

    md_gpio_set_pin_high(ES_LCD_CD_PIN_PORT, ES_LCD_CD_PIN_MD_PIN);

    md_gpio_set_pin_low(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN);

    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , 0xFFFF);

    while (rd_len < len)
    {
        WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , 0xFFFF);

        while ((!(READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_RXNE_MSK)));

        ctrl_byte  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);
        rd_len += 2;

        if (ctrl_byte & 0xff00)
        {
            break;
        }

        if (ctrl_byte & 0x80)
        {
            /* copy directly from input to output */
            ctrl_byte2 = ctrl_byte & 0x7F;
            rd_len += 2 * (ctrl_byte2);

            for (i = 0; i < ctrl_byte2; i++)
            {

                WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , 0xFFFF);

                while ((!(READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_RXNE_MSK)));

                WRITE_REG(ES_LCD_SPI_PERIPHERAL->DATA , READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA));
            }
        }
        else
        {
            rd_len += 2;

            WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , 0xFFFF);

            while ((!(READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_RXNE_MSK)));

            temp16  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);

            for (i = 0; i < ctrl_byte; i++)
            {
                WRITE_REG(ES_LCD_SPI_PERIPHERAL->DATA , temp16);

                while ((READ_REG(ES_LCD_SPI_PERIPHERAL->STAT) & SPI_STAT_TXF_MSK));
            }
        }
    }

    /*等待SPI传输完成*/
    es_lcd_wait_spi_over_cs1();

    es_spiflash_read_data16_end();

    (void)temp16;
}

#pragma GCC pop_options


uint32_t es_spiflash_read_data32(uint32_t addr)
{
    uint32_t data32;

    uint16_t temp16;

    md_gpio_set_pin_low(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN);

    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , (uint16_t)((FLASH_READ << 8) | ((addr >> 16) & 0xFF)));
    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , (uint16_t)(addr & 0xFFFF));

    while ((!(READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_RXNE_MSK)));

    temp16  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);

    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , 0xFFFF);

    while ((!(READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_RXNE_MSK)));

    temp16  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);

    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , 0xFFFF);

    while ((!(READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_RXNE_MSK)));

    temp16  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);

    while ((!(READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_RXNE_MSK)));

    data32 = (temp16 << 16) + READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);

    md_gpio_set_pin_high(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN);

    return data32;
}

uint16_t es_spiflash_read_data16(uint32_t addr)
{
    uint16_t temp16;

    md_gpio_set_pin_low(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN);

    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , (uint16_t)((FLASH_READ << 8) | ((addr >> 16) & 0xFF)));
    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , (uint16_t)(addr & 0xFFFF));

    while ((!(READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_RXNE_MSK)));

    temp16  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);

    WRITE_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA , 0xFFFF);

    while ((!(READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_RXNE_MSK)));

    temp16  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);

    while ((!(READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT) & SPI_STAT_RXNE_MSK)));

    temp16  = READ_REG(ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA);

    md_gpio_set_pin_high(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN);

    return temp16;
}

void es_spiflash_gui_ext_mem_get_img_info(lv_img_dsc_t *img1)
{
    uint32_t g_es_exr_mem_img_num;
    uint32_t g_es_exr_mem_img_info_addr;

    if (!img1)
    {
        return;
    }

    if ((img1->header.always_zero != 2))
    {
        return;
    }

    es_spi_datasize_16(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    g_es_exr_mem_img_num = es_spiflash_read_data32(g_es_gui_ext_mem_base_addr + 24);
    g_es_exr_mem_img_info_addr = g_es_gui_ext_mem_base_addr + es_spiflash_read_data32(g_es_gui_ext_mem_base_addr + 28);

    if (img1->img_id >= g_es_exr_mem_img_num)
    {
        spiflash_data_err_handle();/*信息中没有该对象，更新spiflash*/
    }

    g_es_exr_mem_img_info_addr += img1->img_id * IMG_INFO_SIZE;
    uint16_t img_w = es_spiflash_read_data16(g_es_exr_mem_img_info_addr);
    uint16_t img_h = es_spiflash_read_data16(g_es_exr_mem_img_info_addr + 2);
    /*仅支持普通图片*/
    /*uint32_t format = es_spiflash_read_data32(g_es_exr_mem_img_info_addr + 4);*/
    uint32_t data_addr = es_spiflash_read_data32(g_es_exr_mem_img_info_addr + 8);

    img1->header.w = img_w;
    img1->header.h = img_h;
    img1->header.cf = LV_IMG_CF_TRUE_COLOR;
    img1->data = (uint8_t *)(data_addr + g_es_gui_ext_mem_base_addr);
}

void es_spiflash_gui_ext_mem_show_img(uint32_t x_offset, uint32_t y_offset, uint32_t img_index)
{
    es_spi_datasize_16(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    uint32_t g_es_exr_mem_img_num;
    uint32_t g_es_exr_mem_img_info_addr;

    g_es_exr_mem_img_num = es_spiflash_read_data32(g_es_gui_ext_mem_base_addr + 24);
    g_es_exr_mem_img_info_addr = g_es_gui_ext_mem_base_addr + es_spiflash_read_data32(g_es_gui_ext_mem_base_addr + 28);

    if (img_index >= g_es_exr_mem_img_num)
    {
        spiflash_data_err_handle();/*信息中没有该对象，更新spiflash*/
    }

    g_es_exr_mem_img_info_addr += img_index * IMG_INFO_SIZE;
    uint16_t img_w = es_spiflash_read_data16(g_es_exr_mem_img_info_addr);
    uint16_t img_h = es_spiflash_read_data16(g_es_exr_mem_img_info_addr + 2);
    uint32_t format = es_spiflash_read_data32(g_es_exr_mem_img_info_addr + 4);
    uint32_t data_addr = es_spiflash_read_data32(g_es_exr_mem_img_info_addr + 8);
    uint32_t data_len = es_spiflash_read_data32(g_es_exr_mem_img_info_addr + 12);

    if (format == 0)
    {
        es_st7789_write_area_flash_spi_perh2perh(x_offset, x_offset + img_w - 1, y_offset, y_offset + img_h - 1, data_addr + g_es_gui_ext_mem_base_addr);
    }

    if (format == 1)
    {
        lv_rle16_decompress(0, img_w - 1, 0, img_h - 1, data_addr + g_es_gui_ext_mem_base_addr, data_len);
    }

    es_spi_datasize_8(ES_EXTERN_FLASH_SPI_PERIPHERAL);
}


void es_spiflash_gui_ext_mem_get_font_info(const lv_font_t *font1)
{
    es_spi_datasize_16(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    lv_font_fmt_txt_dsc_t *dsc = (lv_font_fmt_txt_dsc_t *)font1->dsc;

    uint32_t g_es_exr_mem_font_num;
    uint32_t g_es_exr_mem_font_info_addr;

    g_es_exr_mem_font_num = es_spiflash_read_data32(g_es_gui_ext_mem_base_addr + 32);
    g_es_exr_mem_font_info_addr = g_es_gui_ext_mem_base_addr + es_spiflash_read_data32(g_es_gui_ext_mem_base_addr + 36);

    if (font1->value >= g_es_exr_mem_font_num)
    {
        spiflash_data_err_handle();/*信息中没有该对象，更新spiflash*/
    }

    g_es_exr_mem_font_info_addr += font1->value * FONT_INFO_SIZE;
    dsc->min_index = es_spiflash_read_data16(g_es_exr_mem_font_info_addr);
    dsc->max_index = es_spiflash_read_data16(g_es_exr_mem_font_info_addr + 2);
    dsc->glyph_bitmap = (uint8_t *)(es_spiflash_read_data32(g_es_exr_mem_font_info_addr + 8));

    es_spi_datasize_8(ES_EXTERN_FLASH_SPI_PERIPHERAL);
}


const uint8_t *es_spiflash_gui_ext_mem_get_font_txt(const lv_font_t *font1, uint16_t txt_id)
{
    lv_font_fmt_txt_dsc_t *dsc = (lv_font_fmt_txt_dsc_t *)font1->dsc;

    if ((txt_id < dsc->min_index) || (txt_id > dsc->max_index))
    {
        return NULL;
    }

    es_spi_datasize_16(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    uint32_t txt_addr;

    txt_addr = es_spiflash_read_data32((((uint32_t)dsc->glyph_bitmap) + g_es_gui_ext_mem_base_addr + (txt_id - dsc->min_index) * 4));

    if (txt_addr)
    {
        txt_addr += g_es_gui_ext_mem_base_addr + 5;
    }

    es_spi_datasize_8(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    return (const uint8_t *)txt_addr;
}

bool es_spiflash_gui_ext_mem_get_font_bitmap(const lv_font_t *font1, lv_font_glyph_dsc_t *dsc_out, uint16_t txt_id)
{
    es_spi_datasize_16(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    lv_font_fmt_txt_dsc_t *dsc = (lv_font_fmt_txt_dsc_t *)font1->dsc;

    uint32_t txt_addr;

    struct font_chr_info_type info;

    txt_addr = es_spiflash_read_data32((((uint32_t)dsc->glyph_bitmap) + g_es_gui_ext_mem_base_addr + (txt_id - dsc->min_index) * 4));

    es_spi_datasize_8(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    if (txt_addr)
    {
        txt_addr += g_es_gui_ext_mem_base_addr;

        ll_flash_sector_read((uint8_t *)(&info), txt_addr, 5);
        dsc_out->adv_w = info.adv_w;
        dsc_out->box_w = info.box_w;
        dsc_out->box_h = info.box_h;
        dsc_out->ofs_x = info.ofs_x;
        dsc_out->ofs_y = info.ofs_y;

        return true;
    }

    return false;
}

void es_spiflash_gui_ext_mem_get_gif_info(es_gif_info_t *gif1)
{
    es_spi_datasize_16(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    uint32_t g_es_exr_mem_gif_num;
    uint32_t g_es_exr_mem_gif_info_addr;

    g_es_exr_mem_gif_num = es_spiflash_read_data32(g_es_gui_ext_mem_base_addr + 40);
    g_es_exr_mem_gif_info_addr = g_es_gui_ext_mem_base_addr + es_spiflash_read_data32(g_es_gui_ext_mem_base_addr + 44);

    if (gif1->gif_index >= g_es_exr_mem_gif_num)
    {
        spiflash_data_err_handle();/*信息中没有该对象，更新spiflash*/
    }

    g_es_exr_mem_gif_info_addr += gif1->gif_index * GIF_INFO_SIZE;

    gif1->gif_w = es_spiflash_read_data16(g_es_exr_mem_gif_info_addr);
    gif1->gif_h = es_spiflash_read_data16(g_es_exr_mem_gif_info_addr + 2);
    gif1->gif_img_max_index = es_spiflash_read_data32(g_es_exr_mem_gif_info_addr + 4);
    gif1->gif_time_duration = es_spiflash_read_data32(g_es_exr_mem_gif_info_addr + 8);
    gif1->gif_data_spiflash_addr = g_es_gui_ext_mem_base_addr + es_spiflash_read_data32(g_es_exr_mem_gif_info_addr + 12);
    gif1->gif_type = es_spiflash_read_data32(g_es_exr_mem_gif_info_addr + 16);

    es_spi_datasize_8(ES_EXTERN_FLASH_SPI_PERIPHERAL);
}

void es_spiflash_gui_ext_mem_show_gif_frame(es_gif_info_t *gif1, uint32_t frame_id)
{
    es_spi_datasize_16(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    if (gif1->gif_type == 0)
    {
        es_st7789_write_area_flash_spi_perh2perh(gif1->pos_x, gif1->pos_x + gif1->gif_w - 1, \
                gif1->pos_y, gif1->pos_y + gif1->gif_h - 1, \
                gif1->gif_data_spiflash_addr + (gif1->gif_img_index * gif1->gif_w * gif1->gif_h * 2));

    }

    if (gif1->gif_type == 1)
    {
        uint32_t img_data_addr = es_spiflash_read_data32(gif1->gif_data_spiflash_addr + (frame_id << 3));
        uint32_t img_data_len = es_spiflash_read_data32(gif1->gif_data_spiflash_addr + (frame_id << 3) + 4);

        lv_rle16_decompress(gif1->pos_x, gif1->pos_x + gif1->gif_w - 1, \
                            gif1->pos_y, gif1->pos_y + gif1->gif_h - 1, \
                            img_data_addr + g_es_gui_ext_mem_base_addr, img_data_len);
    }

    es_spi_datasize_8(ES_EXTERN_FLASH_SPI_PERIPHERAL);
}


void es_gif_lv_inv_area(es_gif_info_t *gif1)
{
    lv_area_t area_tmp;

    area_tmp.x1 = gif1->pos_x;
    area_tmp.x2 = gif1->pos_x + gif1->gif_w;
    area_tmp.y1 = gif1->pos_y;
    area_tmp.y2 = gif1->pos_y + gif1->gif_h;
    _lv_inv_area(&area_tmp);
}

void es_ui_set_gif(es_gif_info_t *gif1, uint16_t gif_index, uint16_t x, uint16_t y, uint16_t config, void *obj_screen)
{
    if (gif1->gif_state & ES_GIF_STATE_ENABLE)
    {
        es_gif_lv_inv_area(gif1);
    }

    gif1->gif_state = 0;

    gif1->gif_index = gif_index;
    gif1->gif_conf = config;
    gif1->pos_x = x;
    gif1->pos_y = y;
    gif1->gif_in_screenx_addr = obj_screen;
}

void es_ui_set_gif_diable(es_gif_info_t *gif1, uint8_t lvgl_convert_en)
{
    if (lvgl_convert_en)
    {
        if (gif1->gif_state & ES_GIF_STATE_ENABLE)
        {
            es_gif_lv_inv_area(gif1);
        }
    }

    gif1->gif_state = 0;

    gif1->gif_conf &= ~(ES_GIF_CONF_ENABLE);
}

void es_ui_set_gif_enable(es_gif_info_t *gif1)
{
    gif1->gif_conf |= ES_GIF_CONF_ENABLE;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
