/**********************************************************************************
 *
 * @file    .c
 * @brief
 *
 * @date
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2022.11.30      liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "es_mid_gui_conf.h"
#include "lvgl.h"
#include "es_st7789.h"
#include "es_gt91x.h"
#include "md_csu.h"
#include "md_rmu.h"
#include "md_pis.h"
#include "ald_usb.h"
#include "es_spi_flash.h"
/* Private Macros ------------------------------------------------------------ */

#define PIS_INIT_SPIFLASH_TO_LCD(ES_GUI_LCD_SPI_SCK_PIS_CH,ES_GUI_LCD_SPI_MOSI_PIS_CH,\
								ES_GUI_FLASH_SPI_SCK_PIS_PORT,ES_GUI_FLASH_SPI_SCK_PIS_PIN, \
								ES_GUI_FLASH_SPI_MISO_PIS_PORT,ES_GUI_FLASH_SPI_MISO_PIS_PIN) \
	{	\
    MD_SYSCFG_UNLOCK(); \
    md_syscfg_pisiocfg_io##ES_GUI_FLASH_SPI_SCK_PIS_PIN##sel(SYSCFG, MD_SYSCFG_PISIOCFG_P##ES_GUI_FLASH_SPI_SCK_PIS_PORT); \
    md_syscfg_pisiocfg_io##ES_GUI_FLASH_SPI_MISO_PIS_PIN##sel(SYSCFG, MD_SYSCFG_PISIOCFG_P##ES_GUI_FLASH_SPI_MISO_PIS_PORT); \
    MD_SYSCFG_LOCK(); \
    md_pis_enable_ch##ES_GUI_LCD_SPI_SCK_PIS_CH##_output(PIS);\
    md_pis_enable_ch##ES_GUI_LCD_SPI_MOSI_PIS_CH##_output(PIS);\
    pis_init.p_src     = MD_PIS_GPIO_PIN##ES_GUI_FLASH_SPI_SCK_PIS_PIN;\
    pis_init.p_clk     = MD_PIS_CLK_HCLK;\
    pis_init.p_edge    = MD_PIS_EDGE_NONE;\
    pis_init.p_output  = MD_PIS_OUT_LEVEL;\
    pis_init.c_trig    = ES_GUI_LCD_SPI_SCK_PIS_CH;\
    pis_init.c_clk     = MD_PIS_CLK_HCLK;\
    md_pis_init(&pis_init);\
    pis_init.p_src     = MD_PIS_GPIO_PIN##ES_GUI_FLASH_SPI_MISO_PIS_PIN;\
    pis_init.p_clk     = MD_PIS_CLK_HCLK;\
    pis_init.p_edge    = MD_PIS_EDGE_NONE;\
    pis_init.p_output  = MD_PIS_OUT_LEVEL;\
    pis_init.c_trig    = ES_GUI_LCD_SPI_MOSI_PIS_CH;\
    pis_init.c_clk     = MD_PIS_CLK_HCLK;\
    md_pis_init(&pis_init);\
	}
/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
uint16_t g_dma_used_color;
uint8_t g_last_frame1 = 0;
uint8_t g_last_frame2 = 0;
uint8_t g_timer_int_as_tick = 0;
uint8_t g_last_sof_frame_change_tick = 0;
uint8_t g_usb_sof_frame_continue_change_times = 0;
uint8_t g_usb_connect_flag = 0;
unsigned long long g_gui_timer_last = 0U;

extern uint8_t change_bg_flag;
extern uint32_t change_bg_img_index;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
void es_pis_init(void)
{
    md_pis_init_t pis_init;
    memset(&pis_init, 0, sizeof(md_pis_init_t));
	ES_PIS_SPIFLASH_TO_LCD_INIT
}

void es_init_mcu_pin(void)
{
#if ES_GUI_USE_TOUCH
#if ES_GUI_USE_TOUCH_GT91X

    md_gpio_init_t gpio_init;

    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP_DOWN;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_STRONG;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;

    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;

    gpio_init.odos  = MD_GPIO_OPEN_DRAIN;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;

    gpio_init.func  = MD_GPIO_FUNC_1;
    md_gpio_init(ES_TOUCH_I2C_SDA_PIN_PORT, ES_TOUCH_I2C_SDA_PIN_MD_PIN, &gpio_init);

    md_gpio_init(ES_TOUCH_I2C_SCL_PIN_PORT, ES_TOUCH_I2C_SCL_PIN_MD_PIN, &gpio_init);
#endif

#endif

#ifndef ES_LCD_BL_PIN_USER_CONTROL
    md_gpio_set_pin_low(ES_LCD_BL_PIN_PORT, ES_LCD_BL_PIN_MD_PIN);
#endif
	
    md_gpio_set_pin_high(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN);
}

void es_spi_datasize_8(SPI_I2S_TypeDef *SPIx)
{
    md_spi_disable(SPIx);
    md_spi_enable_8_byte_commumication(SPIx);
    md_spi_enable(SPIx);
}

void es_spi_datasize_16(SPI_I2S_TypeDef *SPIx)
{
    md_spi_disable(SPIx);
    md_spi_enable_16_byte_commumication(SPIx);
    md_spi_enable(SPIx);
}

void es_spi_fast(SPI_I2S_TypeDef *SPIx)
{
    md_spi_disable(SPIx);
    md_spi_set_commumication_baud(SPIx, MD_SPI_BAUD_2);
    md_spi_enable(SPIx);
}

void es_spi_normal(SPI_I2S_TypeDef *SPIx)
{
    md_spi_disable(SPIx);
    md_spi_set_commumication_baud(SPIx, MD_SPI_BAUD_4);
    md_spi_enable(SPIx);
}

void es_lcd_wait_spi_over_cs1(void)
{
    while (((READ_REG(ES_LCD_SPI_PERIPHERAL->STAT) & SPI_STAT_TXE_MSK) == 0));

    while (((READ_REG(ES_LCD_SPI_PERIPHERAL->STAT) & SPI_STAT_BUSY_MSK)));

    md_gpio_set_pin_high(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN);
}

void es_st7789_port_lcd_wrte_cmd(uint8_t cmd)
{
    md_gpio_set_pin_low(ES_LCD_CD_PIN_PORT, ES_LCD_CD_PIN_MD_PIN);
    md_gpio_set_pin_low(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN);
    WRITE_REG(ES_LCD_SPI_PERIPHERAL->DATA , cmd);
    es_lcd_wait_spi_over_cs1();
}

void es_st7789_port_lcd_wrte_data(uint8_t data)
{
    md_gpio_set_pin_high(ES_LCD_CD_PIN_PORT, ES_LCD_CD_PIN_MD_PIN);
    md_gpio_set_pin_low(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN);
    WRITE_REG(ES_LCD_SPI_PERIPHERAL->DATA , data);
    es_lcd_wait_spi_over_cs1();
}

#pragma GCC push_options
#pragma GCC optimize("O0" )

void es_lv_lcd_write_area(lv_disp_drv_t *disp_drv, const lv_area_t *area, lv_color_t *color_p)
{
    uint32_t size = (area->x2 - area->x1 + 1) * (area->y2 - area->y1 + 1);
    uint8_t temp8;
    uint16_t dma_transfer_num;
    uint16_t *buf = (uint16_t *)color_p;

    /*等待DMA完成*/
    while (READ_REG(DMA->CHANNEL[ES_LCD_SPI_DMA_CH].CON) & 0x1) {}

    /*等待SPI传输完成*/
    es_lcd_wait_spi_over_cs1();

    /*SPI数据帧长度->8*/
    es_spi_datasize_8(ES_LCD_SPI_PERIPHERAL);

    es_st7789_lcd_write_area_cmd(area->x1, area->x2, area->y1, area->y2);

    /*SPI数据帧长度->16*/
    es_spi_datasize_16(ES_LCD_SPI_PERIPHERAL);

    /*LCD写像素数据+SPI片选*/
    md_gpio_set_pin_high(ES_LCD_CD_PIN_PORT, ES_LCD_CD_PIN_MD_PIN);
    md_gpio_set_pin_low(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN);

    md_dma_set_source_inc(ES_LCD_SPI_DMA_CH, 1);

    /*SPI发送DMA开始*/
    while (size)
    {
        dma_transfer_num = (size > 65535) ? 65535 : size;
        md_dma_set_source_addr(ES_LCD_SPI_DMA_CH, buf);
        md_dma_set_transfer_size(ES_LCD_SPI_DMA_CH, dma_transfer_num);
        buf += dma_transfer_num;

        size -= dma_transfer_num;
        md_dma_enable_channel(ES_LCD_SPI_DMA_CH);
		
/*
#if ES_LVGL_USE_2_DISP_BUF		
		if(size == 0)
			break;
#endif 
*/
		
        /*等待DMA完成*/
        while (READ_REG(DMA->CHANNEL[ES_LCD_SPI_DMA_CH].CON) & 0x1)
        {
        }
    }

    lv_disp_flush_ready(0);

    (void)temp8;
}

#pragma GCC pop_options

void es_st7789_write_area_one_color(uint16_t x1, uint16_t x2, uint16_t y1, uint16_t y2, uint16_t color)
{
    uint32_t size = (x2 - x1 + 1) * (y2 - y1 + 1);
    uint8_t temp8;
    uint16_t dma_transfer_num;

    g_dma_used_color = color;

    md_gpio_set_pin_high(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN);

    /*SPI数据帧长度->8*/
    es_spi_datasize_8(ES_LCD_SPI_PERIPHERAL);

    es_st7789_lcd_write_area_cmd(x1, x2, y1, y2);

    /*SPI数据帧长度->16*/
    es_spi_datasize_16(ES_LCD_SPI_PERIPHERAL);

    /*LCD写像素数据+SPI片选*/
    md_gpio_set_pin_high(ES_LCD_CD_PIN_PORT, ES_LCD_CD_PIN_MD_PIN);
    md_gpio_set_pin_low(ES_LCD_SPI_SOFT_NSS_PIN_PORT, ES_LCD_SPI_SOFT_NSS_PIN_MD_PIN);

    /*SPI发送DMA开始*/
    md_dma_set_source_addr(ES_LCD_SPI_DMA_CH, (void *)&g_dma_used_color);
    md_dma_set_source_inc(ES_LCD_SPI_DMA_CH, 0);

    while (size)
    {
        dma_transfer_num = (size > 65535) ? 65535 : size;
        md_dma_set_transfer_size(ES_LCD_SPI_DMA_CH, dma_transfer_num);

        size -= dma_transfer_num;
        md_dma_enable_channel(ES_LCD_SPI_DMA_CH);

        /*等待DMA完成*/
        while (READ_REG(DMA->CHANNEL[ES_LCD_SPI_DMA_CH].CON) & 0x1)
        {
        }
    }

    /*等待SPI传输完成*/
    es_lcd_wait_spi_over_cs1();

    /*SPI数据帧长度->8*/
    es_spi_datasize_8(ES_LCD_SPI_PERIPHERAL);

    (void)temp8;
}

#if ES_GUI_USE_TOUCH

void es_init_touch_gt91x(void)
{
    gt911_touch_reset();
    gt911_touch_default_config();
}

#endif

void es_init_lcd_st7789(void)
{
    es_st7789_lcd_init();
	
    md_spi_enable_txdma(ES_LCD_SPI_PERIPHERAL);
	md_dma_set_dest_addr(ES_LCD_SPI_DMA_CH, (void*)(&(ES_LCD_SPI_PERIPHERAL->DATA)));
}

void es_mcu_init(void)
{
    MD_SYSCFG_UNLOCK();
#if ES_USE_USB_MSC_DEV_UPDATE_SPI_FLASH
    md_cmu_enable_usbck48();        //USB 48M使能
    md_csu_enable_auto();
    md_csu_enable_cnt();
#endif
    MD_SYSCFG_LOCK();
#if ES_USE_USB_MSC_DEV_UPDATE_SPI_FLASH
    md_rmu_reset_periperal(MD_RMU_PERH_USB);
#endif

	es_pis_init();

    es_init_mcu_pin();

    es_init_extern_flash();
    es_init_lcd_st7789();

#if ES_GUI_USE_TOUCH
    es_init_touch_gt91x();
#endif
}

void es_delay_1ms(uint32_t ms)
{
    md_delay_1us(ms * 1000);
}

void es_gui_systick_irq_handle(void)
{
	uint32_t time_ms = md_cmu_get_clock()/1000;
	uint32_t time_change;
	
	time_change = (CORET->MTIME - g_gui_timer_last)/time_ms;
	
	if(time_change == 0)
	{
		return;
	}
	
	g_gui_timer_last += time_change * time_ms;
	
    lv_tick_inc(time_change);

#if ES_USE_USB_MSC_DEV_UPDATE_SPI_FLASH
    g_timer_int_as_tick++;

    {
        if ((USB->FRAME1 != g_last_frame1) || (USB->FRAME2 != g_last_frame2))
        {
            g_last_frame1 = USB->FRAME1;
            g_last_frame2 = USB->FRAME2;

            g_last_sof_frame_change_tick = g_timer_int_as_tick;

            if (g_usb_sof_frame_continue_change_times < 3)
            {
                g_usb_sof_frame_continue_change_times++;
            }
            else
            {
                g_usb_connect_flag = 1;
            }
        }
        else
        {
            if (g_timer_int_as_tick - g_last_sof_frame_change_tick > 3)
            {
                g_last_sof_frame_change_tick = g_timer_int_as_tick - 3;
                g_usb_sof_frame_continue_change_times = 0;
                g_usb_connect_flag = 0;
            }
        }
    }
#endif
}

void jump2boot(void)
{
    __disable_irq();

    ald_usb_int_unregister();

    MD_SYSCFG_UNLOCK();
    md_syscfg_set_cpu_boot_addr(0);
    MD_SYSCFG_LOCK();

    __enable_irq();

    md_rmu_reset_system();
}

void es_lcd_all_color(uint16_t color)
{
    es_st7789_write_area_one_color(0, ES_LVGL_LCD_DISP_HOR_RES - 1, 0, ES_LVGL_LCD_DISP_VER_RES - 1, color);
}

__weak int _write(int file, const void *ptr, int len)
{
	return 0;
}

__weak void _putchar(char character)
{
	char c = character;
	_write(0,&c,1);
}

void es_spiflash_update(void)
{
#if ES_USE_USB_MSC_DEV_UPDATE_SPI_FLASH

    if (g_usb_connect_flag)
    {
        jump2boot();
    }

#endif
}

void gui_main_init(void)
{
    /*初始化芯片*/
    es_mcu_init();

#if ES_USE_USB_MSC_DEV_UPDATE_SPI_FLASH

    ald_usb_otg_session_request(true);
    ald_usb_dppud_set(ALD_USB_DPDM_PUSH_UP);
    /* software control CID */
    ald_usb_swcid_cidctrl(1);
    /* force to dev mode */
    ald_usb_swcid_host(1);
    ald_usb_dev_connect();

#endif

    extern void es_spiflash_base_addr_refresh(void);
    es_spiflash_base_addr_refresh();

    es_lvgl_init();

    ui_init();

    lv_task_handler();

#ifndef ES_LCD_BL_PIN_USER_CONTROL
    md_gpio_set_pin_high(ES_LCD_BL_PIN_PORT, ES_LCD_BL_PIN_MD_PIN);
#endif
}

void spiflash_data_err_handle(void)
{
    while (1)
    {
		es_gui_systick_irq_handle();
        es_spiflash_update();
    }
}

void gui_main_loop(void)
{
	es_gui_systick_irq_handle();
    /* handle the tasks of LVGL */
    lv_task_handler();

    es_spiflash_update();
}
