/*******************************************************************************
 * Copyright (c) 2012-2019 Hangzhou C-SKY Microsystems Co., Ltd.
 * 
 * All rights reserved. This software is proprietary and confidential to
 * Hangzhou C-SKY Microsystems Co., Ltd. and its licensors.
 *
 * Contributors:
 *     Hangzhou C-SKY Microsystems Co., Ltd.
 *
 * 2019.6.18   Jiang Long(long_jiang@c-sky.com)
 *     Initial API and implementation
 *******************************************************************************/

/**
 * Driver for flash program.
 */

#include "spi_flash.h"

#define	FLASH_PAGE_SIZE	0x1000
#define	WRITE_PAGE_SIZE	0x100 
 
typedef volatile unsigned char    vu8;
typedef          unsigned char     u8;
typedef volatile unsigned short   vu16;
typedef          unsigned short    u16;
typedef volatile unsigned long    vu32;
typedef          unsigned long     u32;

#define M8(adr)  (*((vu8  *) (adr)))
#define M16(adr) (*((vu16 *) (adr)))
#define M32(adr) (*((vu32 *) (adr)))

#define MSC_BASE     0x40081000        //ok
#define MSC          ((MSC_TypeDef*) MSC_BASE)

#define SYSCFG_PROT_ADR     0x40080000
#define TESTKEY0            0x5A962814
#define TESTKEY1            0xE7CB69A5
#define CMU_CSR_ADR         0x40080400 //ok

#define IWDT_LOCK_ADR       0x40008c00
#define IWDT_INTCLR_ADR     0x40008c0C
#define IWDT_CLR_DAT        0x00000001
#define IWDT_UNLOCK_DAT     0x1ACCE551
#define IWDT_LOCK_DAT       0xFFFFFFFF


// MSC Registers
typedef struct {
	vu32 FKEY;     	        	// offset  0x000, RW
	vu32 IKEY;                  // offset  0x004, RW
	vu32 FADDR;                 // offset  0x008, RW
	vu32 FFIFO;                 // offset  0x00C, WO
	vu32 FDL;                   // offset  0x010, RW
	vu32 FDH;                   // offset  0x014, RW
	vu32 FCMD;                  // offset  0x018, RW
	vu32 FCR;                   // offset  0x01C, RW
	vu32 FSR;                   // offset  0x020, RW
	vu32 FPL;                   // offset  0x024, RW
	vu32 FWAIT;                 // offset  0x028, RW
	vu32 FADDRINV;              // offset  0x02C, RW
} MSC_TypeDef;

#define BIT0			0x00000001
#define BIT1			0x00000002
#define BIT2			0x00000004
#define BIT3			0x00000008
#define BIT4			0x00000010
#define BIT5			0x00000020
#define BIT6			0x00000040
#define BIT7			0x00000080
#define BIT8			0x00000100

//----------------
// MSC->FKEY:
//----------------
#define FKEY0               0x8ACE0246
#define FKEY1               0x9BDF1357

//----------------
// MSC->IKEY:
//----------------
#define IKEY0               0x7153BFD9
#define IKEY1               0x0642CEA8

//----------------
// MSC->FCR:
//----------------
#define CMD_MERS            0x000051AE
#define CMD_SERS            0x00005EA1
#define CMD_WRDP            0x00005DA2
#define CMD_FSTP            0x00005CA3
#define IAP_START           0x00000011
#define IAP_RST             0x00000002
#define IAP_FREQ            0x00000010
#define IAP_FREQ_CLR        0xffffffef

//----------------
// MSC->IKEY:
//----------------
#define MAX_LEN            64

//----------------
// MSC->FSR:
//----------------
#define IAP_FACK_F          BIT0
#define IAP_BUSY_F          BIT1
#define IAP_MERS_F          BIT4
#define IAP_SERS_F          BIT5
#define IAP_WRDP_F          BIT6
#define IAP_FSTP_F          BIT7
#define IAP_FSTPREQ_F       BIT8

#define SCU_PROT_DATA       0x55aa6996
#define CLR_SYS				0xffffff00
#define CLKEN_PERI_DATA     0x00000087

/**
 * ERROR TYPE. MUST NOT BE MODIFIED
 */
#define ERROR_INIT      -200
#define ERROR_READID    -201
#define ERROR_PROGRAM   -202
#define ERROR_READ      -203
#define ERROR_ERASE     -204
#define ERROR_CHIPERASE -205
#define ERROR_UNINIT	-206
#define ERROR_CHECKSUM  -207

/**
 * Customize this method to perform any initialization
 * needed to access your flash device.
 *
 * @return: if this method returns an error,MUST RUTURN ERROR_INIT,
 * Otherwise return 0.
 */
int  flashInit(){
	M32( SYSCFG_PROT_ADR )  = SCU_PROT_DATA;
	M32( CMU_CSR_ADR     )  = 0x1; // select HRC as source clock

	M32( IWDT_LOCK_ADR   )  = IWDT_UNLOCK_DAT;
	M32( IWDT_INTCLR_ADR )  = IWDT_CLR_DAT;
	M32( IWDT_LOCK_ADR   )  = IWDT_LOCK_DAT;

    mcu_spi_init();     /* Initialize the SPI module */

    return 0;
}

/**
 * Customize this method to perform any un-initialization
 * needed to access your flash device.
 *
 * @return: if this method returns an error,MUST RUTURN ERROR_UNINIT,
 * Otherwise return 0.
 */
int  flashUnInit(){

    return 0;
}

/**
 * Customize this method to read flash ID
 *
 * @param flashID: returns for flash ID
 *
 * @return: if this method returns an error,MUST RUTURN ERROR_READID,
 * Otherwise return 0.
 */
int  flashID(unsigned int* flashID){
	// TODO 
  return 0;
}

/**
 * This method takes the data pointed to by the src parameter
 * and writes it to the flash blocks indicated by the
 * dst parameter.
 *
 * @param dst : destination address where flash program
 * @param src : address of data
 * @param length : data length
 *
 * @return : if this method returns an error,MUST RUTURN ERROR_PROGRAM,
 * Otherwise return 0.
 */
int flashProgram( volatile char* dst,volatile char *src, volatile int length){
    md_status_t status;
	unsigned int n_page, length_lastpage, w_addr;

    w_addr = (u32)dst & ~(WRITE_PAGE_SIZE - 1);

	if((u32)dst > 0x800000) 	// address overflow 
		return 1;
	
	if(length == 0)
		return 1;

    n_page = length / WRITE_PAGE_SIZE;
	length_lastpage = length % WRITE_PAGE_SIZE;

    while(n_page--)
	{
        status = flash_write(w_addr, src, WRITE_PAGE_SIZE);
	    if(status != MD_OK)
		    return 1;

        if (status == MD_OK)
            flash_wait_unbusy();

        w_addr += WRITE_PAGE_SIZE;
		src += WRITE_PAGE_SIZE;
	}
	status = flash_write(w_addr, src, length_lastpage);
	if(status != MD_OK)
		return 1;

	if (status == MD_OK)
		flash_wait_unbusy();
	
	return (0);	 // ALL WORDS PROG DONE
}

/**
 * Customize this method to read data from a group of flash blocks into a buffer
 *
 * @param dst : reads the contents of those flash blocks into the address pointed to by
 * the dst parameter.
 * @param src : a pointer to a single flash.
 * @param length : data length
 *
 *  @return: if this method returns an error,MUST RUTURN ERROR_READ,
 * Otherwise return 0.
 */
int flashRead(char* dst, char *src, int length){
    md_status_t status;

    status = flash_read((u32)dst, src, length);
	if(status != MD_OK)
		return 1;

    return 0;
}

/**
 * Customize this method to erase a group of flash blocks.
 *
 * @param dst : a pointer to the base of the flash device.
 * NOTE: dst will always be sector aligned, the sector size is stored in FlashDev.c#FlashDevices#Devs#PageSize
 * @param length : erase length
 * NOTE: length will always be sector aligned, the sector size is stored in FlashDev.c#FlashDevices#Devs#PageSize
 *
 * @return : if this method returns an error,MUST RUTURN ERROR_ERASE,
 * Otherwise return 0
 */
int flashErase(char *dst, int length){
    md_status_t status;
	unsigned int n_page, e_addr;

    e_addr = (u32)dst & ~(FLASH_PAGE_SIZE - 1);

	n_page = length / FLASH_PAGE_SIZE;
	if(length == 0)
		return 1;
	else if(length & (FLASH_PAGE_SIZE - 1))
		n_page += 1;

	while(n_page--)
	{
		status = flash_sector_erase(e_addr);
	    if(status != MD_OK)
		    return 1;

        if (status == MD_OK)
            flash_wait_unbusy();

        e_addr += FLASH_PAGE_SIZE;
	}

	return 0;
}

/**
 * Customize this method to erase the whole flash.
 *
 * @return : if this method returns an error,MUST RUTURN ERROR_CHIPERASE,
 * Otherwise return 0.
 */
int flashChipErase( ){

	return 0;
}

/**
 * Customize this method to make the veryfiy process more quickly.
 * 
 * @param dst : a pointer to the base of the flash device.
 * NOTE: dst will always be 4 aligned.
 * @param length : the lenght of the data which will be used for checksum
 * NOTE: the length will always be 4 aligned.
 * @param checksum : the expected checksum value in the programmed file(ihex,bin,elf format)
 * 
 * @return : if the specified data's checksum from dst and lenght is checksum, return 0, else return ERROR_CHECKSUM
 * @example if the flash can be read directly, you can copy following code to replace current "return 0;"
 *
  int i, sum = 0;
  for (i = 0; i < length; i++) {
   sum += dst[i];
  }
  return sum == checksum ? 0 : ERROR_CHECKSUM;
 * 
 */
int flashChecksum(char*dst, int length, int checksum) {
    md_status_t status;
    int i, sum = 0;
    char flash_rxbuf[1];

    for (i = 0; i < length; i++)
	{
        status = flash_read((u32)dst + i, flash_rxbuf, 1);
	    if(status != MD_OK)
		    return 1;

        sum += flash_rxbuf[0];
	}

	return sum == checksum ? 0 : ERROR_CHECKSUM;
}

// NOTING: when debug the driver, this macro defined as 1, and then
// it must be set as 0, for release to flash programmer library
#define DEBUG_DRIVER	0

/**
 * Debug entry for driver.
 *
 * @return : if this method returns an error,MUST RUTURN ERROR_CHIPERASE,
 * Otherwise return 0.
 */
int flashTest(){
#if DEBUG_DRIVER

	unsigned int ID;
	// read flash id
	flashID(&ID);

    // other drivers test
    return ID;
#else
	return 0;
#endif
}

