/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    9 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Jan. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "es_st7789.h"
#include "spi_flash.h"
#include "printf.h"

/* Private Macros ------------------------------------------------------------ */
#define printf printf_/*Reduce ROM usage*/

#define FLASH_CS_SET()  md_gpio_write_pin(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN, 1);
#define FLASH_CS_CLR()  md_gpio_write_pin(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN, 0);
#define FLASH2LCD_SPI_P2P_DMA_CH_INDEX   MD_DMA_CH_3

/* Private Variables --------------------------------------------------------- */

static md_spi_init_t s_spi;
md_dma_config_t g_dma_rx_config;

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref md_status_t.
  */
md_status_t flash_wait_unbusy(void)
{
    uint8_t status;
    int r_flag = 0;

    FLASH_CS_CLR(); /* Choose lower, the selected Flash */

    if (md_spi_send_byte_fast(&s_spi, (uint8_t)FLASH_STATUS) != MD_OK)   /*Send to read status command*/
    {
        FLASH_CS_SET();     /* Pick up and release of Flash */
        return MD_ERROR;
    }

    do
    {
        status = md_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }
    while (status & 0x01);

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t flash_read_id(void)
{
    uint8_t i;
    int r_flag = 0;
    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    FLASH_CS_CLR(); /* Choose lower, the selected Flash */

    for (i = 0; i < sizeof(flash_id); i++)
    {
        if (md_spi_send_byte_fast(&s_spi, flash_id[i]) != MD_OK)
        {
            FLASH_CS_SET();     /* Pick up and release of Flash */
            return MD_ERROR;
        }
    }

    for (i = 0; i < 3; i++)
    {
        flash_id[i] = md_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    FLASH_CS_SET();

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /*Manufacturer ID flash_id [0] and device ID flash_id [1]*/
}

/**
  * @brief spi function
  * @retval None.
  */
void es_init_extern_flash(void)
{
    md_dma_config_t spi_dma_tx_config;
    md_dma_config_t spi_dma_rx_config;

    md_spi_struct_init(&s_spi);
    s_spi.SPIx      = ES_EXTERN_FLASH_SPI_PERIPHERAL;                 /* Using SPI0 */
    s_spi.mode      = MD_SPI_MODE_MASTER;   /* SPI host mode */
    s_spi.baud      = MD_SPI_BAUD_2;      /* clock */
    s_spi.data_size = MD_SPI_DATA_SIZE_8;   /* 8 bit pattern */
    s_spi.polarity  = MD_SPI_CPOL_HIGH;     /* Free high level */
    s_spi.phase     = MD_SPI_CPHA_SECOND;   /* The second edge receiving data */
    s_spi.first_bit = MD_SPI_FIRSTBIT_MSB;  /* Send the MSB first */
    s_spi.dir       = MD_SPI_DIRECTION_2LINES;
    s_spi.ss_en     = DISABLE;
    s_spi.crc_calc  = DISABLE;
    s_spi.crc_poly  = 0;
    md_spi_init(&s_spi);                    /* According to the parameter initialization SPI peripherals */

    memset(&spi_dma_tx_config, 0x0, sizeof(md_dma_config_t));
    memset(&spi_dma_rx_config, 0x0, sizeof(md_dma_config_t));

}

/**
  * @brief  flash write enable function.
  * @retval None.
  */
static uint8_t flash_write_enable(void)
{
    /* wait until flash unbusy */
    flash_wait_unbusy();

    /* enable the selected Flash */
    FLASH_CS_CLR();

    /* send write enable command */
    if (md_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != MD_OK)
    {
        FLASH_CS_SET();
        return 1;
    }

    /* release Flash */
    FLASH_CS_SET();

    /* wait until flash unbusy */
    flash_wait_unbusy();

    return 0;
}

uint8_t ll_flash_chip_erase(void)
{
    uint8_t  cmd_buf;
    uint8_t  i = 0U;
    uint32_t sec_addr;

    /*Flash sector erase command*/
    cmd_buf = FLASH_ERASE_CHIP;
	
	/* enable flash write */
	flash_write_enable();

	/* erase current sector */
	/* enable the selected Flash */
	FLASH_CS_CLR();

	/*Send the sector erase instructions and Flash address three bytes*/

	if (md_spi_send_byte_fast(&s_spi, cmd_buf) != MD_OK)
	{
		FLASH_CS_SET();
		return 1;
	}

	/* release Flash */
	FLASH_CS_SET();

	/* wait until flash not in write progress */
	flash_wait_unbusy();
}

/**
  * @brief  low-level flash sector erase function.
  * @param  start_sec: Specified start sector to be erase.
  * @param  end_sec: Specified end sector to be erase.
  * @retval Status.
  */
uint8_t ll_flash_sector_erase(uint32_t start_sec, uint32_t end_sec)
{
    uint8_t  cmd_buf[4];
    uint8_t  i = 0U;
    uint32_t sec_addr;

    /*Flash sector erase command*/
    cmd_buf[0] = FLASH_SEC_ERASE;

    /*Erase Flash sectors */
    for (; start_sec <= end_sec; start_sec++)
    {
        /* group command and address */
        sec_addr = start_sec * SEC_SIZE;
        cmd_buf[1] = (sec_addr >> 16) & 0xFF;   /*24 bit Flash address*/
        cmd_buf[2] = (sec_addr >> 8) & 0xFF;
        cmd_buf[3] = (sec_addr) & 0xFF;

        /* enable flash write */
        flash_write_enable();

        /* erase current sector */
        /* enable the selected Flash */
        FLASH_CS_CLR();

        /*Send the sector erase instructions and Flash address three bytes*/
        for (i = 0; i < sizeof(cmd_buf); i++)
        {
            if (md_spi_send_byte_fast(&s_spi, cmd_buf[i]) != MD_OK)
            {
                FLASH_CS_SET();
                return 1;
            }
        }

        /* release Flash */
        FLASH_CS_SET();

        /* wait until flash not in write progress */
        flash_wait_unbusy();
    }

    return 0;
}

uint8_t ll_flash_write(const uint8_t *buf, uint32_t addr, uint32_t size)
{
    uint8_t  cmd_buf[4];
    uint32_t  i, len, w_addr = addr;
    uint32_t remain_size = size;

    if (buf == NULL)
        return 1;

    /*Flash page program command*/
    cmd_buf[0] = FLASH_PAGE_PROGRAM;

    while (remain_size)
    {
        cmd_buf[1] = (w_addr >> 16) & 0xff;
        cmd_buf[2] = (w_addr >> 8) & 0xff;
        cmd_buf[3] = (w_addr) & 0xff;

        len = PAGE_SIZE - (w_addr % PAGE_SIZE);
        len = (len >= remain_size) ? (remain_size) : (len);

        /* enable flash write */
        flash_write_enable();

        /* write current page */
        /* enable the selected Flash */
        FLASH_CS_CLR();

        /*Send the sector erase instructions and Flash address three bytes*/
        for (i = 0; i < sizeof(cmd_buf); i++)
        {
            if (md_spi_send_byte_fast(&s_spi, cmd_buf[i]) != MD_OK)
            {
                FLASH_CS_SET();
                return 1;
            }
        }

        /* write the data to the Flash*/
        for (i = 0; i < len; i++)
        {
            if (md_spi_send_byte_fast(&s_spi, buf[i - remain_size + size]) != MD_OK)
            {
                FLASH_CS_SET();
                return 1;
            }
        }

        /* release Flash */
        FLASH_CS_SET();

        /* wait until flash not in write progress */
        flash_wait_unbusy();

        remain_size -= len;
        w_addr += PAGE_SIZE;
    }

    return 0;
}


/**
  * @brief  low-level flash sector write operation function.
  * @param  src_buf: pointer to the expected written data.
  * @param  sec: start sector to write.
  * @param  sec_num: amount of sector.
  * @retval Status, see @ref ald_status_t.
  */
uint8_t ll_flash_sector_write(const uint8_t *src_buf, uint32_t sec, uint32_t sec_num)
{
    uint8_t  cmd_buf[4];
    uint32_t  i, j, k;
    uint32_t page_addr, sec_addr;

    if (src_buf == NULL)
        return 1;

    /*Flash page program command*/
    cmd_buf[0] = FLASH_PAGE_PROGRAM;

    /*Erase Flash sectors */
    for (i = 0; i < sec_num; i++)
    {
        /* calculate current sector start address */
        sec_addr = sec * SEC_SIZE;

        for (j = 0; j < PAGE_PER_SEC; j++)
        {
            /* group command and address */
            page_addr = sec_addr + PAGE_SIZE * j;
            cmd_buf[1] = (page_addr >> 16) & 0xff;
            cmd_buf[2] = (page_addr >> 8) & 0xff;
            cmd_buf[3] = (page_addr) & 0xff;

            /* enable flash write */
            flash_write_enable();

            /* write current page */
            /* enable the selected Flash */
            FLASH_CS_CLR();

            /*Send the sector erase instructions and Flash address three bytes*/
            for (k = 0; k < sizeof(cmd_buf); k++)
            {
                if (md_spi_send_byte_fast(&s_spi, cmd_buf[k]) != MD_OK)
                {
                    FLASH_CS_SET();
                    return 1;
                }
            }

            /* write the data to the Flash*/
            for (k = 0; k < PAGE_SIZE; k++)
            {
                if (md_spi_send_byte_fast(&s_spi, src_buf[i * SEC_SIZE + j * PAGE_SIZE + k]) != MD_OK)
                {
                    FLASH_CS_SET();
                    return 1;
                }
            }

            /* release Flash */
            FLASH_CS_SET();

            /* wait until flash not in write progress */
            flash_wait_unbusy();
        }
    }

    return 0;
}

/**
  * @brief  low-level flash sector read operation function.
  * @param  str_buf: pointer to the first item of the byte array to store read data.
  * @param  sec: start sector.
  * @param  sec_num: amount of sector to be read.
  * @retval Status, see @ref ald_status_t.
  */
uint8_t ll_flash_sector_read(uint8_t *str_buf, uint32_t addr, uint32_t num)
{
    uint32_t i, j;
    uint8_t tmp;

    if (str_buf == NULL)
        return 1;

    while (((DMA->CHANNEL[0].CON) & 0x1) == 0x1) {}

    FLASH_CS_CLR();

    while (ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_TXFLV_MSK);

    while (((ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_BUSY_MSK) == SPI_STAT_BUSY_MSK)) {}

    while (ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_RXFLV_MSK)
    {
        tmp = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
    }

    ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = (uint8_t)(FLASH_READ);
    ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = (uint8_t)((addr >> 16) & 0xFF);
    ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = (uint8_t)((addr >> 8) & 0xFF);
    ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = (uint8_t)((addr) & 0xFF);

    i = 0;

    while (ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_TXFLV_MSK);

    while (((ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_BUSY_MSK) == SPI_STAT_BUSY_MSK)) {}

    while (ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_RXFLV_MSK)
    {
        tmp = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
    }

    while (num != i)
    {
        j = ((ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_TXFLV_MSK) >> SPI_STAT_TXFLV_POSS) ;

#if (ES_EXTERN_FLASH_SPI_PERIPHERAL_FIFO >= 16)

        if (j < 8)
        {
            ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = 0xFF;
            ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = 0xFF;
            ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = 0xFF;
            ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = 0xFF;
            ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = 0xFF;
            ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = 0xFF;
            ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = 0xFF;
            ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = 0xFF;
        }

#else

        for (; j < ES_EXTERN_FLASH_SPI_PERIPHERAL_FIFO; j++)
        {
            ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA = 0xFF;
        }

#endif/*(ES_EXTERN_FLASH_SPI_PERIPHERAL_FIFO >= 16)*/

#if (ES_EXTERN_FLASH_SPI_PERIPHERAL_FIFO >= 16)

        j = ((ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_RXFLV_MSK) >> SPI_STAT_RXFLV_POSS);

        if ((num - i >= j))
        {
            if (j >= 8)
            {
                str_buf[i++] = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
                str_buf[i++] = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
                str_buf[i++] = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
                str_buf[i++] = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
                str_buf[i++] = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
                str_buf[i++] = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
                str_buf[i++] = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
                str_buf[i++] = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
            }
        }
        else
        {
            while ((ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_RXFLV_MSK) && (num != i))
            {
                str_buf[i++] = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
            }
        }

#else

        while ((ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_RXFLV_MSK) && (num != i))
        {
            str_buf[i++] = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
        }

#endif /*(ES_EXTERN_FLASH_SPI_PERIPHERAL_FIFO >= 16)*/

    }

    while (ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_RXFLV_MSK)
    {
        tmp = ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;

        if (((ES_EXTERN_FLASH_SPI_PERIPHERAL->STAT & SPI_STAT_BUSY_MSK) == SPI_STAT_BUSY_MSK))
            break;
    }

    (void)tmp;
    /* release Flash */
    FLASH_CS_SET();

    return 0;
}

void es_st7789_write_area_flash_spi_perh2perh(uint16_t x1, uint16_t x2, uint16_t y1, uint16_t y2,  uint32_t addr)
{
    uint32_t size = (x2 - x1 + 1) * (y2 - y1 + 1);
    uint8_t temp8;
    uint32_t cnt;
    uint16_t dma_transfer_num;
    uint8_t cmd_buf[4];
    uint16_t i = 0U;

    /*LCD写像素数据+SPI片选*/
    es_st7789_lcd_write_area_cmd(x1, x2, y1, y2);

    ES_LCD1_SET_DATA()
    ES_LCD_SPI_CS_OUTPUT_LOW()

    cmd_buf[0] = FLASH_READ;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    FLASH_CS_CLR();     /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(cmd_buf); i++)   /*Send the editor & reader instructions and Flash address three bytes*/
    {
        if (md_spi_send_byte_fast(&s_spi, cmd_buf[i]) != MD_OK)
        {
            FLASH_CS_SET();     /*Pick up and release of Flash*/
            return;
        }
    }

    /*SPI数据帧长度->16*/
    ES_LCD_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_SPIEN_MSK));
    ES_LCD_SPI_PERIPHERAL->CON1 |= (SPI_CON1_FLEN_MSK);
    ES_LCD_SPI_PERIPHERAL->CON1 |= (SPI_CON1_SPIEN_MSK);
    ES_EXTERN_FLASH_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_SPIEN_MSK));
    ES_EXTERN_FLASH_SPI_PERIPHERAL->CON1 |= (SPI_CON1_FLEN_MSK);
    ES_EXTERN_FLASH_SPI_PERIPHERAL->CON1 |= (SPI_CON1_SPIEN_MSK);

    md_dma_enable_it_tc(FLASH2LCD_SPI_P2P_DMA_CH_INDEX);
    memset(&g_dma_rx_config, 0x0, sizeof(md_dma_config_t));
    g_dma_rx_config.src            = (void *)&ES_EXTERN_FLASH_SPI_PERIPHERAL->DATA;
    g_dma_rx_config.dst            = (void *)&ES_LCD_SPI_PERIPHERAL->DATA;
    g_dma_rx_config.size           = 2;
    g_dma_rx_config.src_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    g_dma_rx_config.dst_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    g_dma_rx_config.src_inc        = DISABLE;
    g_dma_rx_config.dst_inc        = DISABLE;
    g_dma_rx_config.R_power        = MD_DMA_R_POWER_1;
    g_dma_rx_config.priority       = MD_DMA_LOW_PRIORITY;
    g_dma_rx_config.mem_to_mem     = DISABLE;
    g_dma_rx_config.circle_mode    = DISABLE;
    g_dma_rx_config.msel           = MD_DMA_MSEL_SPI0;
    g_dma_rx_config.msigsel        = MD_DMA_MSIGSEL_SPI_RNR;
    md_dma_init(FLASH2LCD_SPI_P2P_DMA_CH_INDEX, &g_dma_rx_config);

    md_spi_enable_rxdma(ES_EXTERN_FLASH_SPI_PERIPHERAL);
    md_spi_enable_recv_only(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    /*SPI发送DMA开始*/
    g_es_lvgl_lcd_tx_complete = 0;
    g_es_lvgl_lcd_tx_num = size;

    while (g_es_lvgl_lcd_tx_num)
    {
        dma_transfer_num = (g_es_lvgl_lcd_tx_num > 65535) ? 65535 : g_es_lvgl_lcd_tx_num;
        md_dma_set_transfer_size(FLASH2LCD_SPI_P2P_DMA_CH_INDEX, dma_transfer_num);

        g_es_lvgl_lcd_tx_num -= dma_transfer_num;
        md_dma_enable_channel(FLASH2LCD_SPI_P2P_DMA_CH_INDEX);

        cnt = 10000000U;

        /*等待DMA完成*/
        while ((!g_es_lvgl_lcd_tx_complete) && (--cnt))
        {
        }
    }

    cnt = 40000U;

    while (md_spi_is_active_flag_busy(SPI1) && --cnt);

    md_dma_disable_channel(FLASH2LCD_SPI_P2P_DMA_CH_INDEX);
    md_spi_disable_rxdma(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    /*等待SPI传输完成*/
    while (((ES_LCD_SPI_PERIPHERAL->STAT & SPI_STAT_TXE_MSK) == 0)) {}

    while (((ES_LCD_SPI_PERIPHERAL->STAT & SPI_STAT_BUSY_MSK) == SPI_STAT_BUSY_MSK)) {}

    /*读取SPI接收FIFO直到空*/
    while (ES_LCD_SPI_PERIPHERAL->STAT & SPI_STAT_RXNE_MSK)
    {
        temp8 = (uint8_t)ES_LCD_SPI_PERIPHERAL->DATA;
    }

    ES_LCD_SPI_CS_OUTPUT_HIGH()

    FLASH_CS_SET();     /*Pick up and release of Flash*/

    md_spi_disable_recv_only(ES_EXTERN_FLASH_SPI_PERIPHERAL);

    /*SPI数据帧长度->8*/
    ES_EXTERN_FLASH_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_SPIEN_MSK));
    ES_EXTERN_FLASH_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_FLEN_MSK));
    ES_EXTERN_FLASH_SPI_PERIPHERAL->CON1 |= (SPI_CON1_SPIEN_MSK);
    ES_LCD_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_SPIEN_MSK));
    ES_LCD_SPI_PERIPHERAL->CON1 &= (~(SPI_CON1_FLEN_MSK));
    ES_LCD_SPI_PERIPHERAL->CON1 |= (SPI_CON1_SPIEN_MSK);

    (void)temp8;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
