/**********************************************************************************
 *
 * @file    .c
 * @brief
 *
 * @date
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          2023.04.10      liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "es_xmod.h"
#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
boot_frame_env_t g_frame_env;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Check frame index.
  * @param  idx: frame index.
  * @retval 0-success, other value indicates failed.
  */
static int check_idx(uint8_t idx, boot_frame_env_t *frame_env_t)
{
    uint8_t _idx  = 0U;
    uint8_t next_idx = 0U;

    _idx = frame_env_t->idx;
    next_idx = _idx == 0xFF ? 0 : _idx + 1;

    if ((next_idx != idx) && (idx != _idx))
        return 1;

    return 0;
}

/**
  * @brief  UART send string
  * @param  s: string to send
  * @param  len: string length
  * @param  timeout: tiemout(ms)
  * @retval 0-success 1-timeout
  */
uint32_t uart_send_str(const char *s, uint32_t len, uint32_t timeout)
{
    uint32_t tick, i, status = 0U;

    for (i = 0; i < len; i++)
    {
        tick = md_get_tick() + timeout;

        while ((!md_uart_is_active_flag_tfempty(XMOD_UART)) && (md_get_tick() < tick));

        if (md_get_tick() >= tick)
        {
            status = 1;
            break;
        }

        md_uart_set_send_data8(XMOD_UART, (uint8_t)s[i]);
    }

    return status;
}

uint32_t es32_uart_recv(uint8_t *buf, uint32_t len, uint32_t timeout)
{
    uint32_t status = 0U;

    unsigned long long time;

    for (uint32_t i = 0U; i < len; i++)
    {
        time = CORET->MTIME;

        while ((!md_uart_is_active_flag_rfnempty(XMOD_UART)) && ((CORET->MTIME - time) < (timeout * (md_cmu_get_clock() / 1000))));

        if (((CORET->MTIME - time) > (timeout * (md_cmu_get_clock() / 1000))))
        {
            status = 1;
            break;
        }

        buf[i] = (char)md_uart_recv_data8(XMOD_UART);
    }

    return status;
}

static int xmod_cmd_send(xmod_tx_cmd_t cmd)
{
    uint8_t _cmd = cmd;

    if ((uart_send_str((char *)&_cmd, 1, 100)))
    {
        g_frame_env.state = UART_SEND_ERROR;
        return -1;
    }

    return 0;
}

static int xmod_128_recv(uint32_t start_addr, boot_frame_env_t *frame_env_t)
{
    uint32_t temp;
    uint16_t crc, _crc;
    uint8_t xmode_buf[132];

    if (es32_uart_recv(&frame_env_t->soh, 1, TIME_OUT))
    {
        frame_env_t->state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if (frame_env_t->soh == XMODE_EOT)
    {
        frame_env_t->state = FRAME_SUCCESS;
        return 1;
    }

    if (es32_uart_recv(xmode_buf, 132, TIME_OUT))
    {
        frame_env_t->state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if ((xmode_buf[0] ^ xmode_buf[1]) != 0xFF)
    {
        frame_env_t->state = FRAME_INDEX_ERROR;
        return -2;
    }

    if ((check_idx(xmode_buf[0], frame_env_t)) != 0)
    {
        frame_env_t->state = FRAME_INDEX_ERROR;
        return -2;
    }

    MD_CRC_RESET();

    for (int i = 0; i < 128; i++)
    {
        md_crc_write_data(CRC, xmode_buf[2 + i]);
    }

    _crc = md_crc_get_check_result(CRC);
    crc = (xmode_buf[130] << 8) | xmode_buf[131];

    if (crc != _crc)
    {
        frame_env_t->state = FRAME_VERIFY_ERROR;
        return -3;
    }

    temp = start_addr + (frame_env_t->idx * 128);

    if ((temp % SEC_SIZE) == 0)
    {
        temp = temp / SEC_SIZE;
        ll_flash_sector_erase(temp, temp);
    }

    ll_flash_write((const uint8_t *)&xmode_buf[2], start_addr + (frame_env_t->idx * 128), 128);

    ++frame_env_t->idx;
    return 0;
}

int updating_spiflash(uint32_t start_addr, boot_frame_env_t *frame_env_t)
{
    int ret;

    frame_env_t->idx   = 0;
    frame_env_t->cnt   = 0;
    frame_env_t->state = FRAME_SUCCESS;

    while (md_uart_is_active_flag_rfnempty(XMOD_UART))
    {
        ret = md_uart_recv_data8(XMOD_UART);
    }

    xmod_cmd_send(XMOD_TX_CMD_START);

    while (1)
    {
        ret = xmod_128_recv(start_addr, frame_env_t);

        if ((ret < 0) && (++frame_env_t->cnt < TRY_CNT))
        {
            if (frame_env_t->idx == 0)
                xmod_cmd_send(XMOD_TX_CMD_START);
            else
                xmod_cmd_send(XMOD_TX_CMD_NCK);

            continue;
        }

        if (ret < 0)
        {
            xmod_cmd_send(XMOD_TX_CMD_TERM);
            return ret;
        }

        if (ret == 1)
        {
            xmod_cmd_send(XMOD_TX_CMD_ACK);
            return 0;
        }

        xmod_cmd_send(XMOD_TX_CMD_ACK);
    }
}
void update_info(boot_frame_env_t *frame_env_t)
{
    if (frame_env_t->state == FRAME_SUCCESS)
    {
        printf("\r\nUpdate data sucess !");
    }
    else if (frame_env_t->state == FRAME_INDEX_ERROR)
    {
        printf("\r\nUpdate data error: FRAME_INDEX_ERROR !!!\r\n");
    }
    else if (frame_env_t->state == FRAME_RECEIVE_TIMEOUT)
    {
        printf("\r\nUpdate data error: FRAME_RECEIVE_TIMEOUT !!!\r\n");
    }
    else if (frame_env_t->state == FRAME_VERIFY_ERROR)
    {
        printf("\r\nUpdate data error: FRAME_VERIFY_ERROR !!!\r\n");
    }
    else
    {
    }

    return;
}

void update_spiflash(void)
{
    md_crc_init_t crc_init;
    /* Clear crc_handle_t structure */
    md_crc_init_struct(&crc_init);

    /* Initialize CRC */
    crc_init.mode = MD_CRC_MODE_CCITT;
    crc_init.len = MD_CRC_DATASIZE_8;
    crc_init.order = MD_CRC_BYTORD_LOW;
    crc_init.seed = 0;
    crc_init.chs_inv = DISABLE;
    crc_init.chs_rev = DISABLE;
    crc_init.data_inv = DISABLE;
    crc_init.data_rev = DISABLE;
    md_crc_init(&crc_init);

    memset(&g_frame_env, 0, sizeof(g_frame_env));

    printf("\r\nxmodem start\r\n");

    __disable_irq();

    updating_spiflash(0, &g_frame_env);
    update_info(&g_frame_env);

    __enable_irq();
}


