/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    9 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Jan. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "spi_flash.h"
#include "main.h"
#include "printf.h"

/* Private Macros ------------------------------------------------------------ */
#define printf printf_/*Reduce ROM usage*/

#define ES_EXTERN_FLASH_SPI_PERIPHERAL    (SPI0)

#define ES_EXTERN_FLASH_CS_PIN_PORT      (GPIOA)
#define ES_EXTERN_FLASH_CS_PIN_MD_PIN    (MD_GPIO_PIN_4)
#define ES_EXTERN_FLASH_CS_PIN_MD_FUNC   (MD_GPIO_FUNC_1)

#if USE_BOARD_ES_PDS

    #define ES_EXTERN_FLASH_SPI_SCK_PIN_PORT       (GPIOA)
    #define ES_EXTERN_FLASH_SPI_SCK_PIN_MD_PIN     (MD_GPIO_PIN_5)
    #define ES_EXTERN_FLASH_SPI_SCK_PIN_MD_FUNC    (MD_GPIO_FUNC_2)

    #define ES_EXTERN_FLASH_SPI_MOSI_PIN_PORT       (GPIOA)
    #define ES_EXTERN_FLASH_SPI_MOSI_PIN_MD_PIN     (MD_GPIO_PIN_7)
    #define ES_EXTERN_FLASH_SPI_MOSI_PIN_MD_FUNC    (MD_GPIO_FUNC_2)

    #define ES_EXTERN_FLASH_SPI_MISO_PIN_PORT      (GPIOA)
    #define ES_EXTERN_FLASH_SPI_MISO_PIN_MD_PIN    (MD_GPIO_PIN_6)
    #define ES_EXTERN_FLASH_SPI_MISO_PIN_MD_FUNC   (MD_GPIO_FUNC_2)

#else

    #define ES_EXTERN_FLASH_SPI_SCK_PIN_PORT       (GPIOA)
    #define ES_EXTERN_FLASH_SPI_SCK_PIN_MD_PIN     (MD_GPIO_PIN_1)
    #define ES_EXTERN_FLASH_SPI_SCK_PIN_MD_FUNC    (MD_GPIO_FUNC_4)

    #define ES_EXTERN_FLASH_SPI_MOSI_PIN_PORT       (GPIOA)
    #define ES_EXTERN_FLASH_SPI_MOSI_PIN_MD_PIN     (MD_GPIO_PIN_3)
    #define ES_EXTERN_FLASH_SPI_MOSI_PIN_MD_FUNC    (MD_GPIO_FUNC_4)

    #define ES_EXTERN_FLASH_SPI_MISO_PIN_PORT      (GPIOA)
    #define ES_EXTERN_FLASH_SPI_MISO_PIN_MD_PIN    (MD_GPIO_PIN_2)
    #define ES_EXTERN_FLASH_SPI_MISO_PIN_MD_FUNC   (MD_GPIO_FUNC_4)

#endif/**/

#define FLASH_CS_SET()  md_gpio_write_pin(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN, 1);
#define FLASH_CS_CLR()  md_gpio_write_pin(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN, 0);

/* Private Variables --------------------------------------------------------- */

static md_spi_init_t s_spi;

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref md_status_t.
  */

uint8_t flash_get_state_unbusy(void)
{
    uint8_t status;
    int r_flag = 0;

    FLASH_CS_CLR(); /* Choose lower, the selected Flash */

    md_spi_send_byte_fast(&s_spi, (uint8_t)FLASH_STATUS);
    status = md_spi_recv_byte_fast(&s_spi, &r_flag);

    FLASH_CS_SET();

    return status & 0x01;
}
md_status_t flash_wait_unbusy(void)
{
    uint8_t status;
    int r_flag = 0;

    FLASH_CS_CLR(); /* Choose lower, the selected Flash */

    if (md_spi_send_byte_fast(&s_spi, (uint8_t)FLASH_STATUS) != MD_OK)   /*Send to read status command*/
    {
        FLASH_CS_SET();     /* Pick up and release of Flash */
        return MD_ERROR;
    }

    do
    {
        status = md_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }
    while (status & 0x01);

    FLASH_CS_SET();

    return MD_OK;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t flash_read_id(void)
{
    uint8_t i;
    int r_flag = 0;
    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    FLASH_CS_CLR(); /* Choose lower, the selected Flash */

    for (i = 0; i < sizeof(flash_id); i++)
    {
        if (md_spi_send_byte_fast(&s_spi, flash_id[i]) != MD_OK)
        {
            FLASH_CS_SET();     /* Pick up and release of Flash */
            return MD_ERROR;
        }
    }

    for (i = 0; i < 3; i++)
    {
        flash_id[i] = md_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    FLASH_CS_SET();

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /*Manufacturer ID flash_id [0] and device ID flash_id [1]*/
}

/**
  * @brief spi function
  * @retval None.
  */
void es_init_extern_flash(void)
{
    md_spi_struct_init(&s_spi);
    s_spi.SPIx      = ES_EXTERN_FLASH_SPI_PERIPHERAL;                 /* Using SPI0 */
    s_spi.mode      = MD_SPI_MODE_MASTER;   /* SPI host mode */
    s_spi.baud      = MD_SPI_BAUD_4;      /* clock */
    s_spi.data_size = MD_SPI_DATA_SIZE_8;   /* 8 bit pattern */
    s_spi.polarity  = MD_SPI_CPOL_HIGH;     /* Free high level */
    s_spi.phase     = MD_SPI_CPHA_SECOND;   /* The second edge receiving data */
    s_spi.first_bit = MD_SPI_FIRSTBIT_MSB;  /* Send the MSB first */
    s_spi.dir       = MD_SPI_DIRECTION_2LINES;
    s_spi.ss_en     = DISABLE;
    s_spi.crc_calc  = DISABLE;
    s_spi.crc_poly  = 0;
    md_spi_init(&s_spi);                    /* According to the parameter initialization SPI peripherals */
}

/**
  * @brief  flash write enable function.
  * @retval None.
  */
static uint8_t flash_write_enable(void)
{
    /* wait until flash unbusy */
    flash_wait_unbusy();

    /* enable the selected Flash */
    FLASH_CS_CLR();

    /* send write enable command */
    if (md_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != MD_OK)
    {
        FLASH_CS_SET();
        return 1;
    }

    /* release Flash */
    FLASH_CS_SET();

    /* wait until flash unbusy */
    flash_wait_unbusy();

    return 0;
}

uint8_t ll_flash_chip_erase(void)
{
    uint8_t  cmd_buf;

    /*Flash sector erase command*/
    cmd_buf = FLASH_ERASE_CHIP;

    /* enable flash write */
    flash_write_enable();

    /* erase current sector */
    /* enable the selected Flash */
    FLASH_CS_CLR();

    /*Send the sector erase instructions and Flash address three bytes*/

    if (md_spi_send_byte_fast(&s_spi, cmd_buf) != MD_OK)
    {
        FLASH_CS_SET();
        return 1;
    }

    /* release Flash */
    FLASH_CS_SET();

    /* wait until flash not in write progress */
    flash_wait_unbusy();

    return 0;
}

/**
  * @brief  low-level flash sector erase function.
  * @param  start_sec: Specified start sector to be erase.
  * @param  end_sec: Specified end sector to be erase.
  * @retval Status.
  */

uint8_t ll_flash_erase_1sector_no_wait(uint32_t sec)
{
    uint8_t  cmd_buf[4];
    uint8_t  i = 0U;
    uint32_t sec_addr;

    /*Flash sector erase command*/
    cmd_buf[0] = FLASH_SEC_ERASE;

    /* group command and address */
    sec_addr = sec * SEC_SIZE;
    cmd_buf[1] = (sec_addr >> 16) & 0xFF;   /*24 bit Flash address*/
    cmd_buf[2] = (sec_addr >> 8) & 0xFF;
    cmd_buf[3] = (sec_addr) & 0xFF;

    /* enable flash write */
    flash_write_enable();

    /* erase current sector */
    /* enable the selected Flash */
    FLASH_CS_CLR();

    /*Send the sector erase instructions and Flash address three bytes*/
    for (i = 0; i < sizeof(cmd_buf); i++)
    {
        if (md_spi_send_byte_fast(&s_spi, cmd_buf[i]) != MD_OK)
        {
            FLASH_CS_SET();
            return 1;
        }
    }

    /* release Flash */
    FLASH_CS_SET();

    return 0;

}
uint8_t ll_flash_sector_erase(uint32_t start_sec, uint32_t end_sec)
{
    uint8_t  cmd_buf[4];
    uint8_t  i = 0U;
    uint32_t sec_addr;

    /*Flash sector erase command*/
    cmd_buf[0] = FLASH_SEC_ERASE;

    /*Erase Flash sectors */
    for (; start_sec <= end_sec; start_sec++)
    {
        /* group command and address */
        sec_addr = start_sec * SEC_SIZE;
        cmd_buf[1] = (sec_addr >> 16) & 0xFF;   /*24 bit Flash address*/
        cmd_buf[2] = (sec_addr >> 8) & 0xFF;
        cmd_buf[3] = (sec_addr) & 0xFF;

        /* enable flash write */
        flash_write_enable();

        /* erase current sector */
        /* enable the selected Flash */
        FLASH_CS_CLR();

        /*Send the sector erase instructions and Flash address three bytes*/
        for (i = 0; i < sizeof(cmd_buf); i++)
        {
            if (md_spi_send_byte_fast(&s_spi, cmd_buf[i]) != MD_OK)
            {
                FLASH_CS_SET();
                return 1;
            }
        }

        /* release Flash */
        FLASH_CS_SET();

        /* wait until flash not in write progress */
        flash_wait_unbusy();
    }

    return 0;
}

uint8_t ll_flash_write(const uint8_t *buf, uint32_t addr, uint32_t size)
{
    uint8_t  cmd_buf[4];
    uint32_t  i, len, w_addr = addr;
    uint32_t remain_size = size;

    if (buf == NULL)
        return 1;

    /*Flash page program command*/
    cmd_buf[0] = FLASH_PAGE_PROGRAM;

    while (remain_size)
    {
        cmd_buf[1] = (w_addr >> 16) & 0xff;
        cmd_buf[2] = (w_addr >> 8) & 0xff;
        cmd_buf[3] = (w_addr) & 0xff;

        len = PAGE_SIZE - (w_addr % PAGE_SIZE);
        len = (len >= remain_size) ? (remain_size) : (len);

        /* enable flash write */
        flash_write_enable();

        /* write current page */
        /* enable the selected Flash */
        FLASH_CS_CLR();

        /*Send the sector erase instructions and Flash address three bytes*/
        for (i = 0; i < sizeof(cmd_buf); i++)
        {
            if (md_spi_send_byte_fast(&s_spi, cmd_buf[i]) != MD_OK)
            {
                FLASH_CS_SET();
                return 1;
            }
        }

        /* write the data to the Flash*/
        for (i = 0; i < len; i++)
        {
            if (md_spi_send_byte_fast(&s_spi, buf[i - remain_size + size]) != MD_OK)
            {
                FLASH_CS_SET();
                return 1;
            }
        }

        /* release Flash */
        FLASH_CS_SET();

        /* wait until flash not in write progress */
        flash_wait_unbusy();

        remain_size -= len;
        w_addr += PAGE_SIZE;
    }

    return 0;
}


/**
  * @brief  low-level flash sector write operation function.
  * @param  src_buf: pointer to the expected written data.
  * @param  sec: start sector to write.
  * @param  sec_num: amount of sector.
  * @retval Status, see @ref ald_status_t.
  */
uint8_t ll_flash_sector_write(const uint8_t *src_buf, uint32_t sec, uint32_t sec_num)
{
    uint8_t  cmd_buf[4];
    uint32_t  i, j, k;
    uint32_t page_addr, sec_addr;

    if (src_buf == NULL)
        return 1;

    /*Flash page program command*/
    cmd_buf[0] = FLASH_PAGE_PROGRAM;

    /*Erase Flash sectors */
    for (i = 0; i < sec_num; i++)
    {
        /* calculate current sector start address */
        sec_addr = sec * SEC_SIZE;

        for (j = 0; j < PAGE_PER_SEC; j++)
        {
            /* group command and address */
            page_addr = sec_addr + PAGE_SIZE * j;
            cmd_buf[1] = (page_addr >> 16) & 0xff;
            cmd_buf[2] = (page_addr >> 8) & 0xff;
            cmd_buf[3] = (page_addr) & 0xff;

            /* enable flash write */
            flash_write_enable();

            /* write current page */
            /* enable the selected Flash */
            FLASH_CS_CLR();

            /*Send the sector erase instructions and Flash address three bytes*/
            for (k = 0; k < sizeof(cmd_buf); k++)
            {
                if (md_spi_send_byte_fast(&s_spi, cmd_buf[k]) != MD_OK)
                {
                    FLASH_CS_SET();
                    return 1;
                }
            }

            /* write the data to the Flash*/
            for (k = 0; k < PAGE_SIZE; k++)
            {
                if (md_spi_send_byte_fast(&s_spi, src_buf[i * SEC_SIZE + j * PAGE_SIZE + k]) != MD_OK)
                {
                    FLASH_CS_SET();
                    return 1;
                }
            }

            /* release Flash */
            FLASH_CS_SET();

            /* wait until flash not in write progress */
            flash_wait_unbusy();
        }
    }

    return 0;
}

void spi_flash_init(void)
{
    md_gpio_init_t gpio_init;

    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP_DOWN;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_STRONG;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;

    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;

    gpio_init.func  = ES_EXTERN_FLASH_SPI_SCK_PIN_MD_FUNC;
    md_gpio_init(ES_EXTERN_FLASH_SPI_SCK_PIN_PORT, ES_EXTERN_FLASH_SPI_SCK_PIN_MD_PIN, &gpio_init);

    gpio_init.func  = ES_EXTERN_FLASH_SPI_MOSI_PIN_MD_FUNC;
    md_gpio_init(ES_EXTERN_FLASH_SPI_MOSI_PIN_PORT, ES_EXTERN_FLASH_SPI_MOSI_PIN_MD_PIN, &gpio_init);

    gpio_init.func  = ES_EXTERN_FLASH_CS_PIN_MD_FUNC;
    md_gpio_init(ES_EXTERN_FLASH_CS_PIN_PORT, ES_EXTERN_FLASH_CS_PIN_MD_PIN, &gpio_init);

    gpio_init.mode  = MD_GPIO_MODE_INPUT;

    gpio_init.func  = ES_EXTERN_FLASH_SPI_MISO_PIN_MD_FUNC;
    md_gpio_init(ES_EXTERN_FLASH_SPI_MISO_PIN_PORT, ES_EXTERN_FLASH_SPI_MISO_PIN_MD_PIN, &gpio_init);

    FLASH_CS_SET();

    es_init_extern_flash();
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
