/**
  *********************************************************************************
  * @file   boot_frame.c
  * @brief  boot frames module.
  *
  * @version V1.0
  * @date    28 Feb 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *          28 Feb 2023     AE Team         Port to ES32VF2264
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include <stdio.h>
#include "boot_frame.h"
#include "boot_flash.h"
#include "boot_shell.h"
#include "md_conf.h"
#include "uartstdio.h"
#include "spi_flash.h"
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */


/** @addtogroup Bootloader
  * @{
  */
/** @defgroup Frame Frame
  * @brief    Bootloader Frame Module
  * @{
  */
/** @defgroup Frame_Public_Variables Public Variable
  * @brief    Frame Public Variable
  * @{
  */
/* Public Variables ---------------------------------------------------------- */

boot_frame_env_t g_frame_env;
CRC_TypeDef *g_h_crc;

#if USE_UART_RECV_BY_DMA
    uint32_t g_recv_index = 0U;
    uint32_t g_handle_index = 0U;
    uint8_t g_handle_state = 0U;
    uint8_t g_all_data[1029 * 8];
#endif/**/

/**
  * @}
  */

/** @defgroup Frame_Private_Functions Private Functions
  * @brief    Frame Private Functions
  * @{
  */
/**
  * @brief  Send command to tramsmitor
  * @param  cmd: Command.
  * @retval 0-success, other value indicates failed.
  */
/* Private Function ---------------------------------------------------------- */

static int xmod_cmd_send(xmod_tx_cmd_t cmd)
{
    uint8_t _cmd = cmd;

    if ((uart_send_str((char *)&_cmd, 1, 100)))
    {
        g_frame_env.state = UART_SEND_ERROR;
        return -1;
    }

    return 0;
}

/**
  * @brief  Check frame index.
  * @param  idx: frame index.
  * @retval 0-success, other value indicates failed.
  */
static int check_idx(uint8_t idx)
{
    uint8_t _idx  = 0U;
    uint8_t next_idx = 0U;

    _idx = g_frame_env.idx;
    next_idx = _idx == 0xFF ? 0 : _idx + 1;

    if ((next_idx != idx) && (idx != _idx))
        return 1;

    return 0;
}

#if USE_UART_RECV_BY_DMA

void spi_flash_no_wait_handle(void)
{
    uint8_t *handle_data_p;

    if (g_handle_state == 0)
    {
        if ((g_handle_index & 0x3) == 0x0)
        {
            ll_flash_erase_1sector_no_wait(g_handle_index >> 2);
            g_handle_state = 1;
        }
        else
        {
            g_handle_state = 2;
        }
    }

    if (g_handle_state == 1)
    {
        if (flash_get_state_unbusy() == 0)
        {
            g_handle_state = 2;
        }
    }

    if (g_handle_state == 2)
    {
        handle_data_p = &g_all_data[1029 * (g_handle_index & 0x7)];
        ll_flash_write(&handle_data_p[2], g_handle_index * 1024, 1024);

        g_handle_index++;
        g_handle_state = 0;
    }

}

#endif/**/

#ifdef XMODE_1K
/**
  * @brief  Receive data by 1K_XMODE frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_1K_recv(void)
{
    uint16_t crc, _crc;
    uint8_t *recv_data_p;

    if ((uart_recv_str((char *)&g_frame_env.soh, 1, TIME_OUT)))
    {
        g_frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if (g_frame_env.soh == XMODE_EOT)
    {
        g_frame_env.state = FRAME_SUCCESS;
        return 1;
    }

    if (g_frame_env.soh != XMODE_STX)
    {
        g_frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

#if USE_UART_RECV_BY_DMA

    recv_data_p = &g_all_data[1029 * (g_recv_index & 0x7)];

    md_dma_set_dest_addr(MD_DMA_CH_0, recv_data_p);
    md_dma_enable_channel(MD_DMA_CH_0);

    md_uart_enable_rxdma(BOOT_UARTX);

    uint32_t tick = md_get_tick() + TIME_OUT;

    while (1)
    {
        if (g_download_flag)
        {
            if (g_handle_index != g_recv_index)
            {
                do
                {
                    spi_flash_no_wait_handle();
                }
                while (g_recv_index - g_handle_index > 5);
            }
        }

        if ((READ_REG(DMA->CHANNEL[MD_DMA_CH_0].CON) & DMA_CON_CHEN_MSK) == 0)
        {
            break;
        }

        if (md_get_tick() >= tick)
        {
            md_dma_disable_channel(MD_DMA_CH_0);
            md_uart_disable_rxdma(BOOT_UARTX);
            g_frame_env.state = FRAME_RECEIVE_TIMEOUT;
            return -1;
        }
    }

    md_uart_disable_rxdma(BOOT_UARTX);

#else

    recv_data_p = g_frame_env.data;

    if ((uart_recv_str((char *)recv_data_p, 1028, TIME_OUT)))
    {
        g_frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

#endif/**/

    if ((recv_data_p[0] ^ recv_data_p[1]) != 0xFF)
    {
        g_frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    if ((check_idx(recv_data_p[0])) != 0)
    {
        g_frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    MD_CRC_RESET();

    for (int i = 0; i < 1024; i++)
    {
        md_crc_write_data(g_h_crc, recv_data_p[2 + i]);
    }

    _crc = md_crc_get_check_result(g_h_crc);
    crc = (recv_data_p[1026] << 8) | recv_data_p[1027];

    if (crc != _crc)
    {
        g_frame_env.state = FRAME_VERIFY_ERROR;
        return -3;
    }

    if (write_data_to_flash(g_frame_env.idx, &recv_data_p[2], crc))
    {
        g_frame_env.state = FLASH_WRITE_ERROR;
        return -4;
    }

#if USE_UART_RECV_BY_DMA

    g_recv_index++;

#endif/**/

    ++g_frame_env.idx;
    return 0;
}
#endif /* XMODE_1K */

/**
  * @brief  Receive a frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_frame_recv(void)
{
    int result = 0;

#ifdef XMODE_1K
    result = xmod_1K_recv();
#endif /* XMODE_1K */
    return result;
}

/**
  * @}
  */

/** @defgroup Frame_Public_Functions Public Functions
  * @brief    Frame Public Functions
  * @{
  */

/**
  * @brief  Show update info
  * @retval none.
  */
void boot_update_info(void)
{
    if (g_frame_env.state == FRAME_SUCCESS)
    {
        printf_e("Update program sucess !\r\n");
    }
    else if (g_frame_env.state == FRAME_INDEX_ERROR)
    {
        printf_e("Update program error: FRAME_INDEX_ERROR !!!\r\n");
    }
    else if (g_frame_env.state == FRAME_RECEIVE_TIMEOUT)
    {
        printf_e("Update program error: FRAME_RECEIVE_TIMEOUT !!!\r\n");
    }
    else if (g_frame_env.state == FRAME_VERIFY_ERROR)
    {
        printf_e("Update program error: FRAME_VERIFY_ERROR !!!\r\n");
    }
    else if (g_frame_env.state == UART_SEND_ERROR)
    {
        printf_e("Update program error: UART_SEND_ERROR !!!\r\n");
    }
    else if (g_frame_env.state == FLASH_WRITE_ERROR)
    {
        printf_e("Update program error: FLASH_WRITE_ERROR !!!\r\n");
    }
    else if (g_frame_env.state == APP_VERIFY_ERROR)
    {
        printf_e("Update program error: APP_VERIFY_ERROR !!!\r\n");
    }
    else
    {
    }

    return;
}

void xmodem_crc_init()
{
    md_crc_init_t crc_init;
    /* Clear crc_handle_t structure */
    md_crc_init_struct(&crc_init);

    /* Initialize CRC */
    crc_init.mode = MD_CRC_MODE_CCITT;
    crc_init.len = MD_CRC_DATASIZE_8;
    crc_init.order = MD_CRC_BYTORD_LOW;
    crc_init.seed = 0;
    crc_init.chs_inv = DISABLE;
    crc_init.chs_rev = DISABLE;
    crc_init.data_inv = DISABLE;
    crc_init.data_rev = DISABLE;
    md_crc_init(&crc_init);
}
/**
  * @brief  Initialize frame module
  * @param  hperh: Pointer to uart_handle_t structure.
  * @retval None
  */
void boot_frame_init(UART_TypeDef *hperh)
{
    if (hperh == NULL)
    {
        return;
    }

    g_h_crc = CRC;
    memset(&g_frame_env, 0, sizeof(boot_frame_env_t));
    g_frame_env.h_uart = hperh;

    return;
}

/**
  * @brief  Check app flash and copy info data
  * @retval 0 if success, others if error
  */
int check_app()
{
    /* Check app */
    uint32_t page = 0U;
    uint8_t *data;
    uint32_t dlen = 0U;
    uint32_t dcrc;
    uint32_t *info_data;

#ifdef XMODE_1K
    page = g_frame_env.idx - 1;
#endif /* XMODE_1K */

    data = (uint8_t *)USR_FLASH_ADDER(page);

    if (!VALID_APP_ADDR((uint32_t)(data + 1024 - 4)))
    {
        return -1; /*User app page invalid (receive error)*/
    }

    info_data = (uint32_t *)(data + INFO_ADDR_IN_PAGE);
    dcrc = info_data[0];
    dlen = info_data[1];

    if (!VALID_APP_ADDR(APP_FLASH_S + dlen - 1))
    {
        return -3; /*User app length invalid */
    }

    if (page != APP_LAST_PAGE)
    {
        /* We have to move app_info to the last page of app_flash */
        md_msc_code_program_word(APP_FLASH_INFO_S, ~APP_FLASH_INFO_S, dcrc);
        md_msc_code_program_word(APP_FLASH_INFO_S + 4, ~(APP_FLASH_INFO_S + 4), dlen);
    }

    printf_e("App length=%u\r\n", dlen);
#ifdef __USE_CRC32
    return check_app_crc();
#else
    return 0;
#endif /* __USE_CRC32 */
}
/**
  * @brief  download bootloader program.
  * @retval 0-success, other value indicates failed.
  */
int  boot_image_update(void)
{
    int ret;

    xmodem_crc_init();
    g_frame_env.idx   = 0;
    g_frame_env.cnt   = 0;

#if USE_UART_RECV_BY_DMA

    g_recv_index = 0;
    g_handle_index = 0;
    g_handle_state = 0;

#endif/**/

    g_frame_env.state = FRAME_SUCCESS;
    xmod_cmd_send(XMOD_TX_CMD_START);

    while (1)
    {
        ret = xmod_frame_recv();

        if ((ret < 0) && (++g_frame_env.cnt < 10) && g_frame_env.state == FRAME_RECEIVE_TIMEOUT)
        {
            if (g_frame_env.idx == 0)
                xmod_cmd_send(XMOD_TX_CMD_START);
            else
                xmod_cmd_send(XMOD_TX_CMD_NCK);

            continue;
        }

        if (ret < 0)
        {
            xmod_cmd_send(XMOD_TX_CMD_TERM);
            return ret;
        }

        if (ret == 1)
        {
            xmod_cmd_send(XMOD_TX_CMD_ACK);
            break;
        }

        xmod_cmd_send(XMOD_TX_CMD_ACK);
    }

    if ((g_frame_env.state == FRAME_SUCCESS) && (g_download_flag == 0))
    {
        if (check_app())
        {
            g_frame_env.state = APP_VERIFY_ERROR;
            return -5;
        }
    }

#if USE_UART_RECV_BY_DMA

    if (g_download_flag)
    {
        while (g_handle_index != g_recv_index)
        {
            spi_flash_no_wait_handle();
        }
    }

#endif/**/

    return 0;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
