/**
  ********************************************************************************
  *
  * @file    boot_shell.h
  * @brief   Head file of boot shell module.
  *
  * @version V1.0
  * @date    28 Feb 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *          28 Feb 2023     AE Team         Port to ES32VF2264
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#ifndef __BOOT_SHELL_H__
#define __BOOT_SHELL_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Includes ------------------------------------------------------------------ */

#include "md_uart.h"
#include "md_timer.h"
#include "boot_flash.h"
#include "boot_frame.h"
#include "list.h"

/** @addtogroup Bootloader
  * @{
  */

/** @addtogroup Shell
  * @{
  */

/* Exported Macros ----------------------------------------------------------- */

/* Exported Types ------------------------------------------------------------ */

/** @defgroup Shell_Private_Types Private Types
  * @brief    Shell Private Types
  * @{
  */
/**
  * @brief command execute result
  */
typedef enum
{
    CMD_SUCCESS = 0,    /**< Success */
    CMD_UNKONW  = 1,    /**< Unkonw */
    CMD_ARG_ERR = 2,    /**< Error */
    CMD_NULL    = 3,    /**< NULL */
} shell_cmd_exec_t;

/**
  * @brief command element structure
  */

typedef struct boot_cmd_ent_s
{
    list_head_t link;   /**< Link for list */

    char cmd[16];       /**< Name of this command */
    void (*fun_cmd)(void);  /**< Function of this command */
    uint32_t nr_arg;    /**< Number of this paramters */
} boot_cmd_ent_t;

typedef void (*cfun_arg0_t)(void);
typedef void (*cfun_arg1_t)(uint32_t arg);
typedef void (*cfun_arg2_t)(uint32_t arg0, uint32_t arg1);
typedef void (*cfun_arg3_t)(uint32_t arg0, uint32_t arg1, uint32_t arg2);
typedef void (*cfun_arg4_t)(uint32_t arg0, uint32_t arg1, uint32_t arg2, uint32_t arg3);

/**
  * @brief Shell environment structure
  */
typedef struct boot_shell_env_s
{
    list_head_t list;   /**< Link for list */

    uint16_t nr_cmd;    /**< Number of the command */
    uint8_t idx_cache;  /**< Index of the cache */
    uint8_t cursor;     /**< Cursor in the cmd */
    char fn_key[4];     /**< Handler UP and Down keys */
    char cmd[64];       /**< Command */
    char cmd_cache0[64];    /**< First cache */
    char cmd_cache1[64];    /**< Second cache */
    char cmd_cache2[64];    /**< Third cache */

    uint32_t check : 1; /**< Check the password */
    uint32_t       : 31; /**< Reserve */

    char password[16];  /**< Pointer to password */

    UART_TypeDef *h_uart;  /**< Pointer to uart_handle_t structure */
    char out_buf[64];   /**< buffer for output */
} boot_shell_env_t;
/**
  * @}
  */

/**
  * @addtogroup Shell_Public_Variables
  * @{
  */

/* Exported Variables -------------------------------------------------------- */

extern UART_TypeDef *g_h_uart;

/* Exported Constants -------------------------------------------------------- */

/**
  * @}
  */

/** @addtogroup Shell_Public_Functions
  * @{
  */

/* Exported Functions -------------------------------------------------------- */

extern uint8_t shell_rx_flag_get(void);
extern int32_t shell_cmd_insert(char *cmd, void (*)(void), uint32_t nr_arg);
extern void shell_init(UART_TypeDef *hperh);
extern void shell_task_func(void *arg);
extern void printf_e(const char *fmt, ...);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
extern "C"
}
#endif /* __cplusplus */
#endif /* __BOOT_SHELL_H__ */
