/**
  *********************************************************************************
  *
  * @file    .c
  * @brief  Source file
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "spi_flash_diskio.h"
#include "string.h"
#include "md_msc.h"

/* Private Macros ------------------------------------------------------------ */
#define ES_FATFS_BASE_PATH_MAX_LEN (32)

/* Private Variables ---------------------------------------------------------*/
static FATFS s_es_fatfs;        /* File system object for flash */
static uint8_t s_es_fatfs_path[ES_FATFS_BASE_PATH_MAX_LEN];
static uint8_t s_es_fatfs_new_app_path[ES_FATFS_BASE_PATH_MAX_LEN];

uint32_t g_temp_max_sector_addr;/*jump info sector*/

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes -----------------------------------------------*/
DSTATUS spi_flash_init(BYTE);
DSTATUS spi_flash_stat(BYTE);
DRESULT spi_flash_read(BYTE, BYTE *, DWORD, UINT);
#if  _USE_WRITE
    DRESULT spi_flash_write(BYTE, const BYTE *, DWORD, UINT);
#endif/*_USE_WRITE*/
#if  _USE_IOCTL
    DRESULT spi_flash_ioctl(BYTE, BYTE, void *);
#endif/*_USE_IOCTL*/
/* Public Variables ---------------------------------------------------------- */
const Diskio_drvTypeDef spi_flash_drv =
{
    spi_flash_init,
    spi_flash_stat,
    spi_flash_read,

#if  _USE_WRITE
    spi_flash_write,
#endif/*_USE_WRITE*/

#if  _USE_IOCTL
    spi_flash_ioctl,
#endif/*_USE_IOCTL*/
};

/* Private Functions ---------------------------------------------------------*/
/**
  * @brief  Initializes a Drive
  * @param  NONE
  * @retval DSTATUS: Operation status
  */
DSTATUS spi_flash_init(BYTE lun)
{
    /*uint32_t id;
    ll_spi_flash_init();
    id = flash_read_id();
    printf("\r\nManufacturer ID is %02x & Device ID is %02x %02x\r\n", (uint8_t)(id >> 16), (uint8_t)(id >> 8), (uint8_t)id);*/

    return RES_OK;
}

/**
  * @brief  Gets Disk Status
  * @param  parm_num
  * @param  param
  * @retval DSTATUS: Operation status
  */
DSTATUS spi_flash_stat(BYTE lun)
{
    return RES_OK;
}

/**
  * @brief  Reads Data
  * @param  *buff: Data buffer to store read data
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to read (1..128)
  * @retval DRESULT: Operation result
  */
DRESULT spi_flash_read(BYTE lun, BYTE *buff, DWORD sector, UINT count)
{
    if (sector > g_temp_max_sector_addr)
    {
        g_temp_max_sector_addr = sector;
    }

    if (ll_flash_sector_read(buff, (uint32_t)(sector), count))
    {
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Writes Sector(s)
  * @param  *buff: Data to be written
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to write (1..128)
  * @retval DRESULT: Operation result
  */
#if _USE_WRITE
DRESULT spi_flash_write(BYTE lun, const BYTE *buff, DWORD sector, UINT count)
{
    if (ll_flash_sector_erase(sector, sector + count - 1))
    {
        return RES_ERROR;
    }

    if (ll_flash_sector_write(buff, sector, count))
    {
        return RES_ERROR;
    }

    return RES_OK;
}
#endif/*_USE_WRITE*/

#if  _USE_IOCTL
/**
 * @brief  I/O control operation
 * @param  cmd: Control code
 * @param  *buff: Buffer to send/receive control data
 * @retval DRESULT: Operation result
 */
DRESULT spi_flash_ioctl(BYTE lun, BYTE cmd, void *buff)
{
    DRESULT res = RES_ERROR;

    switch (cmd)
    {
        /* Make sure that no pending write process */
        case CTRL_SYNC:
            res = RES_OK;
            break;

        /* Get number of sectors on the disk (DWORD) */
        case GET_SECTOR_COUNT :
            *(DWORD *)buff = DEV_SIZE * BLK_SIZE;
            res = RES_OK;
            break;

        /* Get R/W sector size (WORD) */
        case GET_SECTOR_SIZE:
            *(WORD *)buff = SEC_SIZE;
            res = RES_OK;
            break;

        /* Get erase block size in unit of sector (DWORD) */
        case GET_BLOCK_SIZE:
            *(DWORD *)buff = BLK_SIZE;
            res = RES_OK;
            break;

        /* Informs the data on the block of sectors can be erased. */
        case CTRL_TRIM:
            if (! ll_flash_sector_erase(*(DWORD *)buff, *((DWORD *)buff + 1)))
                res = RES_OK;

            break;

        default:
            res = RES_PARERR;
            break;
    }

    return res;
}

#endif/*_USE_IOCTL*/

void es_fatfs_register(void)
{
    /* Regeister the disk I/O driver */
    if (FATFS_LinkDriverEx(&spi_flash_drv, (char *)s_es_fatfs_path, 0) == 0)
    {
        /* Register the file system object to the FatFs module */
        if (f_mount(&s_es_fatfs, (const TCHAR *)s_es_fatfs_path, 1) != FR_OK)
        {
            /* Create FAT volume on the logical drive 0 */
#if 1

            if (f_mkfs((const TCHAR *)s_es_fatfs_path, FM_ANY, 0, s_es_fatfs.win, FF_MAX_SS))
            {
#else
            static uint8_t work[FF_MAX_SS];

            if (f_mkfs((const TCHAR *)s_es_fatfs_path, FM_ANY, 0, work, FF_MAX_SS))
            {
#endif/*init fat*/
            }
            else
            {

                /*¸´Î»*/
                MD_SYSCFG_UNLOCK();
                md_rmu_enable_chip_reset();
                MD_SYSCFG_LOCK();
            }

            /*nUnlink the spi Flash disk I/O driver */
            FATFS_UnLinkDriverEx((char *)s_es_fatfs_path, 0);
        }
        else
        {
        }
    }
}

void es_try_fatfs_unregister(void)
{
    if (f_mount((void *)0, (const TCHAR *)s_es_fatfs_path, 0) != FR_OK)
    {
        /* FatFs Initialization Error */
        return;
    }

    /*nUnlink the spi Flash disk I/O driver */
    FATFS_UnLinkDriverEx((char *)s_es_fatfs_path, 0);
}

__attribute__((aligned(4))) uint8_t g_temp_data[512];


#define ES_APP_ADDR  0x8000

extern void uart_out_string(char *buf);

uint32_t read_dir_find_newapp(char *input_fullpath)
{
    char *fullpath = input_fullpath;
    DIR dir;
    FILINFO fno;
    FRESULT res;
    FIL fp[1];
    uint32_t len, index;
    uint8_t temp_flag;
    uint32_t i;
    uint8_t temp8;
    unsigned int get_num;

    res = f_opendir(&dir, fullpath);

    if (res == FR_OK)
    {
//        printf("dir: %s\r\n", fullpath);

        while (1)
        {
            res = f_readdir(&dir, &fno);

            if ((res != FR_OK) || (fno.fname[0] == '\0'))
                break;

//            printf("%-32s ", fno.fname);

            if ((fno.fattrib) & (AM_DIR))
            {
//                printf("<DIR>\r\n");
            }
            else
            {
//                printf("%d\r\n", (unsigned int)(fno.fsize));

//              printf("[temp][%d]\r\n",temp_flag);
//
                temp_flag = 0;

                if ((memcmp(fno.fname, "es_app.bin", strlen("es_app.bin"))) == 0)
                {
                    temp_flag = 1;
                }

                if ((memcmp(fno.fname, "es_gui_data.bin", strlen("es_gui_data.bin"))) == 0)
                {
                    temp_flag = 2;
                }

//              if((memcmp(fno.fname , "es_app208k+gui_data.bin",strlen("es_gui_data.bin"))) == 0)
//              {
//                  temp_flag = 3;
//              }

                if (temp_flag)
                {
                    len = strlen(fullpath);

                    if (len < ES_FATFS_BASE_PATH_MAX_LEN - 1)
                    {
                        memcpy(s_es_fatfs_new_app_path, fullpath, len);
                        index = len;
                        len = strlen(fno.fname);

                        if (len + index < ES_FATFS_BASE_PATH_MAX_LEN - 1)
                        {
                            memcpy(s_es_fatfs_new_app_path + index, fno.fname, len);
                            s_es_fatfs_new_app_path[len + index] = '\0';

//                          printf("%s\r\n",s_es_fatfs_new_app_path);

                        }
                    }

                    f_open(fp, (const char *)s_es_fatfs_new_app_path, FA_READ);
                    f_lseek(fp, 0);

                    g_temp_max_sector_addr = 0;
                    res = f_read(fp, &temp8, 1, &get_num);

                    if (res == FR_OK)
                    {
                        if (g_temp_max_sector_addr)
                        {
                            g_temp_max_sector_addr = g_temp_max_sector_addr * 4096;

                            if (temp_flag == 1)
                            {
//                              printf("es_app.bin handle\r\n");
                                uart_out_string("handle es_app.bin\r\n");

                                ll_flash_read_start(g_temp_max_sector_addr);

                                for (i = 0; i < fno.fsize; i += 512)
                                {
                                    ll_flash_read_data8_len(g_temp_data, 512);

                                    if (memcmp(g_temp_data, (uint8_t *)(ES_APP_ADDR + i), 512))
                                    {
                                        md_msc_code_erase_page((ES_APP_ADDR + i), ~(ES_APP_ADDR + i), 0);
                                        md_msc_code_program_words((ES_APP_ADDR + i), ~(ES_APP_ADDR + i), g_temp_data, 512, 0);
                                    }
                                }

                                ll_flash_read_end();
                            }

                            if (temp_flag == 2)
                            {
//                              printf("es_gui_data.bin handle\r\n");
                                uart_out_string("handle es_gui_data.bin\r\n");

                                uint32_t buf32[4];
                                buf32[0] = 0xAEAE55AA;
                                buf32[1] = g_temp_max_sector_addr;
                                buf32[2] = ~g_temp_max_sector_addr;
                                buf32[3] = 0xEAEAAA55;

                                ll_flash_sector_erase(2047, 2047);
                                ll_flash_sector_write((uint8_t *)buf32, 2047, 1);
                            }
                        }
                    }

                    f_close(fp);
                    f_unlink((const char *)s_es_fatfs_new_app_path);
                }
            }
        }

        res = f_closedir(&dir);
    }

    return 0;
}

void es_boot_fatfs_register(void)
{
    /* Regeister the disk I/O driver */
    read_dir_find_newapp((char *)s_es_fatfs_path);
}

#include "ald_usb.h"

void jump2app(void)
{
    __disable_irq();

    ald_usb_int_unregister();

    MD_SYSCFG_UNLOCK();
    md_syscfg_set_cpu_boot_addr(ES_APP_ADDR);
    MD_SYSCFG_LOCK();

    __enable_irq();

    md_rmu_reset_system();
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
