/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    17 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          17 Jan. 2023    Shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
 /* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "printf.h"
/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
TaskHandle_t g_task_app_handle = NULL;
TaskHandle_t g_task_1_handle = NULL;
TaskHandle_t g_task_2_handle = NULL;
SemaphoreHandle_t g_sema_test;
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */
void uart_pin_init(void);
void uart_init(void);
static void task_app(void *);
static void task_1(void *pvParameters);
static void task_2(void *pvParameters);

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 72000000);
	
    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();
	
	md_mcu_irq_config(CUART1_IRQn, 0, ENABLE);
	md_mcu_irq_config(MACHINE_MODE_SOFT_IRQn, 5, ENABLE);
	CLIC->CLICINT[MACHINE_MODE_SOFT_IRQn].ATTR |= (3); 
	__enable_irq();
	
    uart_pin_init();
    uart_init();
	
	/* Create AppTaskCreate
    * Arguments: TaskProc TaskName StackDepth Param Priority PtrTaskHandle */
    xTaskCreate(task_app, "AppTaskCreate", 128, NULL, 1, &g_task_app_handle);

    /* Start task schedule*/
	vTaskStartScheduler();
	
	/* Should nerver run to here */
    while (1);
}

/**
  * @brief  App task creator
  * @param  parameter: Parameter passed when created
  * @retval None
  */
static void task_app(void *param)
{
    BaseType_t retval = pdPASS;

    taskENTER_CRITICAL();

    /* Create Task1 */
    retval = xTaskCreate(task_1, "Task_1", 128, NULL, 2, &g_task_1_handle);

    if (pdPASS == retval)
        printf("Task 1 Created\r\n");

    /* Create Task2 */
    retval = xTaskCreate(task_2, "Task_2", 128, NULL, 3, &g_task_2_handle);

    if (pdPASS == retval)
        printf("Task 2 Created\r\n");

    /* Delete self */
    vTaskDelete(g_task_app_handle);

    taskEXIT_CRITICAL();
}


/**
  * @brief  LED1 task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_1(void *param)
{

    while (1)
    {
		/* Try to obtain the semaphore */
        if (xSemaphoreTake(g_sema_test, 1000) == pdTRUE)
        {
            printf("Task1 get Semaphore\r\n");
			vTaskDelay(2000);

            /* Release the semaphore */
            xSemaphoreGive(g_sema_test);

            /* Suspend ourseleves to execute thread 2 (lower priority)  */
            vTaskSuspend(NULL);
        }
        else
        {
            printf("Task1 failed to get Semaphore\r\n");
        }
    }
}

/**
  * @brief  LED2 task
  * @param  param Parameter passed when created
  * @retval None
  */
static void task_2(void *param)
{

    while (1)
    {
		/* Try to obtain the semaphore */
        if (xSemaphoreTake(g_sema_test, 1000) == pdTRUE)
        {
            printf("Task2 get Semaphore\r\n");
            /* Resume Thread 1 (higher priority)*/
            vTaskResume(g_task_1_handle);

			vTaskDelay(2000);

            /* Release the semaphore to unblock Thread 1 (higher priority)  */
            xSemaphoreGive(g_sema_test);
        }
        else
        {
            printf("Task2 failed to get Semaphore\r\n");
        };
    }
}

/**
  * @brief  Init UART pin
  * @retval None
  */
void uart_pin_init(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize tx pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_2;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    /* Initialize rx pin */
    gpio_init.mode  = MD_GPIO_MODE_INPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_2;
    md_gpio_init(UART_RX_PORT, UART_RX_PIN, &gpio_init);
}
/**
  * @brief:  Initialize the uart.
  * @param:  None
  * @retval: None
  */
void uart_init(void)
{
    md_uart_init_t uart_init;
    md_uart_init_struct(&uart_init);

    /* Initialize UART */
    uart_init.baud        = 115200;  /* 冗余度: 110155 ~ 121452 */
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_init.mode        = MD_UART_MODE;
    md_uart_init(UARTX, &uart_init);
}

int _write(int file, const void *ptr, int len)
{
    uint16_t cnt; 
	(void)file;
	uint8_t *ptr_char = (uint8_t *)ptr;
	
	for(int i = 0 ; i < len ; i++)
	{
		if (*ptr_char == '\n')
		{
            cnt = 4000;
			while ((READ_BIT(UARTX->STAT, UART_STAT_TFEMPTY_MSK)) != UART_STAT_TFEMPTY_MSK
			        && (--cnt))
			{
			}
			md_uart_set_send_data8(UARTX, '\r');
		}

        cnt = 4000;
        while ((READ_BIT(UARTX->STAT, UART_STAT_TFEMPTY_MSK)) != UART_STAT_TFEMPTY_MSK
		        && (--cnt))
		{
		}
		md_uart_set_send_data8(UARTX, *ptr_char++);
	}

	return len;
}
void _putchar(char chr)
{
	while (!md_uart_is_active_flag_tfempty(UARTX));
	md_uart_set_send_data8(UARTX, (uint8_t)chr);
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
