/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    9 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          9 Jan. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

static md_spi_init_t s_spi;

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

static void spi_pin_init(void);

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  flash write function
  * @param  write_buf: Pointer to write data buffer
  * @param  write_size: size
  * @retval Status.
  */
md_status_t spi_flash_write(const uint8_t *write_buf,  size_t write_size)
{
    uint16_t i;

    for (i = 0; i < write_size; i++)
    {
        if (md_spi_send_byte_fast(&s_spi, write_buf[i]) != MD_OK)
        {
            FLASH_CS_SET();     /*Pick up and release of Flash*/
            return MD_ERROR;
        }
    }

    return MD_OK;
}

/**
  * @brief  flash read function
  * @param  read_buf: Pointer to read data buffer
  * @param  read_size: size
  * @retval Status.
  */
md_status_t spi_flash_read(uint8_t *read_buf, size_t read_size)
{
    uint16_t i;
    int r_flag = 0;

    for (i = 0; i < read_size; i++)
    {

        read_buf[i] = md_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != MD_OK)
        {
            FLASH_CS_SET();
            return MD_ERROR;
        }
    }

    return MD_OK;
}

/**
  * @brief spi function
  * @retval None.
  */
void init_mcu_spi(void)
{
    spi_pin_init();

    md_spi_struct_init(&s_spi);
    s_spi.SPIx      = SPI0;                 /* Using SPI0 */
    s_spi.mode      = MD_SPI_MODE_MASTER;   /* SPI host mode */
    s_spi.baud      = MD_SPI_BAUD_128;      /* clock */
    s_spi.data_size = MD_SPI_DATA_SIZE_8;   /* 8 bit pattern */
    s_spi.polarity  = MD_SPI_CPOL_HIGH;     /* Free high level */
    s_spi.phase     = MD_SPI_CPHA_SECOND;   /* The second edge receiving data */
    s_spi.first_bit = MD_SPI_FIRSTBIT_MSB;  /* Send the MSB first */
    s_spi.dir       = MD_SPI_DIRECTION_2LINES;
    s_spi.ss_en     = DISABLE;
    s_spi.crc_calc  = DISABLE;
    s_spi.crc_poly  = 0;
    md_spi_init(&s_spi);                    /* According to the parameter initialization SPI peripherals */
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void spi_pin_init(void)
{
    md_gpio_init_t l_gpio;

    l_gpio.type  = MD_GPIO_TYPE_CMOS;
    l_gpio.odos  = MD_GPIO_PUSH_PULL;
    l_gpio.pupd  = MD_GPIO_PUSH_UP;
    l_gpio.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    l_gpio.flt   = MD_GPIO_FILTER_DISABLE;

    l_gpio.func  = MD_GPIO_FUNC_1;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_4, &l_gpio);    /* Initialize PA4 for selected pin */
    FLASH_CS_SET();   /* Choose the output high, release of Flash */

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_5, &l_gpio);    /* Initialize PA5 for clock output pin */

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_7, &l_gpio);    /* Initialize PA7 for MOSI pin */

    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_INPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_6, &l_gpio);    /* Initialization for PA6 MISO pins */
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
