/**********************************************************************************
 *
 * @file    .c
 * @brief   Source file
 *
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date               Author          Notes
 *          2023-01-17         liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "usbd_core.h"
#include "usbd_cdc.h"
#include "usbd_hid.h"

/* Private Macros ------------------------------------------------------------ */
#define USB_CDC1_UART  CUART0
#define USB_CDC2_UART  CUART2
#define USB_CDC3_UART  EUART0
#define USB_CDC4_UART  EUART1

#define UART_RX_BUF_SIZE 256
#define ES_TST_UART_DMA 1

/* Private Variables --------------------------------------------------------- */
uint8_t g_uart_loop_rxbuf[1024];
const uint8_t *g_uart_loop_rxbuf_addr[4] =
{
    &g_uart_loop_rxbuf[0],
    &g_uart_loop_rxbuf[256],
    &g_uart_loop_rxbuf[256 * 2],
    &g_uart_loop_rxbuf[256 * 3],
};
const UART_TypeDef *g_uartx[4] =
{
    USB_CDC1_UART,
    USB_CDC2_UART,
    USB_CDC3_UART,
    USB_CDC4_UART,
};

extern uint8_t g_usb_ep_rxbuf[256];
extern uint32_t g_usb_packet_len_flag;
extern uint8_t *g_usb_packet_len;
extern volatile struct cdc_line_coding g_cdc_line_code[4];

uint8_t g_uart_loop_rxbuf_out_index[4] = { 0U };
uint32_t g_usb_packet_len_flag = 0U;
uint8_t *g_usb_packet_len = (uint8_t *)(&g_usb_packet_len_flag);

const uint8_t *g_txbuf_cuart0 = &g_usb_ep_rxbuf[0];
const uint8_t *g_txbuf_cuart2 = &g_usb_ep_rxbuf[64];
const uint8_t *g_txbuf_euart0 = &g_usb_ep_rxbuf[64 * 2];
const uint8_t *g_txbuf_euart1 = &g_usb_ep_rxbuf[64 * 3];
uint8_t g_tx_len_cuart0;
uint8_t g_tx_len_cuart2;
uint8_t g_tx_len_euart0;
uint8_t g_tx_len_euart1;
uint8_t g_tx_index_cuart0;
uint8_t g_tx_index_cuart2;
uint8_t g_tx_index_euart0;
uint8_t g_tx_index_euart1;

uint8_t g_rx_index_uartx[4] = {0U};

/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

/*******************************************************************************/
/* Line Coding Structure                                                       */
/*-----------------------------------------------------------------------------*/
/* Offset | Field       | Size | Value  | Description                          */
/* 0      | dwDTERate   |   4  | Number |Data terminal rate, in bits per second*/
/* 4      | bCharFormat |   1  | Number | Stop bits                            */
/*                                        0 - 1 Stop bit                       */
/*                                        1 - 1.5 Stop bits                    */
/*                                        2 - 2 Stop bits                      */
/* 5      | bParityType |  1   | Number | Parity                               */
/*                                        0 - None                             */
/*                                        1 - Odd                              */
/*                                        2 - Even                             */
/*                                        3 - Mark                             */
/*                                        4 - Space                            */
/* 6      | bDataBits  |   1   | Number Data bits (5, 6, 7, 8 or 16).          */
/*******************************************************************************/

void cdc_uart_config(uint8_t uart_index)
{
    md_uart_init_t  uart_handle;
    md_uart_init_struct(&uart_handle);/*rate = 115200,data = 8,stop = 1,Parity = None*/
    uart_handle.baud        = g_cdc_line_code[uart_index].dwDTERate;

    switch (g_cdc_line_code[uart_index].bDataBits)
    {
        case 7:
            uart_handle.word_length = MD_UART_WORD_LENGTH_7B;
            break;

        case 6:
            uart_handle.word_length = MD_UART_WORD_LENGTH_6B;
            break;

        case 5:
            uart_handle.word_length = MD_UART_WORD_LENGTH_5B;
            break;

        default:
            break;
    }

    if (g_cdc_line_code[uart_index].bCharFormat == 2)
    {
        uart_handle.stop_bits   = MD_UART_STOP_BITS_2;
    }

    switch (g_cdc_line_code[uart_index].bParityType)
    {
        case 1:
            uart_handle.parity = MD_UART_PARITY_ODD;
            break;

        case 2:
            uart_handle.parity = MD_UART_PARITY_EVEN;
            break;

        default:
            break;
    }

    md_uart_init((UART_TypeDef *)g_uartx[uart_index], &uart_handle);

#if ES_TST_UART_DMA
    md_uart_enable_rxdma((UART_TypeDef *)g_uartx[uart_index]);

    if ((md_uart_is_enabled_txdma(USB_CDC3_UART)) == 0)
    {
        md_uart_enable_txdma(USB_CDC3_UART);
    }

    if ((md_uart_is_enabled_txdma(USB_CDC4_UART)) == 0)
    {
        md_uart_enable_txdma(USB_CDC4_UART);
    }

#else
    md_uart_enable_it_rfnempty((UART_TypeDef *)g_uartx[uart_index]);
#endif/**/
}

void uart_init(void)
{
    md_gpio_init_t  gpio_config;
    md_uart_init_t  uart_handle;

    md_gpio_init_struct(&gpio_config);
    /* Initialize tx pin */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.odrv  = MD_GPIO_OUT_DRIVE_STRONG;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;

    /*CUART0_TX PA9*/
    gpio_config.func  = MD_GPIO_FUNC_2;
    md_gpio_init(GPIOA, MD_GPIO_PIN_9, &gpio_config);

    /*CUART2_TX PC10*/
    gpio_config.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIOC, MD_GPIO_PIN_10, &gpio_config);

    /*EUART0_TX PB5*/
    gpio_config.func  = MD_GPIO_FUNC_5;
    md_gpio_init(GPIOB, MD_GPIO_PIN_5, &gpio_config);

    /*EUART1_TX PC14*/
    gpio_config.func  = MD_GPIO_FUNC_2;
    md_gpio_init(GPIOC, MD_GPIO_PIN_14, &gpio_config);

    gpio_config.mode  = MD_GPIO_MODE_INPUT;

    /*CUART0_RX PA10*/
    gpio_config.func  = MD_GPIO_FUNC_2;
    md_gpio_init(GPIOA, MD_GPIO_PIN_10, &gpio_config);

    /*CUART2_RX PC11*/
    gpio_config.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIOC, MD_GPIO_PIN_11, &gpio_config);

    /*EUART0_RX PB4*/
    gpio_config.func  = MD_GPIO_FUNC_5;
    md_gpio_init(GPIOB, MD_GPIO_PIN_4, &gpio_config);

    /*EUART1_RX PA5*/
    gpio_config.func  = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOA, MD_GPIO_PIN_5, &gpio_config);

    /* Initialize UART */
    md_uart_init_struct(&uart_handle);
    uart_handle.baud        = 115200;
    uart_handle.word_length = MD_UART_WORD_LENGTH_8B;
    uart_handle.stop_bits   = MD_UART_STOP_BITS_1;
    uart_handle.parity      = MD_UART_PARITY_NONE;
    uart_handle.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_handle.mode        = MD_UART_MODE;
    md_uart_init(USB_CDC1_UART, &uart_handle);
    md_uart_init(USB_CDC2_UART, &uart_handle);
    md_uart_init(USB_CDC3_UART, &uart_handle);
    md_uart_init(USB_CDC4_UART, &uart_handle);

    md_mcu_irq_config(CUART0_IRQn, 2, ENABLE);
    md_mcu_irq_config(CUART2_IRQn, 2, ENABLE);
    md_mcu_irq_config(EUART0_IRQn, 2, ENABLE);
    md_mcu_irq_config(EUART1_IRQn, 2, ENABLE);

#if ES_TST_UART_DMA
#else
    md_uart_enable_it_rfnempty(USB_CDC1_UART);
    md_uart_enable_it_rfnempty(USB_CDC2_UART);
    md_uart_enable_it_rfnempty(USB_CDC3_UART);
    md_uart_enable_it_rfnempty(USB_CDC4_UART);
#endif/**/
}

#if ES_TST_UART_DMA
void cdc_uart_dma_init(void)
{
    md_dma_config_t dma_config;

    memset(&dma_config, 0x0, sizeof(md_dma_config_t));

    dma_config.src_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.dst_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.src_inc        = DISABLE;
    dma_config.dst_inc        = ENABLE;
    dma_config.R_power        = MD_DMA_R_POWER_1;
    dma_config.priority       = MD_DMA_LOW_PRIORITY;
    dma_config.mem_to_mem     = DISABLE;
    dma_config.circle_mode    = ENABLE;
    dma_config.msigsel        = MD_DMA_MSIGSEL_UART_RNR;
    dma_config.size           = UART_RX_BUF_SIZE;

    dma_config.msel           = MD_DMA_MSEL_CUART0;
    dma_config.src            = (void *)&USB_CDC1_UART->RXBUF;
    dma_config.dst            = (void *)g_uart_loop_rxbuf_addr[0];
    md_dma_init(MD_DMA_CH_0, &dma_config);

    dma_config.msel           = MD_DMA_MSEL_CUART2;
    dma_config.src            = (void *)&USB_CDC2_UART->RXBUF;
    dma_config.dst            = (void *)g_uart_loop_rxbuf_addr[1];
    md_dma_init(MD_DMA_CH_1, &dma_config);

    dma_config.msel           = MD_DMA_MSEL_EUART0;
    dma_config.src            = (void *)&USB_CDC3_UART->RXBUF;
    dma_config.dst            = (void *)g_uart_loop_rxbuf_addr[2];
    md_dma_init(MD_DMA_CH_2, &dma_config);

    dma_config.msel           = MD_DMA_MSEL_EUART1;
    dma_config.src            = (void *)&USB_CDC4_UART->RXBUF;
    dma_config.dst            = (void *)g_uart_loop_rxbuf_addr[3];
    md_dma_init(MD_DMA_CH_3, &dma_config);

    dma_config.msigsel        = MD_DMA_MSIGSEL_UART_TXEMPTY;
    dma_config.circle_mode    = DISABLE;
    dma_config.src_inc        = ENABLE;
    dma_config.dst_inc        = DISABLE;

    dma_config.msel           = MD_DMA_MSEL_EUART0;
    dma_config.src            = (void *)&g_usb_ep_rxbuf[64 * 2];
    dma_config.dst            = (void *)&USB_CDC3_UART->TXBUF;
    md_dma_init(MD_DMA_CH_4, &dma_config);

    dma_config.msel           = MD_DMA_MSEL_EUART1;
    dma_config.src            = (void *)&g_usb_ep_rxbuf[64 * 3];
    dma_config.dst            = (void *)&USB_CDC4_UART->TXBUF;
    md_dma_init(MD_DMA_CH_5, &dma_config);

    md_dma_enable_channel(MD_DMA_CH_0);
    md_dma_enable_channel(MD_DMA_CH_1);
    md_dma_enable_channel(MD_DMA_CH_2);
    md_dma_enable_channel(MD_DMA_CH_3);

    md_uart_enable_rxdma(USB_CDC1_UART);
    md_uart_enable_rxdma(USB_CDC2_UART);
    md_uart_enable_rxdma(USB_CDC3_UART);
    md_uart_enable_rxdma(USB_CDC4_UART);

    md_uart_enable_txdma(USB_CDC3_UART);
    md_uart_enable_txdma(USB_CDC4_UART);
}
#endif/**/

void cdc_uart_init(void)
{
    uart_init();
#if ES_TST_UART_DMA
    cdc_uart_dma_init();
#endif/**/
}

uint16_t cdc_uartx_data_get(uint8_t uart_index, uint32_t *data_addr)
{
    uint8_t temp_dma_rx_num;
    uint8_t len;
    uint8_t last_out_index;

    if (uart_index < 4)
    {
#if ES_TST_UART_DMA
        temp_dma_rx_num = 256 - ((READ_REG(DMA->CHANNEL[uart_index].NDT)) >> 16);
#else
        temp_dma_rx_num = g_rx_index_uartx[uart_index];
#endif/**/

        last_out_index = g_uart_loop_rxbuf_out_index[uart_index];

        if (last_out_index != temp_dma_rx_num)
        {
            *data_addr = ((uint32_t)(g_uart_loop_rxbuf_addr[uart_index])) + last_out_index;

            if (last_out_index < temp_dma_rx_num)
            {
                len = temp_dma_rx_num - last_out_index;
            }
            else
            {
                len = 256 - last_out_index;
            }

            len = (len > 32) ? 32 : len;
            return len;
        }
    }

    return 0;
}

void cdc_uartx_data_update(uint8_t uart_index, uint16_t tx_len)
{
    if (uart_index < 4)
    {
        g_uart_loop_rxbuf_out_index[uart_index] += tx_len;
    }
}

#pragma GCC push_options
#pragma GCC optimize("O0" )

void cdc_uart0_irq_handle(void)
{
    if (md_uart_mask_it_tfempty(USB_CDC1_UART))
    {
        if (g_tx_index_cuart0 < g_tx_len_cuart0)
        {
            md_uart_set_send_data8(USB_CDC1_UART, g_txbuf_cuart0[g_tx_index_cuart0]);
            g_tx_index_cuart0++;
        }
        else
        {
            md_uart_disable_it_tfempty(USB_CDC1_UART);
        }

        md_uart_clear_it_tfempty(USB_CDC1_UART);
    }

#if ES_TST_UART_DMA
#else

    if (md_uart_mask_it_rfnempty(USB_CDC1_UART))
    {
        *((uint8_t *)(((uint32_t)(g_uart_loop_rxbuf_addr[0])) + g_rx_index_uartx[0])) = md_uart_recv_data8(USB_CDC1_UART);
        g_rx_index_uartx[0]++;

        md_uart_clear_flag_rfnempty(USB_CDC1_UART);
    }

#endif/**/
}

void cdc_uart1_irq_handle(void)
{
    if (md_uart_mask_it_tfempty(USB_CDC2_UART))
    {
        if (g_tx_index_cuart2 < g_tx_len_cuart2)
        {
            md_uart_set_send_data8(USB_CDC2_UART, g_txbuf_cuart2[g_tx_index_cuart2]);
            g_tx_index_cuart2++;
        }
        else
        {
            md_uart_disable_it_tfempty(USB_CDC2_UART);
        }

        md_uart_clear_it_tfempty(USB_CDC2_UART);
    }

#if ES_TST_UART_DMA
#else

    if (md_uart_mask_it_rfnempty(USB_CDC2_UART))
    {
        *((uint8_t *)(((uint32_t)(g_uart_loop_rxbuf_addr[1])) + g_rx_index_uartx[1])) = md_uart_recv_data8(USB_CDC2_UART);
        g_rx_index_uartx[1]++;

        md_uart_clear_flag_rfnempty(USB_CDC2_UART);
    }

#endif/**/
}

#pragma GCC pop_options

void cdc_uart2_irq_handle(void)
{
#if ES_TST_UART_DMA
#else

    if (md_uart_mask_it_tfempty(USB_CDC3_UART))
    {
        if (g_tx_index_euart0 < g_tx_len_euart0)
        {
            md_uart_set_send_data8(USB_CDC3_UART, g_txbuf_euart0[g_tx_index_euart0]);
            g_tx_index_euart0++;
        }
        else
        {
            md_uart_disable_it_tfempty(USB_CDC3_UART);
        }

        md_uart_clear_it_tfempty(USB_CDC3_UART);
    }

    if (md_uart_mask_it_rfnempty(USB_CDC3_UART))
    {
        *((uint8_t *)(((uint32_t)(g_uart_loop_rxbuf_addr[2])) + g_rx_index_uartx[2])) = md_uart_recv_data8(USB_CDC3_UART);
        g_rx_index_uartx[2]++;

        md_uart_clear_flag_rfnempty(USB_CDC3_UART);
    }

#endif/**/
}

void cdc_uart3_irq_handle(void)
{
#if ES_TST_UART_DMA
#else

    if (md_uart_mask_it_tfempty(USB_CDC4_UART))
    {
        if (g_tx_index_euart1 < g_tx_len_euart1)
        {
            md_uart_set_send_data8(USB_CDC4_UART, g_txbuf_euart1[g_tx_index_euart1]);
            g_tx_index_euart1++;
        }
        else
        {
            md_uart_disable_it_tfempty(USB_CDC4_UART);
        }

        md_uart_clear_it_tfempty(USB_CDC4_UART);
    }

    if (md_uart_mask_it_rfnempty(USB_CDC4_UART))
    {
        *((uint8_t *)(((uint32_t)(g_uart_loop_rxbuf_addr[3])) + g_rx_index_uartx[3])) = md_uart_recv_data8(USB_CDC4_UART);
        g_rx_index_uartx[3]++;

        md_uart_clear_flag_rfnempty(USB_CDC4_UART);
    }

#endif/**/
}

void uart_send_over(uint8_t index)
{
    if (g_usb_packet_len[index])
    {
        g_usb_packet_len[index] = 0;
#if ES_TEST_USE_IRQ_TRANSFER
        es_usbd_out_ep_recv(index, 0);
#endif/**/
    }
}

void cdc_uartx_send_over_check(void)
{
    if (g_usb_packet_len_flag)
    {
        if ((md_uart_is_enable_it_tfempty(USB_CDC1_UART) == 0))
        {
            uart_send_over(0);
        }

        if ((md_uart_is_enable_it_tfempty(USB_CDC2_UART) == 0))
        {
            uart_send_over(1);
        }

#if ES_TST_UART_DMA

        if ((md_dma_is_enabled(MD_DMA_CH_4) == 0))
#else
        if ((md_uart_is_enable_it_tfempty(USB_CDC3_UART) == 0))
#endif/**/
        {
            uart_send_over(2);
        }

#if ES_TST_UART_DMA

        if ((md_dma_is_enabled(MD_DMA_CH_5) == 0))
#else
        if ((md_uart_is_enable_it_tfempty(USB_CDC4_UART) == 0))
#endif/**/
        {
            uart_send_over(3);
        }
    }
}

void cdc_uartx_send_start(uint8_t uart_index, uint16_t tx_len)
{
    if (uart_index == 0)
    {
        g_tx_len_cuart0 = g_usb_packet_len[0];
        g_tx_index_cuart0 = 0;
        md_uart_enable_it_tfempty(USB_CDC1_UART);
    }

    if (uart_index == 1)
    {
        g_tx_len_cuart2 = g_usb_packet_len[1];
        g_tx_index_cuart2 = 0;
        md_uart_enable_it_tfempty(USB_CDC2_UART);
    }

    if (uart_index == 2)
    {
#if ES_TST_UART_DMA
        md_dma_set_transfer_size(MD_DMA_CH_4, tx_len);
        md_dma_enable_channel(MD_DMA_CH_4);
#else
        g_tx_len_euart0 = g_usb_packet_len[2];
        g_tx_index_euart0 = 0;
        md_uart_enable_it_tfempty(USB_CDC3_UART);
#endif/**/
    }

    if (uart_index == 3)
    {
#if ES_TST_UART_DMA
        md_dma_set_transfer_size(MD_DMA_CH_5, tx_len);
        md_dma_enable_channel(MD_DMA_CH_5);
#else
        g_tx_len_euart1 = g_usb_packet_len[3];
        g_tx_index_euart1 = 0;
        md_uart_enable_it_tfempty(USB_CDC4_UART);
#endif/**/
    }
}

/*******************************************************END OF FILE*************/
