/**********************************************************************************
 *
 * @file    .c
 * @brief   Source file
 *
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date               Author          Notes
 *          2023-01-17         liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes------------------------------------------------------------------- */
#include "usbd_core.h"
#include "usbd_audio.h"
#include "main.h"
#include "wm_i2s.h"
/* Private Macros ------------------------------------------------------------ */
#define TEST_KEY    *(volatile uint32_t*)0x40080100
#define HRC4M       *(volatile uint32_t*)0x40083C10
#define HRC4M_CAL   *(volatile uint32_t*)0x40848

#define USBD_VID           0x30cc
#define USBD_PID           0xae06
#define USBD_MAX_POWER     100
#define USBD_LANGID_STRING 1033

#ifdef CONFIG_USB_HS
    #define EP_INTERVAL 0x04
#else
    #define EP_INTERVAL 0x01
#endif /*CONFIG_USB_HS*/

#define AUDIO_IN_EP  0x81
#define AUDIO_OUT_EP 0x02

#define AUDIO_IN_FU_ID  0x02
#define AUDIO_OUT_FU_ID 0x05

/* AUDIO Class Config */
#define AUDIO_SPEAKER_FREQ            16000U
#define AUDIO_SPEAKER_FRAME_SIZE_BYTE 2U
#define AUDIO_SPEAKER_RESOLUTION_BIT  16U
#define AUDIO_MIC_FREQ                16000U
#define AUDIO_MIC_FRAME_SIZE_BYTE     2U
#define AUDIO_MIC_RESOLUTION_BIT      16U

#define AUDIO_SAMPLE_FREQ(frq) (uint8_t)(frq), (uint8_t)((frq >> 8)), (uint8_t)((frq >> 16))

/* AudioFreq * DataSize (2 bytes) * NumChannels (Stereo: 2) */
#define AUDIO_OUT_PACKET ((uint32_t)((AUDIO_SPEAKER_FREQ * AUDIO_SPEAKER_FRAME_SIZE_BYTE * 2) / 1000))
/* 16bit(2 Bytes) 2 channel(Mono:2) */
#define AUDIO_IN_PACKET ((uint32_t)((AUDIO_MIC_FREQ * AUDIO_MIC_FRAME_SIZE_BYTE * 2) / 1000))

#define ES_USB_AUDIO_STREAM_BUF_SIZE (8192)
#define ES_USB_AUDIO_PLAY_BUF_SIZE (AUDIO_OUT_PACKET)

#define USB_AUDIO_CONFIG_DESC_SIZ (unsigned long)(9 +                                       \
        AUDIO_AC_DESCRIPTOR_INIT_LEN(2) +         \
        AUDIO_SIZEOF_AC_INPUT_TERMINAL_DESC +     \
        AUDIO_SIZEOF_AC_FEATURE_UNIT_DESC(2, 1) + \
        AUDIO_SIZEOF_AC_OUTPUT_TERMINAL_DESC +    \
        AUDIO_SIZEOF_AC_INPUT_TERMINAL_DESC +     \
        AUDIO_SIZEOF_AC_FEATURE_UNIT_DESC(2, 1) + \
        AUDIO_SIZEOF_AC_OUTPUT_TERMINAL_DESC +    \
        AUDIO_AS_DESCRIPTOR_INIT_LEN(1) +         \
        AUDIO_AS_DESCRIPTOR_INIT_LEN(1))

#define AUDIO_AC_SIZ (AUDIO_SIZEOF_AC_HEADER_DESC(2) +          \
                      AUDIO_SIZEOF_AC_INPUT_TERMINAL_DESC +     \
                      AUDIO_SIZEOF_AC_FEATURE_UNIT_DESC(2, 1) + \
                      AUDIO_SIZEOF_AC_OUTPUT_TERMINAL_DESC +    \
                      AUDIO_SIZEOF_AC_INPUT_TERMINAL_DESC +     \
                      AUDIO_SIZEOF_AC_FEATURE_UNIT_DESC(2, 1) + \
                      AUDIO_SIZEOF_AC_OUTPUT_TERMINAL_DESC)
/* Private Variables --------------------------------------------------------- */
static uint16_t s_hrc4m_trim = 0U;
static uint8_t s_es_audio_out_vol = 0U;
static uint8_t s_es_audio_in_vol = 0U;
static uint8_t s_es_usbd_configure_done_flag = 1U;
static struct usbd_interface s_audio_intf1, s_audio_intf2, s_audio_intf3;
static uint32_t s_audio_record_buf[AUDIO_IN_PACKET / 2];
static uint32_t s_audio_play_buf[ES_USB_AUDIO_PLAY_BUF_SIZE / 2];
static uint32_t s_audio_play_stream_buf[(ES_USB_AUDIO_STREAM_BUF_SIZE / 4) + (AUDIO_OUT_PACKET / 4)];
static uint32_t s_audio_play_stream_buf_write_index = 0U;
static uint32_t s_audio_play_stream_buf_read_index = 0U;
static uint32_t s_usb_audio_get_data_tick = 0U;
static uint16_t s_usb_get_packages_times = 0U;
static uint16_t s_audio_play_data_times = 0U;
static const uint8_t audio_descriptor[] =
{
    USB_DEVICE_DESCRIPTOR_INIT(USB_2_0, 0xef, 0x02, 0x01, USBD_VID, USBD_PID, 0x0001, 0x01),
    USB_CONFIG_DESCRIPTOR_INIT(USB_AUDIO_CONFIG_DESC_SIZ, 0x03, 0x01, USB_CONFIG_BUS_POWERED, USBD_MAX_POWER),
    AUDIO_AC_DESCRIPTOR_INIT(0x00, 0x03, AUDIO_AC_SIZ, 0x00, 0x01, 0x02),
    AUDIO_AC_INPUT_TERMINAL_DESCRIPTOR_INIT(0x01, AUDIO_INTERM_MIC, 0x02, 0x0003),
    AUDIO_AC_FEATURE_UNIT_DESCRIPTOR_INIT(0x02, 0x01, 0x01, 0x03, 0x00, 0x00),
    AUDIO_AC_OUTPUT_TERMINAL_DESCRIPTOR_INIT(0x03, AUDIO_TERMINAL_STREAMING, 0x02),
    AUDIO_AC_INPUT_TERMINAL_DESCRIPTOR_INIT(0x04, AUDIO_TERMINAL_STREAMING, 0x02, 0x0003),
    AUDIO_AC_FEATURE_UNIT_DESCRIPTOR_INIT(0x05, 0x04, 0x01, 0x03, 0x00, 0x00),
    AUDIO_AC_OUTPUT_TERMINAL_DESCRIPTOR_INIT(0x06, AUDIO_OUTTERM_SPEAKER, 0x05),
    AUDIO_AS_DESCRIPTOR_INIT(0x01, 0x04, 0x02, AUDIO_SPEAKER_FRAME_SIZE_BYTE, AUDIO_SPEAKER_RESOLUTION_BIT, AUDIO_OUT_EP, 0x09, AUDIO_OUT_PACKET,
    EP_INTERVAL, AUDIO_SAMPLE_FREQ_3B(AUDIO_SPEAKER_FREQ)),
    AUDIO_AS_DESCRIPTOR_INIT(0x02, 0x03, 0x02, AUDIO_MIC_FRAME_SIZE_BYTE, AUDIO_MIC_RESOLUTION_BIT, AUDIO_IN_EP, 0x05, AUDIO_IN_PACKET,
    EP_INTERVAL, AUDIO_SAMPLE_FREQ_3B(AUDIO_MIC_FREQ)),
    /**************************************/
    /*string0 descriptor*/
    /**************************************/
    USB_LANGID_INIT(USBD_LANGID_STRING),
    /**************************************/
    /*string1 descriptor*/
    /**************************************/
    0xE,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    'e', 0x00,                  /* wcChar0 */
    's', 0x00,                  /* wcChar1 */
    's', 0x00,                  /* wcChar2 */
    'e', 0x00,                  /* wcChar3 */
    'm', 0x00,                  /* wcChar4 */
    'i', 0x00,                  /* wcChar5 */
    /**************************************/
    /* string2 descriptor*/
    /**************************************/
    0x20,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    'e', 0x00,                  /* wcChar0 */
    's', 0x00,                  /* wcChar1 */
    's', 0x00,                  /* wcChar2 */
    'e', 0x00,                  /* wcChar3 */
    'm', 0x00,                  /* wcChar4 */
    'i', 0x00,                  /* wcChar5 */
    ' ', 0x00,                  /* wcChar9 */
    'U', 0x00,                  /* wcChar10 */
    'A', 0x00,                  /* wcChar11 */
    'C', 0x00,                  /* wcChar12 */
    ' ', 0x00,                  /* wcChar13 */
    'D', 0x00,                  /* wcChar14 */
    'E', 0x00,                  /* wcChar15 */
    'M', 0x00,                  /* wcChar16 */
    'O', 0x00,                  /* wcChar17 */
    /**************************************/
    /* string3 descriptor*/
    /**************************************/
    0x16,                       /* bLength */
    USB_DESCRIPTOR_TYPE_STRING, /* bDescriptorType */
    '2', 0x00,                  /* wcChar0 */
    '0', 0x00,                  /* wcChar1 */
    '2', 0x00,                  /* wcChar2 */
    '2', 0x00,                  /* wcChar3 */
    '1', 0x00,                  /* wcChar4 */
    '2', 0x00,                  /* wcChar5 */
    '1', 0x00,                  /* wcChar6 */
    '2', 0x00,                  /* wcChar7 */
    '0', 0x00,                  /* wcChar8 */
    '1', 0x00,                  /* wcChar9 */
#ifdef CONFIG_USB_HS
    /**************************************/
    /* device qualifier descriptor*/
    /**************************************/
    0x0a,
    USB_DESCRIPTOR_TYPE_DEVICE_QUALIFIER,
    0x00,
    0x02,
    0xEF,
    0x02,
    0x01,
    0x40,
    0x01,
    0x00,
#endif /*CONFIG_USB_HS*/
    0x00
};

/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
static void audio_data_recv(uint8_t busid, uint8_t ep, uint32_t nbytes);
/* Private Function ---------------------------------------------------------- */

static struct usbd_endpoint s_audio_in_ep =
{
    .ep_cb = NULL,
    .ep_addr = AUDIO_IN_EP
};

static struct usbd_endpoint s_audio_out_ep =
{
    .ep_cb = audio_data_recv,
    .ep_addr = AUDIO_OUT_EP
};

static void audio_data_recv(uint8_t busid, uint8_t ep, uint32_t nbytes)
{
    int ret;

    s_usb_get_packages_times++;

    if (nbytes > 0)
    {
        s_audio_play_stream_buf_write_index += nbytes;
        s_usb_audio_get_data_tick = md_get_tick();
    }
    else
        printf("[err]usbd_ep_start_read AUDIO_OUT_EP nbytes <= 0\r\n");

    ret = usbd_ep_start_read(0, AUDIO_OUT_EP, (uint8_t *)(&(s_audio_play_stream_buf[(s_audio_play_stream_buf_write_index % (ES_USB_AUDIO_STREAM_BUF_SIZE)) / 4])), AUDIO_OUT_PACKET);

    if (ret != 0)
        printf("[err]usbd_ep_start_read AUDIO_OUT_EP fail\r\n");
}

void hrc4m_clk_change(uint16_t t)
{
    HRC4M = 0x55AA0000 + (HRC4M_CAL & 0xFC00) + (t & 0x3FF);

    if (t > 0x3FF)
        printf("[err]hrc4m clk change err\r\n");
}

uint16_t audio_play_data(void *buf, uint16_t length)
{
    uint32_t i, j;
    uint32_t *buf32 = buf;
    int buf_fill_size;
    static uint8_t fill_buf_use_flag = 0U;
    int speed_compare_value;

    s_audio_play_data_times += ES_USB_AUDIO_PLAY_BUF_SIZE / AUDIO_OUT_PACKET;

    buf_fill_size = s_audio_play_stream_buf_write_index - s_audio_play_stream_buf_read_index;

    if (fill_buf_use_flag)
    {
        /*If receiving USB audio data is normal, start calibrating the HRC clock*/
        if ((md_get_tick()) - s_usb_audio_get_data_tick < 2)
        {
            speed_compare_value = s_audio_play_data_times - s_usb_get_packages_times;

            if (speed_compare_value > 2)
            {
                hrc4m_clk_change(--s_hrc4m_trim);
                s_audio_play_data_times = s_usb_get_packages_times;
            }

            if (speed_compare_value < -2)
            {
                hrc4m_clk_change(++s_hrc4m_trim);
                s_audio_play_data_times = s_usb_get_packages_times;
            }
        }
        else
        {
            s_audio_play_data_times = s_usb_get_packages_times;
        }

        /*Fill the data stream into the I2S sending DMA cache*/
        j = (s_audio_play_stream_buf_read_index % ES_USB_AUDIO_STREAM_BUF_SIZE) / 4;

        for (i = 0; i < (ES_USB_AUDIO_PLAY_BUF_SIZE / 4); i++)
            buf32[i] = s_audio_play_stream_buf[j++];

        s_audio_play_stream_buf_read_index += ES_USB_AUDIO_PLAY_BUF_SIZE;

        /*Data flow exception, stop playing*/
        if ((buf_fill_size > (ES_USB_AUDIO_STREAM_BUF_SIZE - ES_USB_AUDIO_PLAY_BUF_SIZE)) || (buf_fill_size < (2 * ES_USB_AUDIO_PLAY_BUF_SIZE)))
        {
            fill_buf_use_flag = 0;
            s_audio_play_stream_buf_read_index = s_audio_play_stream_buf_write_index;
        }

    }
    else
    {

        /*Data flow restored to normal, start playing*/
        if (buf_fill_size > (ES_USB_AUDIO_STREAM_BUF_SIZE / 2))
        {
            s_audio_play_data_times = s_usb_get_packages_times;
            fill_buf_use_flag = 1U;
        }

        /*Play Mute*/
        for (i = 0; i < (ES_USB_AUDIO_PLAY_BUF_SIZE / 4); i++)
            buf32[i] = 0;
    }

    return 0;
}

uint16_t audio_record_data(void *buf, uint16_t length)
{
    usbd_write_packet(AUDIO_IN_EP, buf, AUDIO_IN_PACKET);
    return 0;
}

void usbd_audio_open(uint8_t busid, uint8_t intf)
{
    int ret;

    if (intf == 1)
    {
        wm_set_vol(0x3, s_es_audio_out_vol);

        ret = usbd_ep_start_read(0, AUDIO_OUT_EP, (uint8_t *)(&(s_audio_play_stream_buf[(s_audio_play_stream_buf_write_index % (ES_USB_AUDIO_STREAM_BUF_SIZE)) / 4])), AUDIO_OUT_PACKET);

        if (ret != 0)
            printf("[err]usbd_ep_start_read AUDIO_OUT_EP fail\r\n");
    }
    else
    {
        s_es_audio_in_vol = 0x37;
        wm_set_vol(0x4, s_es_audio_in_vol);
    }

    printf("OPEN %d\r\n", intf);
}

void usbd_audio_close(uint8_t busid, uint8_t intf)
{
    if (intf == 1)
        wm_set_vol(0x3, 0);
    else
        wm_set_vol(0x4, 0);

    printf("CLOSE %d\r\n", intf);
}

void usbd_audio_set_volume(uint8_t busid, uint8_t ep, uint8_t ch, int volume)
{
    /*  0x000000 = -57dB , 0x111001 = 0dB , 0x111111 = +6dB  */
    s_es_audio_out_vol = ((uint8_t)(volume + 57.0)) & 0x3F;
    wm_set_vol(0x3, s_es_audio_out_vol);

    /*  0x000000=-12dB, 0x000001=-11.25dB , 0x010000=0dB , 0x111111=35.25dB  */
    wm_set_vol(0x4, s_es_audio_in_vol);
    printf("set_volume:ep = %d,ch = %d,volume = %d\r\n", ep, ch, volume);
}

void usbd_audio_set_mute(uint8_t busid, uint8_t ep, uint8_t ch, bool mute)
{
    if (mute)
        wm_set_vol(0x7, 0);
    else
    {
        wm_set_vol(0x3, s_es_audio_out_vol);
        wm_set_vol(0x4, s_es_audio_in_vol);
    }

    printf("set_mute:ep = %d,ch = %d,mute = %d\r\n", ep, ch, mute);
}

void usbd_audio_set_sampling_freq(uint8_t busid, uint8_t ep, uint32_t sampling_freq)
{
    printf("set_sampling_freq:ep = %d,sampling_freq = %d\r\n", ep, (unsigned int)sampling_freq);
}

void usbd_audio_get_sampling_freq_table(uint8_t busid, uint8_t ep, uint8_t **sampling_freq_table)
{
    printf("get_sampling_freq_table:ep = %d\r\n", ep);
}

void usbd_event_handler(uint8_t busid, uint8_t event)
{
    switch (event)
    {
        case USBD_EVENT_RESET:
            break;

        case USBD_EVENT_CONNECTED:
            break;

        case USBD_EVENT_DISCONNECTED:
            break;

        case USBD_EVENT_RESUME:
            break;

        case USBD_EVENT_SUSPEND:
            break;

        case USBD_EVENT_CONFIGURED:
        {
            int value;
            s_es_usbd_configure_done_flag = 0;

            MD_SYSCFG_UNLOCK();
            TEST_KEY = 0x5A962814;
            TEST_KEY = 0xE7CB69A5;
            s_hrc4m_trim = HRC4M_CAL & 0x3FF;

            value = (72000000 + (16000 * 256) / 2) / 16 / 256;
            value = (value / 1000) * 1000;
            value = (4000 * value) / (72000000 / 256 / 16);
            value = (value - 4000) / 3;
            s_hrc4m_trim += value;
            hrc4m_clk_change(s_hrc4m_trim);

            break;
        }

        case USBD_EVENT_SET_REMOTE_WAKEUP:
            break;

        case USBD_EVENT_CLR_REMOTE_WAKEUP:
            break;

        default:
            break;
    }
}

/**
  * @brief  Initializate i2c pin
  * @retval None.
  */
static void i2c_pin_init(void)
{
    md_gpio_init_t l_gpio;

    l_gpio.type  = MD_GPIO_TYPE_CMOS;
    l_gpio.odos  = MD_GPIO_OPEN_DRAIN;
    l_gpio.pupd  = MD_GPIO_PUSH_UP;
    l_gpio.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    l_gpio.flt   = MD_GPIO_FILTER_DISABLE;
    l_gpio.func  = MD_GPIO_FUNC_1;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOB, MD_GPIO_PIN_2, &l_gpio);   /*Initialize PB2 for soft I2C SCL pins*/
    md_gpio_init(GPIOB, MD_GPIO_PIN_3, &l_gpio);   /*Initialize PB3 for soft I2C SDA pin*/
}
/**
  * @brief  Initializate i2s pin
  * @retval None.
  */
static void i2s_pin_init(void)
{
    md_gpio_init_t l_gpio;

    l_gpio.type  = MD_GPIO_TYPE_CMOS;
    l_gpio.odos  = MD_GPIO_PUSH_PULL;
    l_gpio.pupd  = MD_GPIO_PUSH_DOWN;
    l_gpio.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    l_gpio.flt   = MD_GPIO_FILTER_DISABLE;
    l_gpio.func  = MD_GPIO_FUNC_2;
    l_gpio.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_4, &l_gpio);   /*Initialize PA4 9  WS  for NSS pins*/
    md_gpio_init(GPIOA, MD_GPIO_PIN_5, &l_gpio);   /*Initialize PA5 10 CK  for SCK pin*/
    md_gpio_init(GPIOA, MD_GPIO_PIN_6, &l_gpio);   /*Initialize PA6 11 SDI for MISO pins*/
    md_gpio_init(GPIOA, MD_GPIO_PIN_7, &l_gpio);   /*Initialize PA7 12 SDO for MOSI pin*/
    l_gpio.func  = MD_GPIO_FUNC_7;
    md_gpio_init(GPIOA, MD_GPIO_PIN_8, &l_gpio);   /*Initialize PA8 5  MCK for MCLK pin*/
}
/**
  * @brief  Initialize WM8978 and I2S module
  * @retval None.
  */
void wm8978_i2s_init(void)
{
    struct wm_i2s_info wm_info;

    i2c_pin_init();

    if (WM8978_Init(I2C0))
    {
        printf("WM8978 I2C init failed\r\n");
        return;
    }

    printf("WM8978 Init ok\r\n");
    i2s_pin_init();
    wm_info.spi_x = (void *)SPI0;
    wm_info.buffers[0] = s_audio_play_buf;
    wm_info.buffers[1] = s_audio_record_buf;
    wm_info.buffer_length = AUDIO_OUT_PACKET;
    wm_info.feed_data = audio_play_data;
    wm_info.recv_data = audio_record_data;
    wm_info.dma_tx_ch = 0;
    wm_info.dma_rx_ch = 1;
    wm_info.sample_rate = AUDIO_SPEAKER_FREQ;
    wm_i2s_init(&wm_info);
    printf("I2S init ok\r\n");
}

struct audio_entity_info audio_entity_table[] =
{
    {
        .bEntityId = AUDIO_IN_FU_ID,
        .bDescriptorSubtype = AUDIO_CONTROL_FEATURE_UNIT,
        .ep = AUDIO_IN_EP
    },
    {
        .bEntityId = AUDIO_OUT_FU_ID,
        .bDescriptorSubtype = AUDIO_CONTROL_FEATURE_UNIT,
        .ep = AUDIO_OUT_EP
    },
};

int es_usbd_audio_init(void)
{
    wm8978_i2s_init();     /*Initialize WM8978 and I2S module*/
    wm_set_mode(WM_SENDRECV);
    wm_set_vol(0xF, 0);
    wm_start();

    md_mcu_irq_config(DMA_IRQn, 2, ENABLE);

    usbd_desc_register(0, audio_descriptor);
    usbd_add_interface(0, usbd_audio_init_intf(0, &s_audio_intf1, 0x0100, audio_entity_table, 2));
    usbd_add_interface(0, usbd_audio_init_intf(0, &s_audio_intf2, 0x0100, audio_entity_table, 2));
    usbd_add_interface(0, usbd_audio_init_intf(0, &s_audio_intf3, 0x0100, audio_entity_table, 2));
    usbd_add_endpoint(0, &s_audio_in_ep);
    usbd_add_endpoint(0, &s_audio_out_ep);

    usbd_initialize(0, 0, usbd_event_handler);

    while (s_es_usbd_configure_done_flag) {}

    return 0;
}

__WEAK void vPortEnterCritical(void)
{

}

__WEAK void vPortExitCritical(void)
{

}

__WEAK void vTaskSuspendAll(void)
{

}

__WEAK void xTaskResumeAll(void)
{

}

