/**
  *********************************************************************************
  *
  * @file    .c
  * @brief  Source file
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "spi_flash_diskio.h"

/* Private Macros ------------------------------------------------------------ */
#define ES_FATFS_BASE_PATH_MAX_LEN (32)

/* Private Variables ---------------------------------------------------------*/
static FATFS s_es_fatfs;        /* File system object for flash */
static uint8_t s_es_fatfs_path[ES_FATFS_BASE_PATH_MAX_LEN];

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes -----------------------------------------------*/
DSTATUS spi_flash_init(BYTE);
DSTATUS spi_flash_stat(BYTE);
DRESULT spi_flash_read(BYTE, BYTE *, DWORD, UINT);
#if  _USE_WRITE
    DRESULT spi_flash_write(BYTE, const BYTE *, DWORD, UINT);
#endif/*_USE_WRITE*/
#if  _USE_IOCTL
    DRESULT spi_flash_ioctl(BYTE, BYTE, void *);
#endif/*_USE_IOCTL*/
/* Public Variables ---------------------------------------------------------- */
const Diskio_drvTypeDef spi_flash_drv =
{
    spi_flash_init,
    spi_flash_stat,
    spi_flash_read,

#if  _USE_WRITE
    spi_flash_write,
#endif/*_USE_WRITE*/

#if  _USE_IOCTL
    spi_flash_ioctl,
#endif/*_USE_IOCTL*/
};

/* Private Functions ---------------------------------------------------------*/
/**
  * @brief  Initializes a Drive
  * @param  NONE
  * @retval DSTATUS: Operation status
  */
DSTATUS spi_flash_init(BYTE lun)
{
    /*uint32_t id;
    ll_spi_flash_init();
    id = flash_read_id();
    printf("\r\nManufacturer ID is %02x & Device ID is %02x %02x\r\n", (uint8_t)(id >> 16), (uint8_t)(id >> 8), (uint8_t)id);*/

    return RES_OK;
}

/**
  * @brief  Gets Disk Status
  * @param  parm_num
  * @param  param
  * @retval DSTATUS: Operation status
  */
DSTATUS spi_flash_stat(BYTE lun)
{
    return RES_OK;
}

/**
  * @brief  Reads Data
  * @param  *buff: Data buffer to store read data
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to read (1..128)
  * @retval DRESULT: Operation result
  */
DRESULT spi_flash_read(BYTE lun, BYTE *buff, DWORD sector, UINT count)
{
    if (ll_flash_sector_read(buff, (uint32_t)(sector), count))
    {
        es_fatfs_error_handler();
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Writes Sector(s)
  * @param  *buff: Data to be written
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to write (1..128)
  * @retval DRESULT: Operation result
  */
#if _USE_WRITE
DRESULT spi_flash_write(BYTE lun, const BYTE *buff, DWORD sector, UINT count)
{
    if (ll_flash_sector_erase(sector, sector + count - 1))
    {
        es_fatfs_error_handler();
        return RES_ERROR;
    }

    if (ll_flash_sector_write(buff, sector, count))
    {
        es_fatfs_error_handler();
        return RES_ERROR;
    }

    return RES_OK;
}
#endif/*_USE_WRITE*/

#if  _USE_IOCTL
/**
 * @brief  I/O control operation
 * @param  cmd: Control code
 * @param  *buff: Buffer to send/receive control data
 * @retval DRESULT: Operation result
 */
DRESULT spi_flash_ioctl(BYTE lun, BYTE cmd, void *buff)
{
    DRESULT res = RES_ERROR;

    switch (cmd)
    {
        /* Make sure that no pending write process */
        case CTRL_SYNC:
            res = RES_OK;
            break;

        /* Get number of sectors on the disk (DWORD) */
        case GET_SECTOR_COUNT :
            *(DWORD *)buff = DEV_SIZE * BLK_SIZE;
            res = RES_OK;
            break;

        /* Get R/W sector size (WORD) */
        case GET_SECTOR_SIZE:
            *(WORD *)buff = SEC_SIZE;
            res = RES_OK;
            break;

        /* Get erase block size in unit of sector (DWORD) */
        case GET_BLOCK_SIZE:
            *(DWORD *)buff = BLK_SIZE;
            res = RES_OK;
            break;

        /* Informs the data on the block of sectors can be erased. */
        case CTRL_TRIM:
            if (! ll_flash_sector_erase(*(DWORD *)buff, *((DWORD *)buff + 1)))
                res = RES_OK;

            break;

        default:
            res = RES_PARERR;
            break;
    }

    return res;
}

#endif/*_USE_IOCTL*/

void es_fatfs_error_handler(void)
{
    printf("[es_fatfs_error]\r\n");
}

void es_fatfs_register(void)
{
    /* Regeister the disk I/O driver */
    if (FATFS_LinkDriverEx(&spi_flash_drv, (char *)s_es_fatfs_path, 0) == 0)
    {
        /* Register the file system object to the FatFs module */
        if (f_mount(&s_es_fatfs, (const TCHAR *)s_es_fatfs_path, 1) != FR_OK)
        {
            /* Create FAT volume on the logical drive 0 */
            printf("not find fatfs,try to mkfs\r\n");
#if 1

            if (f_mkfs((const TCHAR *)s_es_fatfs_path, FM_ANY, 0, s_es_fatfs.win, FF_MAX_SS))
            {
#else
            static uint8_t work[FF_MAX_SS];

            if (f_mkfs((const TCHAR *)s_es_fatfs_path, FM_ANY, 0, work, FF_MAX_SS))
            {
#endif/*init fat*/
                es_fatfs_error_handler();
            }
            else
            {
                printf("mkfs succes.\r\nmcu reset.\r\n");

                /*复位*/
                MD_SYSCFG_UNLOCK();
                md_rmu_enable_chip_reset();
                MD_SYSCFG_LOCK();
            }

            /* FatFs Initialization Error */
            es_fatfs_error_handler();

            /*nUnlink the spi Flash disk I/O driver */
            FATFS_UnLinkDriverEx((char *)s_es_fatfs_path, 0);
        }
        else
        {
            printf("\r\nfatfs mount succes , path :%s\r\n", s_es_fatfs_path);
        }
    }
}

void es_try_fatfs_unregister(void)
{
    if (f_mount((void *)0, (const TCHAR *)s_es_fatfs_path, 0) != FR_OK)
    {
        /* FatFs Initialization Error */
        es_fatfs_error_handler();
    }

    printf("fatfs unmount succes , path :%s", s_es_fatfs_path);

    /*nUnlink the spi Flash disk I/O driver */
    FATFS_UnLinkDriverEx((char *)s_es_fatfs_path, 0);
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
