/**
  **************************************************************************************
  * @file    app_update.c
  * @brief   app_update Program Body.
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
/* Includes-------------------------------------------------------------------*/
#include "app_update.h"
#include "string.h"
#include "md_syscfg.h"
#include "md_rmu.h"
#include "md_cmu.h"
#include "md_pmu.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
uint32_t g_boot_complete = 0U;

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

/** @addtogroup Private_Function
  * @{
  */

/**
  * @brief  Testing the APP legitimacy and jump to the APP
  * @param  None
  * @retval None
  */
uint8_t application_check_and_run(void)
{
    __disable_irq();

    MD_SYSCFG_UNLOCK();
    md_syscfg_set_cpu_boot_addr(APP_FLASH_START_ADDRESS);
    MD_SYSCFG_LOCK();

    __enable_irq();

    md_rmu_reset_system();

    return TRUE;
}

/**
  * @brief  erase app flash
  * @param  None
  * @retval TRUE/FALSE
  */
uint8_t app_flash_erase_area(uint32_t addr, uint32_t size)
{
    uint32_t i;

    for (i = 0; i < size; i += 512)
    {
        if (md_msc_code_erase_page((addr + i), ~(addr + i), 0) == MD_ERROR)
            return FALSE;
    }

    return TRUE;
}
/**
  * @brief  write app flash
  * @param  addr: flash address
  * @param  data: data buff to write
  * @param  data_length: byte length to write
  * @retval TRUE/FALSE
  */
uint8_t app_flash_write(uint32_t addr, const uint8_t *data, uint32_t data_length)
{
    uint8_t buf[64];
    __disable_irq();
    memcpy((void *)buf, (const void *)data, data_length);

    if ((addr >= (APP_FLASH_START_ADDRESS + APP_FLASH_SIZE)) || (addr < APP_FLASH_START_ADDRESS))
        return FALSE;

    if (data_length & 0x07)
    {
        buf[data_length]     = 0xFF;
        buf[data_length + 1] = 0xFF;
        buf[data_length + 2] = 0xFF;
        buf[data_length + 3] = 0xFF;
        data_length += 4;
    }

    if (md_msc_code_program_words(addr, ~addr, buf, data_length, 0) != 0)
    {
        __enable_irq();
        return FALSE;
    }

    __enable_irq();
    return TRUE;
}

/**
  * @brief  get app flash info
  * @param  start_addr: flash start address
  * @param  size: flash size
  * @retval None
  */
void app_flash_info(uint32_t *start_addr, uint32_t *size)
{
    if (start_addr)
        *start_addr = APP_FLASH_START_ADDRESS;

    if (size)
        *size =  APP_FLASH_SIZE;

    return;
}

