/**********************************************************************************
 *
 * @file    .h
 * @brief   Head file of boot flash module.
 *
 * @version V1.0
 * @date    28 Feb 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          31 Dec 2019     AE Team         The first version
 *          28 Feb 2023     AE Team         Port to ES32VF2264
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
#ifndef __BOOT_FRAME_H__
#define __BOOT_FRAME_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Includes ------------------------------------------------------------------ */

#include "md_crc.h"
#include "md_uart.h"
#include "list.h"

/** @addtogroup Bootloader
  * @{
  */
/** @addtogroup Frame
  * @{
  */
  
/* Exported Macros ----------------------------------------------------------- */

/** @defgroup Frame_Public_Marcos Public Macros
  * @brief    Frame Public Macros
  * @{
  */
#define XMODE_STX   0x02    /**< header of 1k-xmode  frames */
#define XMODE_EOT   0x04    /**< frame end byte */
#define XMODE_ACK   0x06    /**< frame ACK byte */
#define XMODE_NACK  0x15    /**< frame NACK byte */
#define XMODE_CAN   0x18    /**< revocation frame transmission */
#define XMODE_EOF   0x1A    /**< padding data */
#define XMODE_CRC   0x43    /**< CRC checkout frame */
#define TIME_OUT    3000    /**< timeout(ms) */
#define TRY_CNT     10  /**< repeat try number */
/**
  * @}
  */

/* Exported Types ------------------------------------------------------------ */

/** @defgroup Frame_Public_Types Public Types
  * @brief    Frame Public Types
  * @{
  */
/**
  * @brief frame error state
  */
typedef enum
{
    FRAME_SUCCESS         = 0x01,   /**< Frame transmission sucess */
    FRAME_INDEX_ERROR     = 0x02,   /**< Frame index error */
    FRAME_RECEIVE_TIMEOUT = 0x04,   /**< Timeout error */
    FRAME_VERIFY_ERROR    = 0x08,   /**< Frame verigy error */
    UART_SEND_ERROR       = 0x10,   /**< UART send error */
    FLASH_WRITE_ERROR     = 0x20,   /**< Write flash error */
	APP_VERIFY_ERROR      = 0x40,   /**< App verify error */
} frames_state_t;

/**
  * @brief Command
  */
typedef enum
{
    XMOD_TX_CMD_START = XMODE_CRC,  /**< Start transfer */
    XMOD_TX_CMD_TERM  = XMODE_CAN,  /**< Terminal transfer */
    XMOD_TX_CMD_ACK   = XMODE_ACK,  /**< Acknowledgement */
    XMOD_TX_CMD_NCK   = XMODE_NACK, /**< Non-Acknowledgement */
} xmod_tx_cmd_t;

/**
  * @brief Frame environment structure
  */
typedef struct boot_frame_env_s
{
    UART_TypeDef *h_uart;      /**< Pointer to uart_handle_t structure */
    uint16_t idx;           /**< Current index of receive frame */
    uint8_t cnt;            /**< Number of repetitions */
    frames_state_t state;       /**< State */
    uint8_t soh;            /**< Start of header */
    uint8_t data[1029];     /**< Cache of the frame */
} boot_frame_env_t;
/**
  * @}
  */

/* Exported Variables -------------------------------------------------------- */

/** @addtogroup Frame_Public_Variables
  * @{
  */
extern CRC_TypeDef* g_h_crc;
extern boot_frame_env_t g_frame_env;
/**
  * @}
  */

/* Exported Functions -------------------------------------------------------- */

/** @addtogroup Frame_Public_Functions
  * @{
  */
extern void boot_update_info(void);
extern void boot_frame_init(UART_TypeDef *hperh);
extern int  boot_image_update(void);
/**
  * @}
  */
  
/* Exported Constants -------------------------------------------------------- */

/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
extern "C"
}
#endif /* __cplusplus */
#endif /* __BOOT_FRAME_H__ */
