/**
  *********************************************************************************
  * @file   boot_shell.c
  * @brief  shell module.
  *
  * @version V1.0
  * @date    28 Feb 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *          28 Feb 2023     AE Team         Port to ES32VF2264
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include "main.h"
#include "boot_shell.h"
#include "boot_flash.h"
#include "uartstdio.h"
#include "printf.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */

/** @addtogroup Bootloader
  * @{
  */
/** @defgroup Shell Shell
  * @brief    Bootloader Shell Module
  * @{
  */
/**
  * @defgroup Shell_Public_Variables Public Variable
  * @brief    Shell Public Variable
  * @{
  */
/* Public Variables ---------------------------------------------------------- */

boot_shell_env_t g_boot_shell_env;
char g_uart_buf[64];
/**
  * @}
  */

/**
  * @defgroup Shell_Private_Funtctions Private Functions
  * @brief    Shell Private Functions
  * @{
  */
/**
  * @brief  Refreshs the command cache.
  * @param  cmd: New command
  * @retval None
  */
/* Private Function ---------------------------------------------------------- */

static void shell_cmd_refresh(char *cmd)
{
    if (!(strcmp(g_boot_shell_env.cmd, g_boot_shell_env.cmd_cache0)))
    {
        return;
    }

    memcpy(g_boot_shell_env.cmd_cache2, g_boot_shell_env.cmd_cache1, 64);
    memcpy(g_boot_shell_env.cmd_cache1, g_boot_shell_env.cmd_cache0, 64);
    memcpy(g_boot_shell_env.cmd_cache0, cmd, 64);

    return;
}

/**
  * @brief  Handler for input shell command.
  * @param  cmd: input comamd by shell
  * @retval result of execute cmd @ref shell_cmd_exec_t
  */
static shell_cmd_exec_t shell_cmd_exec(char *cmd)
{
    char *pcmd = cmd;
    int idx = 0;
    char l_cmd = 0;
    char _arg[16] = {0};
    char _cmd[16] = {0};
    uint32_t arg[4] = {0U};
    boot_cmd_ent_t *ent = NULL;
    list_head_t *pos;

    if (*pcmd == '\0')
    {
        printf_e("\r\nes#");
        return CMD_NULL;
    }

    while (*pcmd)
    {
        int ptr=0;
        if (*pcmd == ' ')
        {
            pcmd++;
            continue;
        }

        ptr = 0;

        if (idx == 0)
        {
            while (*pcmd != ' ' && *pcmd && ptr < 16)
                _cmd[ptr++] = *pcmd++;
        }
        else
        {
            while (*pcmd != ' ' && *pcmd && ptr < 11)
                _arg[ptr++] = *pcmd++;

            arg[idx - 1] = strtoul(_arg, NULL, 0);
        }

        if (idx++ == 5)
            break;
    }

    l_cmd = idx - 1;

    list_for_each(pos, &g_boot_shell_env.list)
    {
        ent = list_entry(pos, boot_cmd_ent_t, link);

        if (strcmp(ent->cmd, _cmd) == 0)
            break;
    }

    if ((pos == &g_boot_shell_env.list) || (ent == NULL))
    {
        printf_e("Unkonw cmd: %s\r\n", _cmd);
        printf_e("es#");
        return CMD_UNKONW;
    }

    if (l_cmd != ent->nr_arg)
    {
        printf_e("Bad paramters, need %d paramters", ent->nr_arg);
        printf_e("es#");
        return CMD_ARG_ERR;
    }

    printf_e("\r\n");

    switch (ent->nr_arg)
    {
        case 0:
            ((cfun_arg0_t)(ent->fun_cmd))();
            break;

        case 1:
            ((cfun_arg1_t)(ent->fun_cmd))(arg[0]);
            break;

        case 2:
            ((cfun_arg2_t)(ent->fun_cmd))(arg[0], arg[1]);
            break;

        case 3:
            ((cfun_arg3_t)(ent->fun_cmd))(arg[0], arg[1], arg[2]);
            break;

        case 4:
            ((cfun_arg4_t)(ent->fun_cmd))(arg[0], arg[1], arg[1], arg[3]);
            break;

        default:
            break;
    }

    if ((strncmp(_cmd, "exit", 16)))
    {
        shell_cmd_refresh(cmd);
        printf_e("es#");
    }

    return CMD_SUCCESS;
}

/**
  * @brief  Check the password
  * @param  cmd: The password
  * @retval None
  */
static void shell_check_password(char *cmd)
{
    if ((strcmp(cmd, g_boot_shell_env.password)))
    {
        printf_e("\r\nPassword is incorrect\r\n");
        return;
    }

    g_boot_shell_env.check = 1;
    printf_e("es#");

    return;
}

/**
  * @brief  Handler for "UP" key
  * @retval None
  */
static void shell_up_func(void)
{
    uint8_t len   = 0U;
    uint8_t l_cmd = g_boot_shell_env.cursor;

    switch (g_boot_shell_env.idx_cache)
    {
        case 0:
            if (g_boot_shell_env.cmd_cache0[0] == '\0')
                return;

            memcpy(g_boot_shell_env.cmd, g_boot_shell_env.cmd_cache0, 64);
            break;

        case 1:
            if (g_boot_shell_env.cmd_cache1[0] == '\0')
                return;

            memcpy(g_boot_shell_env.cmd, g_boot_shell_env.cmd_cache1, 64);
            break;

        case 2:
            if (g_boot_shell_env.cmd_cache2[0] == '\0')
                return;

            memcpy(g_boot_shell_env.cmd, g_boot_shell_env.cmd_cache2, 64);
            break;

        default:
            return;
    }

    while (g_boot_shell_env.cmd[len] != '\0')
        len++;

    if (l_cmd)
    {
        memset(g_boot_shell_env.out_buf, 0x0, 64);
        memset(g_boot_shell_env.out_buf, 0x8, l_cmd);
        printf_e("%s", g_boot_shell_env.out_buf);
        memset(g_boot_shell_env.out_buf, ' ', l_cmd);
        printf_e("%s", g_boot_shell_env.out_buf);
        memset(g_boot_shell_env.out_buf, 0x8, l_cmd);
        printf_e("%s", g_boot_shell_env.out_buf);
    }

    g_boot_shell_env.cursor = len;
    printf_e("%s", g_boot_shell_env.cmd);
    ++g_boot_shell_env.idx_cache;

    return;
}

/**
  * @brief  Handler for "DOWN" key
  * @retval None
  */
static void shell_down_func(void)
{
    uint8_t len   = 0U;
    uint8_t l_cmd = g_boot_shell_env.cursor;

    switch (g_boot_shell_env.idx_cache)
    {
        case 3:
            if (g_boot_shell_env.cmd_cache1[0] == '\0')
                return;

            memcpy(g_boot_shell_env.cmd, g_boot_shell_env.cmd_cache1, 64);
            break;

        case 2:
            if (g_boot_shell_env.cmd_cache0[0] == '\0')
                return;

            memcpy(g_boot_shell_env.cmd, g_boot_shell_env.cmd_cache0, 64);
            break;

        case 1:
            memset(g_boot_shell_env.cmd, 0x0, 64);
            break;

        default:
            return;
    }

    while (g_boot_shell_env.cmd[len])
        ++len;

    if (l_cmd)
    {
        memset(g_boot_shell_env.out_buf, 0x0, 64);
        memset(g_boot_shell_env.out_buf, 0x8, l_cmd);
        printf_e("%s", g_boot_shell_env.out_buf);
        memset(g_boot_shell_env.out_buf, ' ', l_cmd);
        printf_e(g_boot_shell_env.out_buf);
        memset(g_boot_shell_env.out_buf, 0x8, l_cmd);
        printf_e(g_boot_shell_env.out_buf);
    }

    g_boot_shell_env.cursor = len;
    uart_send_str(g_boot_shell_env.cmd, len, 100);
    --g_boot_shell_env.idx_cache;

    return;
}

/**
  * @brief  Handler for "TAB" key
  * @retval None
  */
static void shell_tab_func(void)
{
    uint8_t k, len = 0U;
    uint32_t nr = 0U;
    list_head_t *pos;
    boot_cmd_ent_t *ent, *_ent;

    if (!g_boot_shell_env.cursor)
    {
        printf_e("\r\n");

        list_for_each(pos, &g_boot_shell_env.list)
        {
            ent = list_entry(pos, boot_cmd_ent_t, link);

            printf_e("%s\t", ent->cmd);

            if (!(++nr & 0x3))
                printf_e("\r\n");
        }

        if (nr & 0x3)
            printf_e("\r\nes#");
        else
            printf_e("\r\nes#");

        return;
    }

    list_for_each(pos, &g_boot_shell_env.list)
    {
        ent = list_entry(pos, boot_cmd_ent_t, link);

        if ((strncmp(g_boot_shell_env.cmd, ent->cmd, g_boot_shell_env.cursor)) == 0)
        {
            if (pos->next != &g_boot_shell_env.list)
            {
                _ent = list_entry(pos->next, boot_cmd_ent_t, link);

                if ((strncmp(g_boot_shell_env.cmd, _ent->cmd, g_boot_shell_env.cursor)) != 0)
                    goto match_1;
                else
                    goto match_x;
            }
            else
            {
                goto match_1;
            }
        }
    }

    return;

match_1:

    for (k = g_boot_shell_env.cursor; ent->cmd[k]; ++k)
    {
        g_boot_shell_env.cmd[k] = ent->cmd[k];
        ++len;
    }

    uart_send_str(&g_boot_shell_env.cmd[g_boot_shell_env.cursor], len, 100);
    g_boot_shell_env.cursor = k;
    return;

match_x:
    printf_e("\r\n");

    list_for_each(pos, &g_boot_shell_env.list)
    {
        ent = list_entry(pos, boot_cmd_ent_t, link);

        if ((strncmp(g_boot_shell_env.cmd, ent->cmd, g_boot_shell_env.cursor)) == 0)
        {
            printf_e("%s\t", ent->cmd);

            if (!(++nr & 0x3))
                printf_e("\r\n");
        }
    }

    if (nr & 0x3)
        printf_e("\r\nes#");
    else
        printf_e("\r\nes#");

    uart_send_str(g_boot_shell_env.cmd, g_boot_shell_env.cursor, 100);

    return;
}

/**
  * @brief  Receive a byte, This function is invoked in interrupt.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None
  */
void shell_uart_rx_cbk(void)
{
    g_boot_shell_env.rx_flag = 1;
	g_boot_shell_env.c_buf=md_uart_recv_data8(g_boot_shell_env.h_uart);
    return;
}

/**
  * @}
  */

/** @defgroup Shell_Public_Functions Public Functions
  * @brief    Shell Public Functions
  * @{
  */
/**
  * @brief  Get receive flag
  * @retval 1-Set, 0-Reset.
  */
uint8_t shell_rx_flag_get(void)
{
    return g_boot_shell_env.rx_flag;
}

/**
  * @brief  Clear receive flag
  * @retval None
  */
void shell_rx_flag_clear(void)
{
    g_boot_shell_env.rx_flag = 0;
    return;
}

/**
  * @brief  Check timeout
  */
/**
  * @brief  Insert a command into list.
  * @param  cmd: Name of the command.
  * @param  func: Function of the command.
  * @param  nr_arg: Number of the command's function.
  * @retval Status:
  *           0 - Success
  *          -1 - Pointer to uart_handle_t is empty
  *          -2 - Memory is empty
  */
int32_t shell_cmd_insert(char *cmd, void (*func)(void), uint32_t nr_arg)
{
    list_head_t *pos;
    boot_cmd_ent_t *ent, *_ent;

    if (g_boot_shell_env.h_uart == NULL)
        return -1;

    ent = (boot_cmd_ent_t *)malloc(sizeof(boot_cmd_ent_t));

    if (ent == NULL)
        return -2;

    memset(ent, 0x0, sizeof(boot_cmd_ent_t));
    ent->fun_cmd = func;
    ent->nr_arg = nr_arg;
    memcpy(ent->cmd, cmd, 15);

    list_for_each(pos, &g_boot_shell_env.list)
    {
        _ent = list_entry(pos, boot_cmd_ent_t, link);

        if ((strcmp(ent->cmd, _ent->cmd)) < 0)
            break;
    }

    list_add_tail(&ent->link, pos);
    ++g_boot_shell_env.nr_cmd;

    return 0;
}

/**
  * @brief  Initialize shell module
  * @param  hperh: Pointer to uart_handle_t structure.
  * @retval None
  */
void shell_init(UART_TypeDef *hperh)
{
    if (hperh == NULL)
    {
        g_boot_shell_env.h_uart = hperh;
        return;
    }

    memset(&g_boot_shell_env, 0x0, sizeof(g_boot_shell_env));
    INIT_LIST_HEAD(&g_boot_shell_env.list);
    g_boot_shell_env.h_uart = hperh;
	
    strcpy(g_boot_shell_env.password, "eastsoft");
    g_boot_shell_env.check = SET;

    printf_e("\r\n");
    printf_e("*************************************\r\n");
    printf_e("*        Welcome to Eastsoft        *\r\n");
    printf_e("*************************************\r\n\r\n");

    return;
}

/**
  * @brief  Function for shell's task.
  * @param  arg: The parameter.
  * @retval None
  */
void shell_task_func(void *arg)
{
    char cur;

    cur = g_boot_shell_env.c_buf;

    if (cur == 0x1B)  /*ESC*/
    {
        g_boot_shell_env.fn_key[0] = 0x1B;
        return;
    }

    if (g_boot_shell_env.fn_key[0] == 0x1B)   /* Up and Down */
    {
        if (cur == '[')
        {
            g_boot_shell_env.fn_key[1] = cur;
        }
        else if (cur == 'A')
        {
            shell_up_func();
            memset(g_boot_shell_env.fn_key, 0x0, 4);
        }
        else if (cur == 'B')
        {
            shell_down_func();
            memset(g_boot_shell_env.fn_key, 0x0, 4);
        }
        else
        {
            g_boot_shell_env.fn_key[0] = 0;
        }

        return;
    }

    if ((cur == ' ' || (cur >= 'A' && cur <= 'Z') || (cur >= 'a' && cur <= 'z')
            || (cur >= '0' && cur <= '9') || (cur == '-')
            || (cur == '_')) && (g_boot_shell_env.cursor < 64))
    {
        g_boot_shell_env.cmd[g_boot_shell_env.cursor++] = cur;

        if (g_boot_shell_env.check)
            uart_send_str(&cur, 1, 100);
    }

    switch (cur)
    {
        case 0x08:  /* Backspace */
            if (!g_boot_shell_env.cursor)
                break;

            g_boot_shell_env.cmd[--g_boot_shell_env.cursor] = 0;

            if (!g_boot_shell_env.check)
                break;

            uart_send_str(&cur, 1, 100);

            break;

        case 0x09:  /* TAB */
            shell_tab_func();
            break;

        case 0x0D:  /* Enter */
            if (!g_boot_shell_env.check)
            {
                shell_check_password(g_boot_shell_env.cmd);
                memset(g_boot_shell_env.cmd, 0x0, 64);
                g_boot_shell_env.cursor = 0;
                break;
            }

            shell_cmd_exec(g_boot_shell_env.cmd);
            memset(g_boot_shell_env.cmd, 0x0, 64);
            g_boot_shell_env.idx_cache = 0;
            g_boot_shell_env.cursor    = 0;
            break;

        default:
            break;
    }

    return;
}

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    vsnprintf(g_uart_buf, 64, fmt, args);
    va_end(args);

	uart_send_str(g_uart_buf, strlen(g_uart_buf), 100);
    md_delay_1ms(1);
    return;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
