/**
  *********************************************************************************
  *
  * @file    boot_fatfs.h
  * @brief   Header file for fatfs
  *
  * @version V1.0
  * @date    31 Dec 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef   __FATFS_H__
#define   __FATFS_H__

#ifdef __cplusplus
extern "C" {
#endif/*__cplusplus*/

/* Includes ------------------------------------------------------------------ */
#include "ald_utils.h"
#include "string.h"
#include "md_msc.h"
#include "md_syscfg.h"
#include "boot_conf.h"
#include "main.h"

/** @addtogroup Bootloader
  * @{
  */
/** @addtogroup USB_Fatfs
  * @{
  */
/** @defgroup Fatfs_Public_Macros Public Macros
  * @brief    Fatfs Public Macros
  * @{
  */

/**
  * @}
  */

/** @addtogroup Fatfs_Public_Variables
  * @{
  */
/**
  * @}
  */

/* Exported Macros ----------------------------------------------------------- */
/* Exported Types ------------------------------------------------------------ */

/** @defgroup Fatfs_Public_Types Public Types
  * @brief    Fatfs Public Types
  * @{
  */
typedef enum
{
    TYPE_UKNOW = 0, /**< Unkown */
    TYPE_FAT12 = 1, /**< FAT12 */
    TYPE_FAT16 = 2, /**< FAT16 */
    TYPE_FAT32 = 3, /**< FAT32 */
} fat_type_e_t;

typedef struct bin_hdr_s
{
    uint32_t vid;       /**< producer ID */
    uint32_t pid;       /**< product ID */
    uint32_t addr;      /**< Start address of bin file */
    uint32_t len;       /**< length of bin file */
    uint32_t crc;       /**< crc of bin file */
    uint32_t rsv[3];    /**< reserve */
} bin_hdr_t;

typedef struct dbr_s
{
    uint8_t g_es_fatfs_bs_jmpboot[3];      /**< Jump Instruction */
    uint8_t g_es_fatfs_bs_oemname[8];      /**< MSWIN4.1 */
    uint8_t g_es_fatfs_bpb_bytepersec[2];  /**< bytes per sector */
    uint8_t g_es_fatfs_bpb_secperclus[1];  /**< sector per cluster */
    uint8_t g_es_fatfs_bpb_rsvseccnt[2];   /**< reserve sector number */
    uint8_t g_es_fatfs_bpb_numfats[1];     /**< backup number of FAT table */
    uint8_t g_es_fatfs_bpb_rootentcnt[2];  /**< boot table number of FATFS */
    uint8_t g_es_fatfs_bpb_totsec16[2];    /**< total sector of FAT16 */
    uint8_t g_es_fatfs_bpb_media[1];       /**< memory medium */
    uint8_t g_es_fatfs_bpb_fatsz16[2];     /**< sector per FAT table */
    uint8_t g_es_fatfs_bpb_secpertrk[2];   /**< sector per magnetic track */
    uint8_t g_es_fatfs_bpb_numheads[2];    /**< number of magnetic head */
    uint8_t g_es_fatfs_bpb_hiddsec[4];     /**< hide sector before FAT sector */
    uint8_t g_es_fatfs_bpb_totsec32[4];    /**< total sector of FAT32 */
} dbr_t;

typedef struct fat12_16_s
{
    uint8_t g_es_fatfs_bs_drcnum[1];       /**< disk driver parameter */
    uint8_t g_es_fatfs_bs_reserved1[1];    /**< reserve for NT */
    uint8_t g_es_fatfs_bs_bootsig[1];      /**< Extended Boot Signature */
    uint8_t g_es_fatfs_bs_volid[4];        /**< Volume Serial Number */
    uint8_t g_es_fatfs_bs_vollab[11];      /**< Volume Label */
    uint8_t g_es_fatfs_bs_filsystype[8];   /**< System ID */
} fat12_16_t;

typedef struct fat32_s
{
    uint8_t BPB_FATSz32[4];     /**< Sectors Per FAT */
    uint8_t BPS_ExtFlags[2];    /**< Extended Flag */
    uint8_t BPB_FSVer[2];       /**< File ystem Version */
    uint8_t BPB_RootClus[4];    /**< Root Cluster Number */
    uint8_t BPB_FSInfo[2];      /**< File System Information SectorNumber */
    uint8_t BPB_BkBootSec[2];   /**< backup guide sector */
    uint8_t BPB_Reserved[12];   /**< reserve */
    uint8_t g_es_fatfs_bs_drcnum[1];       /**< Physical Drive Number */
    uint8_t g_es_fatfs_bs_reserved1[1];    /**< reserve for NT */
    uint8_t g_es_fatfs_bs_bootsig[1];      /**< Extended Boot Signature */
    uint8_t g_es_fatfs_bs_volid[4];        /**< Volume Serial Number */
    uint8_t g_es_fatfs_bs_vollab[11];      /**< Volume Label */
    uint8_t g_es_fatfs_bs_filsystype[8];   /**< System ID */
} fat32_t;

typedef struct fat12_16_dbr_s
{
    dbr_t dbr;      /**< FAT16 DBR sector information */
    fat12_16_t fat;     /**< FAT16 Extended information */
} fat12_16_dbr_t;

typedef struct fat32_dbr_s
{
    dbr_t dbr;      /**< FAT32 DBR sector information */
    fat32_t fat;        /**< FAT32 Extended information */
} fat32_dbr_t;

typedef enum
{
    ATTR_READ_ONLY  = 0x01, /**< read only file */
    ATTR_HIDDEN = 0x02, /**< hide file */
    ATTR_SYSTEM = 0x04, /**< system file */
    ATTR_VOLUME_ID  = 0x08, /**< Volume file */
    ATTR_DIRECTORY  = 0x10, /**< directory file */
    ATTR_ARCHIEVE   = 0x20, /**< archive file */
} fat_dir_attr_e_t;


typedef struct fat_dir_s
{
    uint8_t Dir_Flag;       /**< directory flag */
    uint8_t Dir_Name[7];        /**< directory name */
    uint8_t Dir_exName[3];      /**< directory Extended name */
    uint8_t Dir_Attr;       /**< directory attribute */
    uint8_t Dir_NTRes;      /**< reserve */
    uint8_t Dir_CrtTimeTeeth;   /**< directory create time */
    uint8_t Dir_CrtTime[2];     /**< directory create time */
    uint8_t Dir_CrtData[2];     /**< directory create date */
    uint8_t Dir_LastAccData[2]; /**< directory last access date */
    uint8_t Dir_FstClusHI[2];   /**< directory cluster high byte */
    uint8_t Dir_WrtTime[2];     /**< directory last write time */
    uint8_t Dir_WrtData[2];     /**< directory last write date */
    uint8_t FstClusLO[2];       /**< directory cluster low byte */
    uint16_t Dir_FileSize[2];   /**< directory size */
} fat_dir_t;

typedef struct fat_init_arg_s
{
    uint8_t BPB_Sector_No;      /**< BPB sector */
    uint32_t Total_Size;        /**< disk capacity */
    uint32_t FirstDirClust;     /**< first cluster of boot */
    uint32_t FirstDataSector;   /**< first sector of data */
    uint32_t BytesPerSector;    /**< Bytes per sector*/
    uint32_t FATsectors;        /**< sector per FAT*/
    uint32_t SectorsPerClust;   /**< sector per cluster*/
    uint32_t FirstFATSector;    /**< first sector of FAT */
    uint32_t FirstDirSector;    /**< first sector of directory */
    uint32_t RootDirSectors;    /**< sector of boot directory */
    uint32_t RootDirCount;      /**< file number of boot */
} fat_init_arg_t;
/**
  * @}
  */

/* Exported Variables -------------------------------------------------------- */
extern fat_type_e_t g_es_fatfs_type;

/* Exported Constants -------------------------------------------------------- */
/* Exported Functions -------------------------------------------------------- */

/** @addtogroup Fatfs_Public_Functions
  * @{
  */
extern void get_fatfs_type(void *src);
extern uint32_t get_dir_param(void *src);
extern type_bool_t curfat_is_bat(uint32_t FATContent);
extern type_bool_t curfat_is_eof(uint32_t FATContent);
extern void init_fat12_fs(void);
extern void fs_flash_read(uint8_t *buf, uint32_t sector, uint32_t num_block);
extern void fs_flash_write(uint8_t *buf, uint32_t sector, uint32_t num_block);
extern void app_ram_write(uint8_t *buf, uint32_t len);
extern void app_flash_write(uint8_t *buf, uint32_t len);
extern uint8_t *get_bin_hdr(void *src, bin_hdr_t *b_hdr, uint32_t StFAT);
extern type_bool_t check_bin_crc(uint8_t *data, bin_hdr_t *b_hdr);
extern void run_in_flash(void);
extern void run_in_ram(void);
extern void set_run_flag(uint8_t flag, uint32_t size);
extern uint8_t get_run_flag(uint32_t *size);
extern uint32_t get_rootclust_sec(void *src);
extern uint32_t get_curfat_sec(void *src, uint32_t crufat);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
extern "C"
}
#endif/*__cplusplus*/
#endif/*__FATFS_H__*/
