/**********************************************************************************
 *
 * @file    .c
 * @brief   Source file
 *
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date               Author          Notes
 *          2023-01-17         liuhy           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes-------------------------------------------------------------------*/
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

void uart_string_out(const char *input_str)
{
    uint8_t *str = (uint8_t *)input_str;

    while (*str)
    {
        while ((READ_BIT(CUART1->STAT, UART_STAT_TFEMPTY_MSK)) != UART_STAT_TFEMPTY_MSK)
        {
        }

        md_uart_set_send_data8(CUART1, *str);
        str++;
    }

    while ((READ_BIT(CUART1->STAT, UART_STAT_TFEMPTY_MSK)) != UART_STAT_TFEMPTY_MSK)
    {
    }

    while ((READ_BIT(CUART1->STAT, UART_STAT_TSBUSY_MSK)) == UART_STAT_TSBUSY_MSK)
    {
    }
}

static void uart_stdio_init(void)
{
    md_gpio_init_t  gpio_config;
    md_uart_init_t  uart_handle;

    md_gpio_init_struct(&gpio_config);
    /* Initialize tx pin */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.odrv  = MD_GPIO_OUT_DRIVE_STRONG;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_2;
    md_gpio_init(GPIOA, MD_GPIO_PIN_2, &gpio_config);

    gpio_config.mode  = MD_GPIO_MODE_INPUT;
    md_gpio_init(GPIOA, MD_GPIO_PIN_3, &gpio_config);

    /* Initialize UART */
    md_uart_init_struct(&uart_handle);
    uart_handle.baud        = 115200;
    uart_handle.word_length = MD_UART_WORD_LENGTH_8B;
    uart_handle.stop_bits   = MD_UART_STOP_BITS_1;
    uart_handle.parity      = MD_UART_PARITY_NONE;
    uart_handle.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_handle.mode        = MD_UART_MODE;
    md_uart_init(CUART1, &uart_handle);
}

/**
  * @brief user_system_config system configuration.
  * @param none
  * @retval none
  */
static void user_system_config(void)
{
#if 0 /*如果有8M外部晶振，且需要更精确和高速的时钟*/
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 72000000);
#else
    md_cmu_clock_config_default();
#endif /*时钟选择*/

    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();
    md_init_1ms_tick();
    __enable_irq();

    return;
}

void jump2app(void)
{
    __disable_irq();

    MD_SYSCFG_UNLOCK();
    md_syscfg_set_cpu_boot_addr(APP_FLASH_S);
    MD_SYSCFG_LOCK();

    __enable_irq();

    md_rmu_reset_system();
}

uint8_t update_app(void)
{
    uint32_t i, j, addr = *((uint32_t *)UPDATE_INFO_NEW_APP_ADDR);

    if ((addr != 0x21000) && (addr != 0x20000)) /*目前2种分配方式：8+120+128、8+124+124*/
    {
        uart_string_out("[boot]new app addr err\r\n");

        return -1;
    }

    /*将新APP覆盖到运行APP*/
    i = APP_FLASH_S;
    j = addr;

    for (;;)
    {
        if (md_msc_code_erase_page(i, ~i, 0) == MD_ERROR)
        {
            uart_string_out("[boot]erase page fail\r\n");
        }

        if (md_msc_code_program_words(i, ~i, (uint8_t *)j, DISK_BLOCK_SIZE, 0) != 0)
        {
            uart_string_out("[boot]program page fail\r\n");
        }

        i += DISK_BLOCK_SIZE;
        j += DISK_BLOCK_SIZE;

        if ((i >= addr) || (j >= INFO_FLAG_FLASH_E))
            break;
    }

    /*复制结果是否正确*/
    i = APP_FLASH_S;
    j = addr;

    for (;;)
    {
        if ((*(uint32_t *)(i)) != (*(uint32_t *)(j)))
        {
            uart_string_out("[boot]app update err\r\n");

            while (1);
        }

        i += 4;
        j += 4;

        if ((i >= addr) || (j >= INFO_FLAG_FLASH_E))
            break;
    }

    return 0;
}

void clear_flag_and_info(void)
{
    if (md_msc_code_erase_page(INFO_FLAG_FLASH_S, ~(INFO_FLAG_FLASH_S), 0) == MD_ERROR)
        uart_string_out("[boot]clear flag err\r\n");
}

uint8_t check_new_app_crc(void)
{
    uint32_t i, end, addr = *((uint32_t *)UPDATE_INFO_NEW_APP_ADDR);
    md_crc_init_t g_crc_init;

    md_crc_init_struct(&g_crc_init);
    g_crc_init.mode = MD_CRC_MODE_32;
    g_crc_init.len = MD_CRC_DATASIZE_32;
    g_crc_init.order = MD_CRC_BYTORD_LOW;
    g_crc_init.seed = 0xFFFFFFFF;
    g_crc_init.chs_inv = ENABLE;
    g_crc_init.chs_rev = ENABLE;
    g_crc_init.data_inv = DISABLE;
    g_crc_init.data_rev = ENABLE;
    md_crc_init(&g_crc_init);

    if ((addr != 0x21000) && (addr != 0x20000)) /*目前2种分配方式：8+120+128、8+124+124*/
    {
        uart_string_out("[boot]new app addr err\r\n");

        return 2;
    }

    i = addr;
    end = addr + (*((uint32_t *)(UPDATE_INFO_CRC_LEN)));

    if ((i & 0x3) || (end & 0x3) || (end > INFO_FLAG_FLASH_S))
        return 2;

    for (; i < end ; i += 4)
    {
        md_crc_write_data(CRC, *((uint32_t *)(i)));
    }

    if ((*((uint32_t *)(UPDATE_INFO_CRC_VALUE)) != md_crc_get_check_result(CRC)))
        return 1;

    return 0;
}

/**
  * @brief main.
  * @param none
  * @retval none
  */
int main(void)
{
    uint32_t addr = *((uint32_t *)UPDATE_INFO_NEW_APP_ADDR);

    user_system_config();
    uart_stdio_init();

    if ((addr == 0x21000) || (addr == 0x20000)) /*目前2种分配方式：8+120+128、8+124+124*/
    {
        uart_string_out("[boot]check len\r\n");

        if (((*((uint32_t *)(UPDATE_INFO_CRC_LEN))) > 0) && (((*((uint32_t *)(UPDATE_INFO_CRC_LEN))) <= (INFO_FLAG_FLASH_S - addr))))
        {
            uart_string_out("[boot]updating\r\n");

            if (!(check_new_app_crc()))
            {
                update_app();
            }
            else
            {
                uart_string_out("[boot]new app crc err\r\n");
            }

            clear_flag_and_info();
        }
    }

    uart_string_out("[boot]jump to app\r\n");

    jump2app();

    while (1)
    {
        md_delay_1ms(1000);
    }
}

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE ****/
