#include "usbh_core.h"
#include "platform.h"
#include "reg_sysc_cpu.h"
#include "musb_type.h"
#include "soc_pinmux.h"

void USB_Handler(void);

#define ES_USB_PERH_EP_MAX_INDEX     (6U)
#define ES_USB_HOST_DEFAULT_FIFO_SIZE (1024U)
#define ES_USB_HOST_FIFO_STATIC_USED  (ES_USB_PERH_EP_MAX_INDEX * ES_USB_HOST_DEFAULT_FIFO_SIZE)

#define musb_get_active_ep()   (USB0->EPIDX)
#define musb_set_active_ep(XXX)    ((USB0->EPIDX) = (XXX))

typedef enum {
    USB_EP0_STATE_SETUP = 0x0,      /**< SETUP DATA */
    USB_EP0_STATE_IN_DATA = 0x1,    /**< IN DATA */
    USB_EP0_STATE_OUT_DATA = 0x3,   /**< OUT DATA */
    USB_EP0_STATE_IN_STATUS = 0x4,  /**< IN status */
    USB_EP0_STATE_OUT_STATUS = 0x5, /**< OUT status */
} ep0_state_t;

struct musb_pipe {
    uint8_t dev_addr;
    uint8_t ep_addr;
    uint8_t ep_type;
    uint8_t ep_interval;
    uint8_t speed;
    uint16_t ep_mps;
    bool inuse;
    uint32_t xfrd;
    volatile bool waiter;
    usb_osal_sem_t waitsem;
    struct usbh_hubport *hport;
    struct usbh_urb *urb;
};

struct musb_hcd {
    volatile bool port_csc;
    volatile bool port_pec;
    volatile bool port_pe;
    struct musb_pipe pipe_pool[ES_USB_PERH_EP_MAX_INDEX + 1][2]; /* Support Bidirectional ep */
} g_musb_hcd;

typedef union {
    uint8_t   u8;
    uint16_t  u16;
    uint32_t  u32;
} hw_fifo_t;
static volatile uint8_t usb_ep0_state = USB_EP0_STATE_SETUP;

uint16_t es_usbh_get_pipe_info_ep_mps(struct musb_pipe * p)
{
	if(p)
		return p->ep_mps;
	else
		return 0;
}

__WEAK void usb_hc_low_level_init(void)
{
}

__WEAK void usb_hc_low_level_deinit(void)
{
}

static void es_usbd_ep_write_packet_8bit(uint8_t ep, void *buf, unsigned len)
{
    volatile hw_fifo_t *reg = (volatile hw_fifo_t*)((&USB0->FIFO0_WORD) + ep);
    uintptr_t addr = (uintptr_t)buf;
    while (len >= 4) {
        reg->u32 = *(uint32_t const *)addr;
        addr += 4;
        len  -= 4;
    }
    if (len >= 2) {
        reg->u16 = *(uint16_t const *)addr;
        addr += 2;
        len  -= 2;
    }
    if (len) {
        reg->u8 = *(uint8_t const *)addr;
    }
}

static void es_usbd_ep_read_packet_8bit(uint8_t ep, void *buf, unsigned len)
{
    volatile hw_fifo_t *reg = (volatile hw_fifo_t*)((&USB0->FIFO0_WORD) + ep);
    uintptr_t addr = (uintptr_t)buf;
    while (len >= 4) {
        *(uint32_t *)addr = reg->u32;
        addr += 4;
        len  -= 4;
    }
    if (len >= 2) {
        *(uint16_t *)addr = reg->u16;
        addr += 2;
        len  -= 2;
    }
    if (len) {
        *(uint8_t *)addr = reg->u8;
    }
}

void musb_control_pipe_init(struct musb_pipe *pipe, struct usb_setup_packet *setup, uint8_t *buffer, uint32_t buflen)
{
    uint8_t old_ep_index;

    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(0);
	
    USB0->EPIDX = 0;
    USB0->FADDR = pipe->dev_addr;
	
	//ald_usb_host_ep_config(0, 64, 0, 0, (ALD_USB_EP_MODE_CTRL | ALD_USB_EP_HOST_OUT));

    es_usbd_ep_write_packet_8bit(0, (uint8_t *)setup, 8);
    USB0->CSRL0 = (USB_CSRL0_TXRDY | USB_CSRL0_SETUP);
    musb_set_active_ep(old_ep_index);
}

void musb_bulk_pipe_init(struct musb_pipe *pipe, uint8_t *buffer, uint32_t buflen)
{
    uint8_t ep_idx;
    uint8_t old_ep_index;

    ep_idx = pipe->ep_addr & 0x7f;
    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
	
    USB0->EPIDX = ep_idx;
    USB0->FADDR = pipe->dev_addr;
	
    if (pipe->ep_addr & 0x80) 
	{
            USB0->RXTYPE = (uint8_t)USB_RXTYPE1_PROTO_BULK | (0xF & ep_idx);
            USB0->RXINTERVAL = (uint8_t)pipe->ep_interval;
            USB0->RXMAXP = (uint8_t)pipe->ep_mps;
			
        USB0->RXCSRL |= USB_RXCSRL1_REQPKT;
    } 
	else 
	{
            USB0->TXTYPE = (uint8_t)USB_TXTYPE1_PROTO_BULK | (0xF & ep_idx);
            USB0->TXINTERVAL = (uint8_t)pipe->ep_interval;
            USB0->TXMAXP = (uint8_t)pipe->ep_mps;

        if (buflen > pipe->ep_mps)
            buflen = pipe->ep_mps;

        es_usbd_ep_write_packet_8bit(ep_idx, buffer, buflen);
		USB0->TXCSRL = USB_TXCSRL1_TXRDY;
    }
	
    musb_set_active_ep(old_ep_index);
}

void musb_intr_pipe_init(struct musb_pipe *pipe, uint8_t *buffer, uint32_t buflen)
{
    uint8_t ep_idx;
    uint8_t old_ep_index;

    ep_idx = pipe->ep_addr & 0x7f;
    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
	
    USB0->EPIDX = ep_idx;
    USB0->FADDR = pipe->dev_addr;
	
    if (pipe->ep_addr & 0x80) 
	{
            USB0->RXTYPE = (uint8_t)USB_RXTYPE1_PROTO_INT | (0xF & ep_idx);
            USB0->RXINTERVAL = (uint8_t)pipe->ep_interval;
            USB0->RXMAXP = (uint8_t)pipe->ep_mps;
			
        USB0->RXCSRL |= USB_RXCSRL1_REQPKT;
    } 
	else 
	{
            USB0->TXTYPE = (uint8_t)USB_TXTYPE1_PROTO_INT | (0xF & ep_idx);
            USB0->TXINTERVAL = (uint8_t)pipe->ep_interval;
            USB0->TXMAXP = (uint8_t)pipe->ep_mps;

        if (buflen > pipe->ep_mps)
            buflen = pipe->ep_mps;

        es_usbd_ep_write_packet_8bit(ep_idx, buffer, buflen);
        USB0->TXCSRL = USB_TXCSRL1_TXRDY;
    }
    
    musb_set_active_ep(old_ep_index);
}

void musb_iso_pipe_init(struct musb_pipe *pipe, uint8_t *buffer, uint32_t buflen)
{
    uint8_t ep_idx;
    uint8_t old_ep_index;

    ep_idx = pipe->ep_addr & 0x7f;
    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
	
    USB0->EPIDX = ep_idx;
    USB0->FADDR = pipe->dev_addr;
	
    if (pipe->ep_addr & 0x80) 
	{
            USB0->RXTYPE = (uint8_t)USB_RXTYPE1_PROTO_ISOC | (0xF & ep_idx);
            USB0->RXINTERVAL = (uint8_t)pipe->ep_interval;
            USB0->RXMAXP = (uint8_t)pipe->ep_mps;
			
        USB0->RXCSRL |= USB_RXCSRL1_REQPKT;
    } 
	else 
	{
            USB0->TXTYPE = (uint8_t)USB_TXTYPE1_PROTO_ISOC | (0xF & ep_idx);
            USB0->TXINTERVAL = (uint8_t)pipe->ep_interval;
            USB0->TXMAXP = (uint8_t)pipe->ep_mps;

        if (buflen > pipe->ep_mps)
            buflen = pipe->ep_mps;

        es_usbd_ep_write_packet_8bit(ep_idx, buffer, buflen);
        USB0->TXCSRL = USB_TXCSRL1_TXRDY;
    }
    
    musb_set_active_ep(old_ep_index);
}

static int usbh_reset_port(const uint8_t port)
{
    g_musb_hcd.port_pe = 0;
	USB0->POWER |= USB_POWER_RESET;
    usb_osal_msleep(50);
	USB0->POWER &= ~USB_POWER_RESET;
    usb_osal_msleep(20);
    g_musb_hcd.port_pe = 1;
    return 0;
}

static uint8_t usbh_get_port_speed(const uint8_t port)
{
    uint8_t speed = USB_SPEED_UNKNOWN;
	
	if (USB0->DEVCTL & USB_DEVCTL_FSDEV)
		return USB_SPEED_FULL;

	if (USB0->DEVCTL & USB_DEVCTL_LSDEV)
		return USB_SPEED_LOW;

	return speed;
}

int usb_hc_init(void)
{
    memset(&g_musb_hcd, 0, sizeof(struct musb_hcd));

    for (uint8_t i = 0; i < ES_USB_PERH_EP_MAX_INDEX; i++) {
        g_musb_hcd.pipe_pool[i][0].waitsem = usb_osal_sem_create(0);
        g_musb_hcd.pipe_pool[i][1].waitsem = usb_osal_sem_create(0);
    }

    usb_hc_low_level_init();

	csi_vic_disable_irq(USB_IRQn);
	CLIC->CLICINT[USB_IRQn].ATTR |= (1);
	pinmux_usb_init(1);
	
    rv_set_int_isr(USB_IRQn, USB_Handler);
    csi_vic_clear_pending_irq(USB_IRQn);
    csi_vic_enable_irq(USB_IRQn);
	
	
    SYSC_CPU->PD_CPU_CLKG = SYSC_CPU_CLKG_CLR_USB_MASK;
    SYSC_CPU->PD_CPU_SRST = SYSC_CPU_SRST_CLR_USB_N_MASK;
    SYSC_CPU->PD_CPU_SRST = SYSC_CPU_SRST_SET_USB_N_MASK;
	
	
	csi_vic_enable_irq(USB_IRQn);
    SYSC_CPU->PD_CPU_CLKG = SYSC_CPU_CLKG_SET_USB_MASK;
	
    usb_status_set(true);
	
    //USB0->POWER |= 0x1;
    //USB0->IE = 0xFF;
    USB0->IE = 0xF7;
    USB0->TXIE = 0x7F;
    USB0->RXIE = 0x7E;
	
	
    //USB0->IS = 0xFF;
    //USB0->TXIS = 0x7F;
    //USB0->RXIS = 0x7E;

    //dcd_connect();
	
	//USB0->DEVCTL &= 0xFE;
	
	for(uint32_t i = 0; i <= ES_USB_PERH_EP_MAX_INDEX; i++)
	{
        USB0->EPIDX     = i;
        USB0->TXFIFO_SIZE[0] = 8;
        USB0->TXFIFO_SIZE[1] = 3 << USB_TX_FIFO_SIZE_POS;
        USB0->RXFIFO_SIZE[0] = 8;
        USB0->RXFIFO_SIZE[1] = 3 << USB_RX_FIFO_SIZE_POS;
	}
	
	USB0->EPIDX = 0;
	
    // unsigned cnt;
	//cnt = 2880000;
    // while (cnt--) __NOP();
	
	USB0->DEVCTL |= 0x2;
	
	//cnt = 288000;
    // while (cnt--) __NOP();
	 
	USB0->DEVCTL |= 0x1;
	
	//cnt = 288000;
    // while (cnt--) __NOP();
	 
	//usbh_reset_port(0);
	
    return 0;
}

int usb_hc_deinit(void)
{
    return 0;
}

int usbh_roothub_control(struct usb_setup_packet *setup, uint8_t *buf)
{
    uint8_t nports;
    uint8_t port;
    uint32_t status;

    nports = CONFIG_USBHOST_MAX_RHPORTS;
    port = setup->wIndex;
    if (setup->bmRequestType & USB_REQUEST_RECIPIENT_DEVICE) {
        switch (setup->bRequest) {
            case HUB_REQUEST_CLEAR_FEATURE:
                switch (setup->wValue) {
                    case HUB_FEATURE_HUB_C_LOCALPOWER:
                        break;
                    case HUB_FEATURE_HUB_C_OVERCURRENT:
                        break;
                    default:
                        return -EPIPE;
                }
                break;
            case HUB_REQUEST_SET_FEATURE:
                switch (setup->wValue) {
                    case HUB_FEATURE_HUB_C_LOCALPOWER:
                        break;
                    case HUB_FEATURE_HUB_C_OVERCURRENT:
                        break;
                    default:
                        return -EPIPE;
                }
                break;
            case HUB_REQUEST_GET_DESCRIPTOR:
                break;
            case HUB_REQUEST_GET_STATUS:
                memset(buf, 0, 4);
                break;
            default:
                break;
        }
    } else if (setup->bmRequestType & USB_REQUEST_RECIPIENT_OTHER) {
        switch (setup->bRequest) {
            case HUB_REQUEST_CLEAR_FEATURE:
                if (!port || port > nports) {
                    return -EPIPE;
                }

                switch (setup->wValue) {
                    case HUB_PORT_FEATURE_ENABLE:
                        break;
                    case HUB_PORT_FEATURE_SUSPEND:
                    case HUB_PORT_FEATURE_C_SUSPEND:
                        break;
                    case HUB_PORT_FEATURE_POWER:
                        break;
                    case HUB_PORT_FEATURE_C_CONNECTION:
                        g_musb_hcd.port_csc = 0;
                        break;
                    case HUB_PORT_FEATURE_C_ENABLE:
                        g_musb_hcd.port_pec = 0;
                        break;
                    case HUB_PORT_FEATURE_C_OVER_CURREN:
                        break;
                    case HUB_PORT_FEATURE_C_RESET:
                        break;
                    default:
                        return -EPIPE;
                }
                break;
            case HUB_REQUEST_SET_FEATURE:
                if (!port || port > nports) {
                    return -EPIPE;
                }

                switch (setup->wValue) {
                    case HUB_PORT_FEATURE_SUSPEND:
                        break;
                    case HUB_PORT_FEATURE_POWER:
                        break;
                    case HUB_PORT_FEATURE_RESET:
                        usbh_reset_port(port);
                        break;

                    default:
                        return -EPIPE;
                }
                break;
            case HUB_REQUEST_GET_STATUS:
                if (!port || port > nports) {
                    return -EPIPE;
                }

                status = 0;
                if (g_musb_hcd.port_csc) {
                    status |= (1 << HUB_PORT_FEATURE_C_CONNECTION);
                }
                if (g_musb_hcd.port_pec) {
                    status |= (1 << HUB_PORT_FEATURE_C_ENABLE);
                }

                if (g_musb_hcd.port_pe) {
                    status |= (1 << HUB_PORT_FEATURE_CONNECTION);
                    status |= (1 << HUB_PORT_FEATURE_ENABLE);
                    if (usbh_get_port_speed(port) == USB_SPEED_LOW) {
                        status |= (1 << HUB_PORT_FEATURE_LOWSPEED);
                    } else if (usbh_get_port_speed(port) == USB_SPEED_HIGH) {
                        status |= (1 << HUB_PORT_FEATURE_HIGHSPEED);
                    }
                }

                memcpy(buf, &status, 4);
                break;
            default:
                break;
        }
    }
    return 0;
}

int usbh_ep_pipe_reconfigure(usbh_pipe_t pipe, uint8_t dev_addr, uint8_t ep_mps, uint8_t mult)
{
    struct musb_pipe *ppipe = (struct musb_pipe *)pipe;

    ppipe->dev_addr = dev_addr;
    ppipe->ep_mps = ep_mps;

    return 0;
}

int usbh_pipe_alloc(usbh_pipe_t *pipe, const struct usbh_endpoint_cfg *ep_cfg)
{
    struct musb_pipe *ppipe;
    uint8_t old_ep_index;
    uint8_t ep_idx;
    usb_osal_sem_t waitsem;

    ep_idx = ep_cfg->ep_addr & 0x7f;

    if (ep_idx > ES_USB_PERH_EP_MAX_INDEX) {
        return -ENOMEM;
    }

    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(ep_idx);

    if (ep_cfg->ep_addr & 0x80) {
        ppipe = &g_musb_hcd.pipe_pool[ep_idx][1];
    } else {
        ppipe = &g_musb_hcd.pipe_pool[ep_idx][0];
    }

    /* store variables */
    waitsem = ppipe->waitsem;

    memset(ppipe, 0, sizeof(struct musb_pipe));

    ppipe->ep_addr = ep_cfg->ep_addr;
    ppipe->ep_type = ep_cfg->ep_type;
    ppipe->ep_mps = ep_cfg->ep_mps;
    ppipe->ep_interval = ep_cfg->ep_interval;
    ppipe->speed = ep_cfg->hport->speed;
    ppipe->dev_addr = ep_cfg->hport->dev_addr;
    ppipe->hport = ep_cfg->hport;

    if (ep_cfg->ep_type == USB_ENDPOINT_TYPE_CONTROL) 
	{
		
    } 
	else 
	{
		if((ep_cfg->ep_mps) > ES_USB_HOST_DEFAULT_FIFO_SIZE)
		{
			USB_LOG_ERR("please change usb_fifo");
		}
		else
		{
			
		}
    }
    /* restore variable */
    ppipe->inuse = true;
    ppipe->waitsem = waitsem;

    musb_set_active_ep(old_ep_index);
    *pipe = (usbh_pipe_t)ppipe;
    return 0;
}

int usbh_pipe_free(usbh_pipe_t pipe)
{
    struct musb_pipe *ppipe;
    struct usbh_urb *urb;

    ppipe = (struct musb_pipe *)pipe;

    if (!ppipe) {
        return -EINVAL;
    }

    urb = ppipe->urb;

    if (urb) {
        usbh_kill_urb(urb);
    }

    return 0;
}

int usbh_submit_urb(struct usbh_urb *urb)
{
    struct musb_pipe *pipe;
    size_t flags;
    int ret = 0;

    if (!urb) {
        return -EINVAL;
    }

    pipe = urb->pipe;

    if (!pipe) {
        return -EINVAL;
    }

    if (!pipe->hport->connected) {
        return -ENODEV;
    }

    if (pipe->urb) {
        return -EBUSY;
    }

    flags = usb_osal_enter_critical_section();

    pipe->waiter = false;
    pipe->xfrd = 0;
    pipe->urb = urb;
    urb->errorcode = -EBUSY;
    urb->actual_length = 0;

    if (urb->timeout > 0) {
        pipe->waiter = true;
    }
    usb_osal_leave_critical_section(flags);

    switch (pipe->ep_type) {
        case USB_ENDPOINT_TYPE_CONTROL:
            usb_ep0_state = USB_EP0_STATE_SETUP;
            musb_control_pipe_init(pipe, urb->setup, urb->transfer_buffer, urb->transfer_buffer_length);
            break;
        case USB_ENDPOINT_TYPE_BULK:
            musb_bulk_pipe_init(pipe, urb->transfer_buffer, urb->transfer_buffer_length);
            break;
        case USB_ENDPOINT_TYPE_INTERRUPT:
            musb_intr_pipe_init(pipe, urb->transfer_buffer, urb->transfer_buffer_length);
            break;
        case USB_ENDPOINT_TYPE_ISOCHRONOUS:
            musb_iso_pipe_init(pipe, urb->transfer_buffer, urb->transfer_buffer_length);            break;
        default:
            break;
    }
    if (urb->timeout > 0) {
        /* wait until timeout or sem give */
        ret = usb_osal_sem_take(pipe->waitsem, urb->timeout);
        if (ret < 0) {
            goto errout_timeout;
        }

        ret = urb->errorcode;
    }
    return ret;
errout_timeout:
    pipe->waiter = false;
    usbh_kill_urb(urb);
    return ret;
}

int usbh_kill_urb(struct usbh_urb *urb)
{
    struct musb_pipe *pipe;

    pipe = urb->pipe;

    if (!urb || !pipe) {
        return -EINVAL;
    }

    if (pipe->waiter) {
        pipe->waiter = false;
        urb->errorcode = -ESHUTDOWN;
        usb_osal_sem_give(pipe->waitsem);
    }

    return 0;
}

static inline void musb_pipe_waitup(struct musb_pipe *pipe)
{
    struct usbh_urb *urb;

    urb = pipe->urb;
    pipe->urb = NULL;

    if (pipe->waiter) {
        pipe->waiter = false;
        usb_osal_sem_give(pipe->waitsem);
    }

    if (urb->complete) {
        if (urb->errorcode < 0) {
            urb->complete(urb->arg, urb->errorcode);
        } else {
            urb->complete(urb->arg, urb->actual_length);
        }
    }
}
void handle_ep0(void)
{
    struct musb_pipe *pipe;
    struct usbh_urb *urb;
    uint32_t size;

    pipe = (struct musb_pipe *)&g_musb_hcd.pipe_pool[0][0];
    urb = pipe->urb;
    if (urb == NULL) {
        return;
    }

    musb_set_active_ep(0);
	
    if ((USB0->CSRL0) & USB_CSRL0_STALLED) 
	{
        USB0->CSRL0 &= ~USB_CSRL0_STALLED;
        usb_ep0_state = USB_EP0_STATE_SETUP;
        urb->errorcode = -EPERM;
        musb_pipe_waitup(pipe);
        return;
    }
    if ((USB0->CSRL0) & USB_CSRL0_ERROR) 
	{
        USB0->CSRL0 &= ~USB_CSRL0_ERROR;
		
        if ((USB0->CSRL0 & (USB_CSRL0_RXRDY | USB_CSRL0_TXRDY)) != 0)
            USB0->CSRH0 |= USB_CSRH0_FLUSH;
			
        usb_ep0_state = USB_EP0_STATE_SETUP;
        urb->errorcode = -EIO;
        musb_pipe_waitup(pipe);
        return;
    }

    switch (usb_ep0_state) {
        case USB_EP0_STATE_SETUP:
            urb->actual_length += 8;
            if (urb->transfer_buffer_length) {
                if (urb->setup->bmRequestType & 0x80) {
                    usb_ep0_state = USB_EP0_STATE_IN_DATA;
                    USB0->CSRL0 = USB_CSRL0_REQPKT;
                } else {
                    usb_ep0_state = USB_EP0_STATE_OUT_DATA;
                    size = urb->transfer_buffer_length;
                    if (size > pipe->ep_mps) {
                        size = pipe->ep_mps;
                    }

                    es_usbd_ep_write_packet_8bit(0, urb->transfer_buffer, size);
                    USB0->CSRL0 = USB_CSRL0_TXRDY;

                    urb->transfer_buffer += size;
                    urb->transfer_buffer_length -= size;
                    urb->actual_length += size;
                }
            } 
			else {
                usb_ep0_state = USB_EP0_STATE_IN_STATUS;
                USB0->CSRL0 = (USB_CSRL0_REQPKT | USB_CSRL0_STATUS);
            }
            break;
        case USB_EP0_STATE_IN_DATA:
            if ((USB0->CSRL0) & USB_CSRL0_RXRDY) 
			{
                size = urb->transfer_buffer_length;
                if (size > pipe->ep_mps) {
                    size = pipe->ep_mps;
				}

                size = MIN(size, USB0->COUNT0);
                es_usbd_ep_read_packet_8bit(0, urb->transfer_buffer, size);
                USB0->CSRL0 &= ~USB_CSRL0_RXRDY;
                urb->transfer_buffer += size;
                urb->transfer_buffer_length -= size;
                urb->actual_length += size;

                if ((size < pipe->ep_mps) || (urb->transfer_buffer_length == 0)) {
                    usb_ep0_state = USB_EP0_STATE_OUT_STATUS;
                    USB0->CSRL0 = (USB_CSRL0_TXRDY | USB_CSRL0_STATUS);
                } else {
                    USB0->CSRL0 = USB_CSRL0_REQPKT;
                }
            }
            break;
        case USB_EP0_STATE_OUT_DATA:
            if (urb->transfer_buffer_length > 0) {
                size = urb->transfer_buffer_length;
                if (size > pipe->ep_mps) {
                    size = pipe->ep_mps;
                }

                es_usbd_ep_write_packet_8bit(0, urb->transfer_buffer, size);
                USB0->CSRL0 |= USB_CSRL0_TXRDY;

                urb->transfer_buffer += size;
                urb->transfer_buffer_length -= size;
                urb->actual_length += size;
            } else {
                usb_ep0_state = USB_EP0_STATE_IN_STATUS;
				USB0->CSRL0 = (USB_CSRL0_REQPKT | USB_CSRL0_STATUS);
            }
            break;
        case USB_EP0_STATE_OUT_STATUS:
            urb->errorcode = 0;
            musb_pipe_waitup(pipe);
            break;
        case USB_EP0_STATE_IN_STATUS:
            if ((USB0->CSRL0) & (USB_CSRL0_RXRDY | USB_CSRL0_STATUS)) 
			{
                USB0->CSRL0 &= ~(USB_CSRL0_RXRDY | USB_CSRL0_STATUS);
                urb->errorcode = 0;
                musb_pipe_waitup(pipe);
            }
            break;
    }
}

const uint8_t __lowest_bit_bitmap[] =
{
    /* 00 */ 0, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 10 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 20 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 30 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 40 */ 6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 50 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 60 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 70 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 80 */ 7, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 90 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* A0 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* B0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* C0 */ 6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* D0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* E0 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* F0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0
};

void __attribute__((interrupt)) USB_Handler(void)
{
    uint8_t is, txis, rxis,old_ep_idx;
    uint8_t ep_idx;
    uint8_t status;
    struct musb_pipe *pipe;
    struct usbh_urb *urb;


    is   = USB0->IS;   /* read and clear interrupt status */
    txis = USB0->TXIS; /* read and clear interrupt status */
    rxis = USB0->RXIS; /* read and clear interrupt status */

	//printf("[%x %x %x]\r\n",is,txis,rxis);
    
    old_ep_idx = musb_get_active_ep();
	
    //is &= USB0->IE; /* Clear disabled interrupts */
    if (is & 0x20) 
    //if (is & USB_IS_DISCON)
    {
		//printf("discon\r\n");
        g_musb_hcd.port_csc = 1;
        g_musb_hcd.port_pec = 1;
        g_musb_hcd.port_pe = 0;
        usbh_roothub_thread_wakeup(1);
		
		//es_usbh_config_usb_perh();
    }
	
    if (is & 0x10)
    //if (is & USB_IS_CONN)
    {
		//printf("con\r\n");
        g_musb_hcd.port_csc = 1;
        g_musb_hcd.port_pec = 1;
        g_musb_hcd.port_pe = 1;
        usbh_roothub_thread_wakeup(1);
    }

 	txis &= USB0->TXIE; /* Clear disabled interrupts */
    if (txis & USB_TXIE_EP0)
	{
			do
			{
				handle_ep0();
			}
			while(USB0->CSRL0 & 0x1);
        txis &= 0xFE;
    }
	
    while (txis) {
        ep_idx = __lowest_bit_bitmap[txis];
		//printf("ep%d tx\r\n",ep_idx);
        //process_edpt_n(tu_edpt_addr(num, TUSB_DIR_IN));
        txis &= ~BIT(ep_idx);
        musb_set_active_ep(ep_idx);
        
        {
            pipe = &g_musb_hcd.pipe_pool[ep_idx][0];
            urb = pipe->urb;
			
			if ((USB0->TXCSRL) & USB_TXCSRL1_ERROR) 
			{
                USB0->TXCSRL &= ~USB_TXCSRL1_ERROR;
                urb->errorcode = -EIO;
            } 
			else if ((USB0->TXCSRL) & USB_TXCSRL1_NAKTO) 
			{
                USB0->TXCSRL &= ~USB_TXCSRL1_NAKTO;
                urb->errorcode = -EBUSY;
            } 
			else if ((USB0->TXCSRL) & USB_TXCSRL1_STALLED) 
			{
                USB0->TXCSRL &= ~USB_TXCSRL1_STALLED;
                urb->errorcode = -EPERM;
            }
			else
			{
                uint32_t size = urb->transfer_buffer_length;

                if (size > pipe->ep_mps) {
                    size = pipe->ep_mps;
                }

                urb->transfer_buffer += size;
                urb->transfer_buffer_length -= size;
                urb->actual_length += size;

                if (urb->transfer_buffer_length == 0) 
				{
                    urb->errorcode = 0;
                } 
				else 
				{
					es_usbd_ep_write_packet_8bit(ep_idx, urb->transfer_buffer, size);
                    USB0->TXCSRL |= USB_TXCSRL1_TXRDY;
                }
			}
			
			if(((urb->errorcode) != (-EBUSY))||((urb->transfer_buffer_length) == 0))
				musb_pipe_waitup(pipe);
        }
    }

     rxis &= USB0->RXIE; /* Clear disabled interrupts */
    while (rxis) {                              
        ep_idx = __lowest_bit_bitmap[rxis];  
        //USB0->RXICR = 1U << ep_idx;  
        rxis &= ~BIT(ep_idx);

        musb_set_active_ep(ep_idx);
		
		pipe = &g_musb_hcd.pipe_pool[ep_idx][1];
		urb = pipe->urb;
		
		if((USB0->RXCSRL) & USB_RXCSRL1_ERROR)
		{
			USB0->RXCSRL &= ~USB_RXCSRL1_ERROR;
			urb->errorcode = -EIO;
		}
		else if((USB0->RXCSRL) & USB_RXCSRL1_NAKTO)
		{
			USB0->RXCSRL &= ~USB_RXCSRL1_NAKTO;
			urb->errorcode = -EBUSY;
		}
		else if((USB0->RXCSRL) & USB_RXCSRL1_STALLED)
		{
			USB0->RXCSRL &= ~USB_RXCSRL1_STALLED;
			urb->errorcode = -EPERM;
		}
		else
		{
			uint32_t size = USB0->RXCOUNT;
			
			if (size > pipe->ep_mps)
				size = pipe->ep_mps;
				
			size = MIN(size, (urb->transfer_buffer_length));
			
			es_usbd_ep_read_packet_8bit(ep_idx, urb->transfer_buffer, size);
			
			USB0->RXCSRL &= ~( USB_RXCSRL1_ERROR | USB_RXCSRL1_STALLED);
			USB0->RXCSRL &= ~USB_RXCSRL1_RXRDY;
			
			urb->transfer_buffer += size;
			urb->transfer_buffer_length -= size;
			urb->actual_length += size;
			
			if ((size < pipe->ep_mps) || ((urb->transfer_buffer_length) < (pipe->ep_mps))) 
				urb->errorcode = 0;
			else 
				USB0->RXCSRL |= USB_RXCSRL1_REQPKT;
		}
		
		if(((urb->errorcode) != (-EBUSY))||((urb->transfer_buffer_length) < (pipe->ep_mps)))
			musb_pipe_waitup(pipe);
    }

    musb_set_active_ep(old_ep_idx);
}
