#include "usbh_core.h"
#include "platform.h"
#include "reg_sysc_cpu.h"
#include "musb_type.h"
#include "soc_pinmux.h"
#include "sys_stat.h"

void USB_Handler(void);

#define ES_USB_PERH_EP_MAX_INDEX     (6U)
#define ES_USB_HOST_DEFAULT_FIFO_SIZE (1024U)
#define ES_USB_HOST_FIFO_STATIC_USED  (ES_USB_PERH_EP_MAX_INDEX * ES_USB_HOST_DEFAULT_FIFO_SIZE)

#define musb_get_active_ep()   (USB0->EPIDX)
#define musb_set_active_ep(XXX)    ((USB0->EPIDX) = (XXX))

typedef enum {
    USB_EP0_STATE_SETUP = 0x0,      /**< SETUP DATA */
    USB_EP0_STATE_IN_DATA = 0x1,    /**< IN DATA */
    USB_EP0_STATE_OUT_DATA = 0x3,   /**< OUT DATA */
    USB_EP0_STATE_IN_STATUS = 0x4,  /**< IN status */
    USB_EP0_STATE_OUT_STATUS = 0x5, /**< OUT status */
} ep0_state_t;

struct musb_pipe {
    uint8_t chidx;
    bool inuse;
    uint32_t xfrd;
    volatile uint8_t ep0_state;
    usb_osal_sem_t waitsem;
    struct usbh_urb *urb;
};

struct musb_hcd {
    volatile bool port_csc;
    volatile bool port_pec;
    volatile bool port_pe;
    struct musb_pipe pipe_pool[CONFIG_USBHOST_PIPE_NUM];
} g_musb_hcd[CONFIG_USBHOST_MAX_BUS];

typedef union {
    uint8_t   u8;
    uint16_t  u16;
    uint32_t  u32;
} hw_fifo_t;
static volatile uint8_t usb_ep0_state = USB_EP0_STATE_SETUP;

static void es_usbd_ep_write_packet_8bit(uint8_t ep, void *buf, unsigned len)
{
    volatile hw_fifo_t *reg = (volatile hw_fifo_t*)((&USB0->FIFO0_WORD) + ep);
    uintptr_t addr = (uintptr_t)buf;
    while (len >= 4) {
        reg->u32 = *(uint32_t const *)addr;
        addr += 4;
        len  -= 4;
    }
    if (len >= 2) {
        reg->u16 = *(uint16_t const *)addr;
        addr += 2;
        len  -= 2;
    }
    if (len) {
        reg->u8 = *(uint8_t const *)addr;
    }
}

static void es_usbd_ep_read_packet_8bit(uint8_t ep, void *buf, unsigned len)
{
    volatile hw_fifo_t *reg = (volatile hw_fifo_t*)((&USB0->FIFO0_WORD) + ep);
    uintptr_t addr = (uintptr_t)buf;
    while (len >= 4) {
        *(uint32_t *)addr = reg->u32;
        addr += 4;
        len  -= 4;
    }
    if (len >= 2) {
        *(uint16_t *)addr = reg->u16;
        addr += 2;
        len  -= 2;
    }
    if (len) {
        *(uint8_t *)addr = reg->u8;
    }
}

void musb_control_urb_init(struct usbh_bus *bus, uint8_t chidx, struct usbh_urb *urb, struct usb_setup_packet *setup, uint8_t *buffer, uint32_t buflen)
{
    uint8_t old_ep_index;

    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(0);
	
    USB0->EPIDX = 0;
    USB0->FADDR = urb->hport->dev_addr;
	
	//ald_usb_host_ep_config(0, 64, 0, 0, (ALD_USB_EP_MODE_CTRL | ALD_USB_EP_HOST_OUT));

    es_usbd_ep_write_packet_8bit(0, (uint8_t *)setup, 8);
    USB0->CSRL0 = (USB_CSRL0_TXRDY | USB_CSRL0_SETUP);
    musb_set_active_ep(old_ep_index);
}

void musb_bulk_urb_init(struct usbh_bus *bus, uint8_t chidx, struct usbh_urb *urb, uint8_t *buffer, uint32_t buflen)
{
    uint8_t ep_idx;
    uint8_t old_ep_index;

    ep_idx = chidx & 0x7f;
    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
	
    USB0->EPIDX = ep_idx;
    USB0->FADDR = urb->hport->dev_addr;
	
    if (urb->ep->bEndpointAddress & 0x80) 
	{
            USB0->RXTYPE = (uint8_t)USB_RXTYPE1_PROTO_BULK | (0xF & ep_idx);
            USB0->RXINTERVAL = (uint8_t)urb->ep->bInterval;
            USB0->RXMAXP = (uint8_t)urb->ep->wMaxPacketSize;
			
        USB0->RXCSRL |= USB_RXCSRL1_REQPKT;
    } 
	else 
	{
            USB0->TXTYPE = (uint8_t)USB_TXTYPE1_PROTO_BULK | (0xF & ep_idx);
            USB0->TXINTERVAL = (uint8_t)urb->ep->bInterval;
            USB0->TXMAXP = (uint8_t)urb->ep->wMaxPacketSize;

        if (buflen > urb->ep->wMaxPacketSize)
            buflen = urb->ep->wMaxPacketSize;

        es_usbd_ep_write_packet_8bit(ep_idx, buffer, buflen);
		USB0->TXCSRL = USB_TXCSRL1_TXRDY;
    }
	
    musb_set_active_ep(old_ep_index);
}

void musb_intr_urb_init(struct usbh_bus *bus, uint8_t chidx, struct usbh_urb *urb, uint8_t *buffer, uint32_t buflen)
{
    uint8_t ep_idx;
    uint8_t old_ep_index;

    ep_idx = chidx & 0x7f;
    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
	
    USB0->EPIDX = ep_idx;
    USB0->FADDR = urb->hport->dev_addr;
	
    if (urb->ep->bEndpointAddress & 0x80) 
	{
            USB0->RXTYPE = (uint8_t)USB_RXTYPE1_PROTO_INT | (0xF & ep_idx);
            USB0->RXINTERVAL = (uint8_t)urb->ep->bInterval;
            USB0->RXMAXP = (uint8_t)urb->ep->wMaxPacketSize;
			
        USB0->RXCSRL |= USB_RXCSRL1_REQPKT;
    } 
	else 
	{
            USB0->TXTYPE = (uint8_t)USB_TXTYPE1_PROTO_INT | (0xF & ep_idx);
            USB0->TXINTERVAL = (uint8_t)urb->ep->bInterval;
            USB0->TXMAXP = (uint8_t)urb->ep->wMaxPacketSize;

        if (buflen > urb->ep->wMaxPacketSize)
            buflen = urb->ep->wMaxPacketSize;

        es_usbd_ep_write_packet_8bit(ep_idx, buffer, buflen);
        USB0->TXCSRL = USB_TXCSRL1_TXRDY;
    }
    
    musb_set_active_ep(old_ep_index);
}

void musb_iso_urb_init(struct usbh_bus *bus, uint8_t chidx, struct usbh_urb *urb, uint8_t *buffer, uint32_t buflen)
{
    uint8_t ep_idx;
    uint8_t old_ep_index;

    ep_idx = chidx & 0x7f;
    old_ep_index = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
	
    USB0->EPIDX = ep_idx;
    USB0->FADDR = urb->hport->dev_addr;
	
    if (urb->ep->bEndpointAddress & 0x80) 
	{
            USB0->RXTYPE = (uint8_t)USB_RXTYPE1_PROTO_ISOC | (0xF & ep_idx);
            USB0->RXINTERVAL = (uint8_t)urb->ep->bInterval;
            USB0->RXMAXP = (uint8_t)urb->ep->wMaxPacketSize;
			
        USB0->RXCSRL |= USB_RXCSRL1_REQPKT;
    } 
	else 
	{
            USB0->TXTYPE = (uint8_t)USB_TXTYPE1_PROTO_ISOC | (0xF & ep_idx);
            USB0->TXINTERVAL = (uint8_t)urb->ep->bInterval;
            USB0->TXMAXP = (uint8_t)urb->ep->wMaxPacketSize;

        if (buflen > urb->ep->wMaxPacketSize)
            buflen = urb->ep->wMaxPacketSize;

        es_usbd_ep_write_packet_8bit(ep_idx, buffer, buflen);
        USB0->TXCSRL = USB_TXCSRL1_TXRDY;
    }
    
    musb_set_active_ep(old_ep_index);
}

static int usbh_reset_port(struct usbh_bus *bus, const uint8_t port)
{
    g_musb_hcd[bus->hcd.hcd_id].port_pe = 0;
	USB0->POWER |= USB_POWER_RESET;
    usb_osal_msleep(20);
	USB0->POWER &= ~USB_POWER_RESET;
    usb_osal_msleep(20);
    g_musb_hcd[bus->hcd.hcd_id].port_pe = 1;
    return 0;
}
/*
static uint8_t usbh_get_port_speed(const uint8_t port)
{
    uint8_t speed = USB_SPEED_UNKNOWN;
	
	if (USB0->DEVCTL & USB_DEVCTL_FSDEV)
		return USB_SPEED_FULL;

	if (USB0->DEVCTL & USB_DEVCTL_LSDEV)
		return USB_SPEED_LOW;

	return speed;
}
*/
static void musb_pipe_free(struct musb_pipe *pipe)
{
#if 0
    pipe->inuse = false;
#endif
}

__WEAK void usb_hc_low_level_init(struct usbh_bus *bus)
{
    (void)bus;
}

__WEAK void usb_hc_low_level_deinit(struct usbh_bus *bus)
{
    (void)bus;
}

int usb_hc_init(struct usbh_bus *bus)
{
    memset(&g_musb_hcd[bus->hcd.hcd_id], 0, sizeof(struct musb_hcd));

    for (uint8_t i = 0; i < CONFIG_USBHOST_PIPE_NUM; i++) {
        g_musb_hcd[bus->hcd.hcd_id].pipe_pool[i].waitsem = usb_osal_sem_create(0);
    }

    usb_hc_low_level_init(bus);

	csi_vic_disable_irq(USB_IRQn);
	CLIC->CLICINT[USB_IRQn].ATTR |= (1);
	pinmux_usb_init(1);
	
    rv_set_int_isr(USB_IRQn, USB_Handler);
    csi_vic_clear_pending_irq(USB_IRQn);
    csi_vic_enable_irq(USB_IRQn);
	
	
    SYSC_CPU->PD_CPU_CLKG = SYSC_CPU_CLKG_CLR_USB_MASK;
    SYSC_CPU->PD_CPU_SRST = SYSC_CPU_SRST_CLR_USB_N_MASK;
    SYSC_CPU->PD_CPU_SRST = SYSC_CPU_SRST_SET_USB_N_MASK;
	
	
	csi_vic_enable_irq(USB_IRQn);
    SYSC_CPU->PD_CPU_CLKG = SYSC_CPU_CLKG_SET_USB_MASK;
	
    usb_status_set(true);
	
    //USB0->POWER |= 0x1;
    //USB0->IE = 0xFF;
    USB0->IE = 0xF7;
    USB0->TXIE = 0x7F;
    USB0->RXIE = 0x7E;
	
	
    //USB0->IS = 0xFF;
    //USB0->TXIS = 0x7F;
    //USB0->RXIS = 0x7E;

    //dcd_connect();
	
	//USB0->DEVCTL &= 0xFE;
	
	for(uint32_t i = 0; i <= ES_USB_PERH_EP_MAX_INDEX; i++)
	{
        USB0->EPIDX     = i;
        USB0->TXFIFO_SIZE[0] = 8;
        USB0->TXFIFO_SIZE[1] = 3 << USB_TX_FIFO_SIZE_POS;
        USB0->RXFIFO_SIZE[0] = 8;
        USB0->RXFIFO_SIZE[1] = 3 << USB_RX_FIFO_SIZE_POS;
	}
	
	USB0->EPIDX = 0;
	
    // unsigned cnt;
	//cnt = 2880000;
    // while (cnt--) __NOP();
	
	USB0->DEVCTL |= 0x2;
	
	//cnt = 288000;
    // while (cnt--) __NOP();
	 
	USB0->DEVCTL |= 0x1;
	
	//cnt = 288000;
    // while (cnt--) __NOP();
	 
	//usbh_reset_port(0);
	
    return 0;
}

int usb_hc_deinit(struct usbh_bus *bus)
{
    return 0;
}

int usbh_roothub_control(struct usbh_bus *bus, struct usb_setup_packet *setup, uint8_t *buf)
{
    uint8_t nports;
    uint8_t port;
    uint32_t status;

    nports = CONFIG_USBHOST_MAX_RHPORTS;
    port = setup->wIndex;
    if (setup->bmRequestType & USB_REQUEST_RECIPIENT_DEVICE) {
        switch (setup->bRequest) {
            case HUB_REQUEST_CLEAR_FEATURE:
                switch (setup->wValue) {
                    case HUB_FEATURE_HUB_C_LOCALPOWER:
                        break;
                    case HUB_FEATURE_HUB_C_OVERCURRENT:
                        break;
                    default:
                        return -USB_ERR_INVAL;
                }
                break;
            case HUB_REQUEST_SET_FEATURE:
                switch (setup->wValue) {
                    case HUB_FEATURE_HUB_C_LOCALPOWER:
                        break;
                    case HUB_FEATURE_HUB_C_OVERCURRENT:
                        break;
                    default:
                        return -USB_ERR_INVAL;
                }
                break;
            case HUB_REQUEST_GET_DESCRIPTOR:
                break;
            case HUB_REQUEST_GET_STATUS:
                memset(buf, 0, 4);
                break;
            default:
                break;
        }
    } else if (setup->bmRequestType & USB_REQUEST_RECIPIENT_OTHER) {
        switch (setup->bRequest) {
            case HUB_REQUEST_CLEAR_FEATURE:
                if (!port || port > nports) {
                    return -USB_ERR_INVAL;
                }

                switch (setup->wValue) {
                    case HUB_PORT_FEATURE_ENABLE:
                        break;
                    case HUB_PORT_FEATURE_SUSPEND:
                    case HUB_PORT_FEATURE_C_SUSPEND:
                        break;
                    case HUB_PORT_FEATURE_POWER:
                        break;
                    case HUB_PORT_FEATURE_C_CONNECTION:
                        g_musb_hcd[bus->hcd.hcd_id].port_csc = 0;
                        break;
                    case HUB_PORT_FEATURE_C_ENABLE:
                        g_musb_hcd[bus->hcd.hcd_id].port_pec = 0;
                        break;
                    case HUB_PORT_FEATURE_C_OVER_CURREN:
                        break;
                    case HUB_PORT_FEATURE_C_RESET:
                        break;
                    default:
                        return -USB_ERR_INVAL;
                }
                break;
            case HUB_REQUEST_SET_FEATURE:
                if (!port || port > nports) {
                    return -USB_ERR_INVAL;
                }

                switch (setup->wValue) {
                    case HUB_PORT_FEATURE_SUSPEND:
                        break;
                    case HUB_PORT_FEATURE_POWER:
                        break;
                    case HUB_PORT_FEATURE_RESET:
                        usbh_reset_port(bus, port);
                        break;

                    default:
                        return -USB_ERR_INVAL;
                }
                break;
            case HUB_REQUEST_GET_STATUS:
                if (!port || port > nports) {
                    return -USB_ERR_INVAL;
                }

                status = 0;
                if (g_musb_hcd[bus->hcd.hcd_id].port_csc) {
                    status |= (1 << HUB_PORT_FEATURE_C_CONNECTION);
                }
                if (g_musb_hcd[bus->hcd.hcd_id].port_pec) {
                    status |= (1 << HUB_PORT_FEATURE_C_ENABLE);
                }

                if (g_musb_hcd[bus->hcd.hcd_id].port_pe) {
                    status |= (1 << HUB_PORT_FEATURE_CONNECTION);
                    status |= (1 << HUB_PORT_FEATURE_ENABLE);
                }

                status |= (1 << HUB_PORT_FEATURE_POWER);
                memcpy(buf, &status, 4);
                break;
            default:
                break;
        }
    }
    return 0;
}

int usbh_submit_urb(struct usbh_urb *urb)
{
    struct musb_pipe *pipe;
    struct usbh_bus *bus;
    int chidx;
    size_t flags;
    int ret = 0;

    if (!urb || !urb->hport || !urb->ep || !urb->hport->bus) {
        return -USB_ERR_INVAL;
    }

    if (!urb->hport->connected) {
        return -USB_ERR_NOTCONN;
    }

    if (urb->errorcode == -USB_ERR_BUSY) {
        return -USB_ERR_BUSY;
    }

    bus = urb->hport->bus;

    if (USB_GET_ENDPOINT_TYPE(urb->ep->bmAttributes) == USB_ENDPOINT_TYPE_CONTROL) {
        chidx = 0;
    } else {
        chidx = (urb->ep->bEndpointAddress & 0x0f);

        if (chidx > (CONFIG_USBHOST_PIPE_NUM - 1)) {
            return -USB_ERR_RANGE;
        }
    }

    flags = usb_osal_enter_critical_section();

    pipe = &g_musb_hcd[bus->hcd.hcd_id].pipe_pool[chidx];
    pipe->chidx = chidx;
    pipe->urb = urb;

    urb->hcpriv = pipe;
    urb->errorcode = -USB_ERR_BUSY;
    urb->actual_length = 0;

    switch (USB_GET_ENDPOINT_TYPE(urb->ep->bmAttributes)) {
        case USB_ENDPOINT_TYPE_CONTROL:
            pipe->ep0_state = USB_EP0_STATE_SETUP;
            musb_control_urb_init(bus, 0, urb, urb->setup, urb->transfer_buffer, urb->transfer_buffer_length);
            break;
        case USB_ENDPOINT_TYPE_BULK:
            musb_bulk_urb_init(bus, chidx, urb, urb->transfer_buffer, urb->transfer_buffer_length);
            break;
        case USB_ENDPOINT_TYPE_INTERRUPT:
            musb_intr_urb_init(bus, chidx, urb, urb->transfer_buffer, urb->transfer_buffer_length);
            break;
        case USB_ENDPOINT_TYPE_ISOCHRONOUS:
            musb_iso_urb_init(bus, chidx, urb, urb->transfer_buffer, urb->transfer_buffer_length);
            break;
        default:
            break;
    }
    usb_osal_leave_critical_section(flags);

    if (urb->timeout > 0) {
        /* wait until timeout or sem give */
        ret = usb_osal_sem_take(pipe->waitsem, urb->timeout);
        if (ret < 0) {
            goto errout_timeout;
        }
        urb->timeout = 0;
        ret = urb->errorcode;
        /* we can free pipe when waitsem is done */
        musb_pipe_free(pipe);
    }
    return ret;
errout_timeout:
    urb->timeout = 0;
    usbh_kill_urb(urb);
    return ret;
}

int usbh_kill_urb(struct usbh_urb *urb)
{
    struct musb_pipe *pipe;
    struct usbh_bus *bus;
    size_t flags;

    if (!urb || !urb->hcpriv || !urb->hport->bus) {
        return -USB_ERR_INVAL;
    }

    bus = urb->hport->bus;

    ARG_UNUSED(bus);

    flags = usb_osal_enter_critical_section();

    pipe = (struct musb_pipe *)urb->hcpriv;
    urb->hcpriv = NULL;
    urb->errorcode = -USB_ERR_SHUTDOWN;
    pipe->urb = NULL;

    if (urb->timeout) {
        usb_osal_sem_give(pipe->waitsem);
    } else {
        musb_pipe_free(pipe);
    }

    usb_osal_leave_critical_section(flags);
    return 0;
}

static void musb_urb_waitup(struct usbh_urb *urb)
{
    struct musb_pipe *pipe;

    pipe = (struct musb_pipe *)urb->hcpriv;
    pipe->urb = NULL;
    urb->hcpriv = NULL;

    if (urb->timeout) {
        usb_osal_sem_give(pipe->waitsem);
    } else {
        musb_pipe_free(pipe);
    }

    if (urb->complete) {
        if (urb->errorcode < 0) {
            urb->complete(urb->arg, urb->errorcode);
        } else {
            urb->complete(urb->arg, urb->actual_length);
        }
    }
}

void handle_ep0(struct usbh_bus *bus)
{
    //uint8_t ep0_status;
    struct musb_pipe *pipe;
    struct usbh_urb *urb;
    uint32_t size;

    pipe = (struct musb_pipe *)&g_musb_hcd[bus->hcd.hcd_id].pipe_pool[0];
    urb = pipe->urb;
    if (urb == NULL) {
        return;
    }

    musb_set_active_ep(0);
	
    if ((USB0->CSRL0) & USB_CSRL0_STALLED) 
	{
        USB0->CSRL0 &= ~USB_CSRL0_STALLED;
        usb_ep0_state = USB_EP0_STATE_SETUP;
        urb->errorcode = -USB_ERR_STALL;
        musb_urb_waitup(urb);
        return;
    }
    if ((USB0->CSRL0) & USB_CSRL0_ERROR) 
	{
        USB0->CSRL0 &= ~USB_CSRL0_ERROR;
		
        if ((USB0->CSRL0 & (USB_CSRL0_RXRDY | USB_CSRL0_TXRDY)) != 0)
            USB0->CSRH0 |= USB_CSRH0_FLUSH;
			
        pipe->ep0_state = USB_EP0_STATE_SETUP;
        urb->errorcode = -USB_ERR_IO;
        musb_urb_waitup(urb);
        return;
    }

    switch (pipe->ep0_state) {
        case USB_EP0_STATE_SETUP:
            urb->actual_length += 8;
            if (urb->transfer_buffer_length) {
                if (urb->setup->bmRequestType & 0x80) {
                    pipe->ep0_state = USB_EP0_STATE_IN_DATA;
                    USB0->CSRL0 = USB_CSRL0_REQPKT;
                } else {
                    pipe->ep0_state = USB_EP0_STATE_OUT_DATA;
                    size = urb->transfer_buffer_length;
                    if (size > urb->ep->wMaxPacketSize) {
                        size = urb->ep->wMaxPacketSize;
                    }

                    es_usbd_ep_write_packet_8bit(0, urb->transfer_buffer, size);
                    USB0->CSRL0 = USB_CSRL0_TXRDY;

                    urb->transfer_buffer += size;
                    urb->transfer_buffer_length -= size;
                    urb->actual_length += size;
                }
            } 
			else {
                pipe->ep0_state = USB_EP0_STATE_IN_STATUS;
                USB0->CSRL0 = (USB_CSRL0_REQPKT | USB_CSRL0_STATUS);
            }
            break;
        case USB_EP0_STATE_IN_DATA:
            if ((USB0->CSRL0) & USB_CSRL0_RXRDY) 
			{
                size = urb->transfer_buffer_length;
                if (size > urb->ep->wMaxPacketSize) {
                    size = urb->ep->wMaxPacketSize;
				}

                size = MIN(size, USB0->COUNT0);
                es_usbd_ep_read_packet_8bit(0, urb->transfer_buffer, size);
                USB0->CSRL0 &= ~USB_CSRL0_RXRDY;
                urb->transfer_buffer += size;
                urb->transfer_buffer_length -= size;
                urb->actual_length += size;

                if ((size < urb->ep->wMaxPacketSize) || (urb->transfer_buffer_length == 0)) {
                    pipe->ep0_state = USB_EP0_STATE_OUT_STATUS;
                    USB0->CSRL0 = (USB_CSRL0_TXRDY | USB_CSRL0_STATUS);
                } else {
                    USB0->CSRL0 = USB_CSRL0_REQPKT;
                }
            }
            break;
        case USB_EP0_STATE_OUT_DATA:
            if (urb->transfer_buffer_length > 0) {
                size = urb->transfer_buffer_length;
                if (size > urb->ep->wMaxPacketSize) {
                    size = urb->ep->wMaxPacketSize;
                }

                es_usbd_ep_write_packet_8bit(0, urb->transfer_buffer, size);
                USB0->CSRL0 |= USB_CSRL0_TXRDY;

                urb->transfer_buffer += size;
                urb->transfer_buffer_length -= size;
                urb->actual_length += size;
            } else {
                pipe->ep0_state = USB_EP0_STATE_IN_STATUS;
				USB0->CSRL0 = (USB_CSRL0_REQPKT | USB_CSRL0_STATUS);
            }
            break;
        case USB_EP0_STATE_OUT_STATUS:
            urb->errorcode = 0;
            musb_urb_waitup(urb);
            break;
        case USB_EP0_STATE_IN_STATUS:
            if ((USB0->CSRL0) & (USB_CSRL0_RXRDY | USB_CSRL0_STATUS)) 
			{
                USB0->CSRL0 &= ~(USB_CSRL0_RXRDY | USB_CSRL0_STATUS);
                urb->errorcode = 0;
                musb_urb_waitup(urb);
            }
            break;
    }
}

const uint8_t __lowest_bit_bitmap[] =
{
    /* 00 */ 0, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 10 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 20 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 30 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 40 */ 6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 50 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 60 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 70 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 80 */ 7, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 90 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* A0 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* B0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* C0 */ 6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* D0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* E0 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* F0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0
};

void __attribute__((interrupt)) USB_Handler(void)
{
    uint8_t is, txis, rxis,old_ep_idx;
    uint8_t ep_idx;
    //uint8_t status;
    struct musb_pipe *pipe;
    struct usbh_urb *urb;
    struct usbh_bus *bus;

    bus = &g_usbhost_bus[0];

    is   = USB0->IS;   /* read and clear interrupt status */
    txis = USB0->TXIS; /* read and clear interrupt status */
    rxis = USB0->RXIS; /* read and clear interrupt status */

	//printf("[%x %x %x]\r\n",is,txis,rxis);
    
    old_ep_idx = musb_get_active_ep();
	
    //is &= USB0->IE; /* Clear disabled interrupts */
    if (is & 0x20) 
    //if (is & USB_IS_DISCON)
    {        
		g_musb_hcd[bus->hcd.hcd_id].port_csc = 1;
        g_musb_hcd[bus->hcd.hcd_id].port_pec = 1;
        g_musb_hcd[bus->hcd.hcd_id].port_pe = 0;
        bus->hcd.roothub.int_buffer[0] = (1 << 1);
        usbh_hub_thread_wakeup(&bus->hcd.roothub);
		
		//es_usbh_config_usb_perh();
    }
	
    if (is & 0x10)
    //if (is & USB_IS_CONN)
    {
        g_musb_hcd[bus->hcd.hcd_id].port_csc = 1;
        g_musb_hcd[bus->hcd.hcd_id].port_pec = 1;
        g_musb_hcd[bus->hcd.hcd_id].port_pe = 1;
        bus->hcd.roothub.int_buffer[0] = (1 << 1);
        usbh_hub_thread_wakeup(&bus->hcd.roothub);
    }

 	txis &= USB0->TXIE; /* Clear disabled interrupts */
    if (txis & USB_TXIE_EP0)
	{
			do
			{
				handle_ep0(bus);
			}
			while(USB0->CSRL0 & 0x1);
        txis &= 0xFE;
    }
	
    while (txis) {
        ep_idx = __lowest_bit_bitmap[txis];
		//printf("ep%d tx\r\n",ep_idx);
        //process_edpt_n(tu_edpt_addr(num, TUSB_DIR_IN));
        txis &= ~BIT(ep_idx);
        musb_set_active_ep(ep_idx);
        
        {
            pipe = &g_musb_hcd[bus->hcd.hcd_id].pipe_pool[ep_idx];
            urb = pipe->urb;
			
			if ((USB0->TXCSRL) & USB_TXCSRL1_ERROR) 
			{
                USB0->TXCSRL &= ~USB_TXCSRL1_ERROR;
                urb->errorcode = -USB_ERR_IO;
            } 
			else if ((USB0->TXCSRL) & USB_TXCSRL1_NAKTO) 
			{
                USB0->TXCSRL &= ~USB_TXCSRL1_NAKTO;
                urb->errorcode = -USB_ERR_NAK;
            } 
			else if ((USB0->TXCSRL) & USB_TXCSRL1_STALLED) 
			{
                USB0->TXCSRL &= ~USB_TXCSRL1_STALLED;
                urb->errorcode = -USB_ERR_STALL;
            }
			else
			{
                uint32_t size = urb->transfer_buffer_length;

                if (size > urb->ep->wMaxPacketSize) {
                    size = urb->ep->wMaxPacketSize;
                }

                urb->transfer_buffer += size;
                urb->transfer_buffer_length -= size;
                urb->actual_length += size;

                if (urb->transfer_buffer_length == 0) 
				{
                    urb->errorcode = 0;
                } 
				else 
				{
					es_usbd_ep_write_packet_8bit(ep_idx, urb->transfer_buffer, size);
                    USB0->TXCSRL |= USB_TXCSRL1_TXRDY;
                }
			}
			
			if(((urb->errorcode) != (-USB_ERR_BUSY))||((urb->transfer_buffer_length) == 0))
				musb_urb_waitup(urb);
        }
    }

     rxis &= USB0->RXIE; /* Clear disabled interrupts */
    while (rxis) {                              
        ep_idx = __lowest_bit_bitmap[rxis];  
        //USB0->RXICR = 1U << ep_idx;  
        rxis &= ~BIT(ep_idx);

        musb_set_active_ep(ep_idx);
		
            pipe = &g_musb_hcd[bus->hcd.hcd_id].pipe_pool[ep_idx];
		urb = pipe->urb;
		
		if((USB0->RXCSRL) & USB_RXCSRL1_ERROR)
		{
			USB0->RXCSRL &= ~USB_RXCSRL1_ERROR;
			urb->errorcode = -USB_ERR_IO;
		}
		else if((USB0->RXCSRL) & USB_RXCSRL1_NAKTO)
		{
			USB0->RXCSRL &= ~USB_RXCSRL1_NAKTO;
			urb->errorcode = -USB_ERR_NAK;
		}
		else if((USB0->RXCSRL) & USB_RXCSRL1_STALLED)
		{
			USB0->RXCSRL &= ~USB_RXCSRL1_STALLED;
			urb->errorcode = -USB_ERR_STALL;
		}
		else
		{
			uint32_t size = USB0->RXCOUNT;
			
			if (size > urb->ep->wMaxPacketSize)
				size = urb->ep->wMaxPacketSize;
				
			size = MIN(size, (urb->transfer_buffer_length));
			
			es_usbd_ep_read_packet_8bit(ep_idx, urb->transfer_buffer, size);
			
			USB0->RXCSRL &= ~( USB_RXCSRL1_ERROR | USB_RXCSRL1_STALLED);
			USB0->RXCSRL &= ~USB_RXCSRL1_RXRDY;
			
			urb->transfer_buffer += size;
			urb->transfer_buffer_length -= size;
			urb->actual_length += size;
			
			if ((size < urb->ep->wMaxPacketSize) || ((urb->transfer_buffer_length) < (urb->ep->wMaxPacketSize))) 
				urb->errorcode = 0;
			else 
				USB0->RXCSRL |= USB_RXCSRL1_REQPKT;
		}
		
		if(((urb->errorcode) != (-USB_ERR_BUSY))||((urb->transfer_buffer_length) < (urb->ep->wMaxPacketSize)))
			musb_urb_waitup(urb);
    }

    musb_set_active_ep(old_ep_idx);
}
