#include "usbd_core.h"
#include "platform.h"
#include "reg_sysc_cpu.h"
#include "musb_type.h"
#include "soc_pinmux.h"
#include "sys_stat.h"

void USB_Handler(void);

#define ES_USB_PERH_EP_MAX_INDEX     (6U)

#define musb_get_active_ep()   (USB0->EPIDX)
#define musb_set_active_ep(XXX)    ((USB0->EPIDX) = (XXX))

typedef enum {
    USB_EP0_STATE_SETUP = 0x0,      /**< SETUP DATA */
    USB_EP0_STATE_IN_DATA = 0x1,    /**< IN DATA */
    USB_EP0_STATE_OUT_DATA = 0x3,   /**< OUT DATA */
    USB_EP0_STATE_IN_STATUS = 0x4,  /**< IN status */
    USB_EP0_STATE_OUT_STATUS = 0x5, /**< OUT status */
    USB_EP0_STATE_IN_ZLP = 0x6,     /**< OUT status */
    USB_EP0_STATE_STALL = 0x7,      /**< STALL status */
} ep0_state_t;

/* Endpoint state */
struct musb_ep_state {
    uint16_t ep_mps;    /* Endpoint max packet size */
    uint8_t ep_type;    /* Endpoint type */
    uint8_t ep_stalled; /* Endpoint stall flag */
    uint8_t ep_enable;  /* Endpoint enable */
    uint8_t *xfer_buf;
    uint32_t xfer_len;
    uint32_t actual_xfer_len;
};

/* Driver state */
struct musb_udc {
    volatile uint8_t dev_addr;
    volatile uint32_t fifo_size_offset;
    __attribute__((aligned(32))) struct usb_setup_packet setup;
    struct musb_ep_state in_ep[ES_USB_PERH_EP_MAX_INDEX + 1];  /*!< IN endpoint parameters*/
    struct musb_ep_state out_ep[ES_USB_PERH_EP_MAX_INDEX + 1]; /*!< OUT endpoint parameters */
} g_musb_udc;

typedef union {
    uint8_t   u8;
    uint16_t  u16;
    uint32_t  u32;
} hw_fifo_t;

static volatile uint8_t usb_ep0_state = USB_EP0_STATE_SETUP;
volatile bool zlp_flag = 0;

__WEAK void usb_dc_low_level_init(void)
{
}

__WEAK void usb_dc_low_level_deinit(void)
{
}

int usb_dc_init(uint8_t busid)
{
	csi_vic_disable_irq(USB_IRQn);
	CLIC->CLICINT[USB_IRQn].ATTR |= (1);
	pinmux_usb_init(0);
	
    rv_set_int_isr(USB_IRQn, USB_Handler);
    csi_vic_clear_pending_irq(USB_IRQn);
	/*csi_vic_set_prio(USB_IRQn, 0xF);
	csi_vic_set_prio(FLASH_SWINT_NUM, 0xF);
    csi_vic_enable_irq(USB_IRQn);*/
	
	
    SYSC_CPU->PD_CPU_CLKG = SYSC_CPU_CLKG_CLR_USB_MASK;
    SYSC_CPU->PD_CPU_SRST = SYSC_CPU_SRST_CLR_USB_N_MASK;
    SYSC_CPU->PD_CPU_SRST = SYSC_CPU_SRST_SET_USB_N_MASK;
	
	
#ifndef ES_USB_Handler_NOT_IN_IRQ
	csi_vic_enable_irq(USB_IRQn);
#endif

    SYSC_CPU->PD_CPU_CLKG = SYSC_CPU_CLKG_SET_USB_MASK;
	
    usb_status_set(true);
	
    USB0->POWER |= 0x1;
	
#ifndef ES_USB_Handler_NOT_IN_IRQ
    USB0->IE = 0xF7;
    USB0->TXIE = 0x7F;
    USB0->RXIE = 0x7E;
#endif
	
	USB0->DEVCTL = 0;
	
	for(uint32_t i = 0; i <= ES_USB_PERH_EP_MAX_INDEX; i++)
	{
        USB0->EPIDX     = i;
        USB0->TXFIFO_SIZE[0] = 8;
        USB0->TXFIFO_SIZE[1] = 3 << USB_TX_FIFO_SIZE_POS;
        USB0->RXFIFO_SIZE[0] = 8;
        USB0->RXFIFO_SIZE[1] = 3 << USB_RX_FIFO_SIZE_POS;
	}
	
	USB0->EPIDX = 0;
	
     unsigned cnt = 2880000;
     while (cnt--) __NOP();
	
	USB0->DEVCTL = 1;
	
    return 0;
}

int usb_dc_deinit(uint8_t busid)
{
    //SYSC_CPU->PD_CPU_CLKG = SYSC_CPU_CLKG_CLR_USB_MASK;
    //csi_vic_disable_irq(USB_IRQn);
    return 0;
}

int usbd_set_address(uint8_t busid, const uint8_t addr)
{
    if (addr == 0) {
        USB0->FADDR = 0;
    }

    g_musb_udc.dev_addr = addr;
    return 0;
}

uint8_t usbd_get_port_speed(const uint8_t port)
{
    return USB_SPEED_FULL;
}

uint8_t usbd_force_full_speed(const uint8_t port)
{    
    return 0;
}

int usbd_ep_open(uint8_t busid, const struct usb_endpoint_descriptor *ep_cfg)
{
    //uint16_t used = 0;
    //uint16_t fifo_index = 0;
    uint8_t ep_idx = USB_EP_GET_IDX(ep_cfg->bEndpointAddress);
    uint8_t old_ep_idx;
    //uint32_t ep_conf_flags = 0;

    if (ep_idx == 0) {
        g_musb_udc.out_ep[0].ep_mps = USB_CTRL_EP_MPS;
        g_musb_udc.out_ep[0].ep_type = 0x00;
        g_musb_udc.out_ep[0].ep_enable = true;
        g_musb_udc.in_ep[0].ep_mps = USB_CTRL_EP_MPS;
        g_musb_udc.in_ep[0].ep_type = 0x00;
        g_musb_udc.in_ep[0].ep_enable = true;
        return 0;
    }

    if (ep_idx > (ES_USB_PERH_EP_MAX_INDEX)) {
        USB_LOG_ERR("Ep addr %d overflow\r\n", ep_cfg->bEndpointAddress);
        return -1;
    }

    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
                 
    if (USB_EP_DIR_IS_OUT(ep_cfg->bEndpointAddress)) 
    {
        g_musb_udc.out_ep[ep_idx].ep_mps = ep_cfg->wMaxPacketSize;
        g_musb_udc.out_ep[ep_idx].ep_type = (USB_GET_ENDPOINT_TYPE(ep_cfg->bmAttributes));
        g_musb_udc.out_ep[ep_idx].ep_enable = true;
            
        USB0->RXMAXP = ep_cfg->wMaxPacketSize;
        USB0->RXCSRH = ((USB_GET_ENDPOINT_TYPE(ep_cfg->bmAttributes)) == USB_ENDPOINT_TYPE_ISOCHRONOUS) ? USB_RXCSRH1_ISO : 0;
        if (USB0->RXCSRL & USB_RXCSRL1_RXRDY)
            USB0->RXCSRL = USB_RXCSRL1_CLRDT | USB_RXCSRL1_FLUSH;
        else
            USB0->RXCSRL = USB_RXCSRL1_CLRDT;
    } 
    else 
    {
        g_musb_udc.in_ep[ep_idx].ep_mps = ep_cfg->wMaxPacketSize;
        g_musb_udc.in_ep[ep_idx].ep_type = (USB_GET_ENDPOINT_TYPE(ep_cfg->bmAttributes));
        g_musb_udc.in_ep[ep_idx].ep_enable = true;
                                 
        USB0->TXMAXP = ep_cfg->wMaxPacketSize;
        USB0->TXCSRH = ((USB_GET_ENDPOINT_TYPE(ep_cfg->bmAttributes)) == USB_ENDPOINT_TYPE_ISOCHRONOUS) ? USB_TXCSRH1_ISO : 0;
        if (USB0->TXCSRL & USB_TXCSRL1_TXRDY)
            USB0->TXCSRL = USB_TXCSRL1_CLRDT | USB_TXCSRL1_FLUSH;
        else
            USB0->TXCSRL = USB_TXCSRL1_CLRDT;
    }
    
    musb_set_active_ep(old_ep_idx);
    return 0;
}

int usbd_ep_close(uint8_t busid, const uint8_t ep)
{
    return 0;
}

int usbd_ep_set_stall(uint8_t busid, const uint8_t ep)
{
    uint8_t old_ep_idx;
	
    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep&0x7F);
		
	if(ep&0x7F)
	{
		if(ep&0x80)
            USB0->TXCSRL = USB_TXCSRL1_STALL;
        else
            USB0->RXCSRL = USB_RXCSRL1_STALL;
	}
	else 
	{
		USB0->CSRL0 = USB_CSRL0_STALL;
		//USB0->CSRL0 = (USB_CSRL0_DATAEND);
	}
	
	musb_set_active_ep(old_ep_idx);
	
    return 0;
}

int usbd_ep_clear_stall(uint8_t busid, const uint8_t ep)
{
    uint8_t ep_idx = USB_EP_GET_IDX(ep);
    uint8_t old_ep_idx;

    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);

    if (ep_idx == 0)
	{
        USB0->CSRL0 &= (uint8_t)(~USB_CSRL0_STALLED);
	}
    else if (USB_EP_DIR_IS_OUT(ep))
    {
        //USB0->RXCSRL = 0;
        //USB0->RXCSRL &= (uint8_t)(~(0x60));
        //USB0->RXCSRL &= (uint8_t)(~(USB_RXCSRL1_OVER));
        //USB0->RXCSRL &= (uint8_t)(~(USB_RXCSRL1_OVER/*USB_RXCSRL1_STALL*/ | USB_RXCSRL1_STALLED));
		
        //USB0->RXCSRL |= (uint8_t)USB_RXCSRL1_CLRDT;
        USB0->RXCSRL = (uint8_t)0x80;
	}
    else
    {
        //USB0->TXCSRL = 0;
        //USB0->TXCSRL &= (uint8_t)(~(0x24));
        //USB0->TXCSRL &= (uint8_t)(~(USB_TXCSRL1_UNDRN));
        //USB0->TXCSRL &= (uint8_t)(~(USB_TXCSRL1_UNDRN/*USB_TXCSRL1_STALL*/ | USB_TXCSRL1_STALLED));
		
        //USB0->TXCSRL |= (uint8_t)USB_TXCSRL1_CLRDT;
		USB0->TXCSRL = (uint8_t)0x40;
	}

    musb_set_active_ep(old_ep_idx);
    return 0;
}

int usbd_ep_is_stalled(uint8_t busid, const uint8_t ep, uint8_t *stalled)
{
    uint8_t old_ep; 
    
    old_ep = USB0->EPIDX;
               
    *stalled = 0x0;
    
    if(ep & 0x80)
    {
        USB0->EPIDX = ep;

        if(ep == 0)
        {
            if((USB0->CSRL0) & (USB_CSRL0_STALL))
                *stalled = 0x1;
        }
        else
        {
            if((USB0->TXCSRL) & (USB_TXCSRL1_STALL))
                *stalled = 0x1;
        }
    }
    else
    {
        USB0->EPIDX = ep;

        if(ep == 0)
        {
            if((USB0->CSRL0) & (USB_CSRL0_STALL))
                *stalled = 0x1;
        }
        else
        {
            if((USB0->RXCSRL) & (USB_RXCSRL1_STALL))
                *stalled = 0x1;
        }
    }
    
    USB0->EPIDX = old_ep;
    
    return 0;
}

int usb_ep_out_data_avail(uint8_t ep_addr)
{
	if(ep_addr)
		return USB0->RXCOUNT;
	else 
		return USB0->COUNT0;
}

static inline uint32_t es_usbd_ep_tx_ready_state(uint8_t ep_addr)
{
    return 0;
}

int usb_ep_in_data_avail(uint8_t ep_addr)
{
	if(ep_addr)
		return USB0->TXMAXP;
	else 
		return 64;
}

int usb_ep_wait_in_data_avail(uint8_t ep_addr)
{
    return 0;
}

static void es_usbd_ep_write_packet_8bit(uint8_t ep, void *buf, unsigned len)
{
    volatile hw_fifo_t *reg = (volatile hw_fifo_t*)((&USB0->FIFO0_WORD) + ep);
    uintptr_t addr = (uintptr_t)buf;
    while (len >= 4) {
        reg->u32 = *(uint32_t const *)addr;
        addr += 4;
        len  -= 4;
    }
    if (len >= 2) {
        reg->u16 = *(uint16_t const *)addr;
        addr += 2;
        len  -= 2;
    }
    if (len) {
        reg->u8 = *(uint8_t const *)addr;
    }
}

static void es_usbd_ep_read_packet_8bit(uint8_t ep, void *buf, unsigned len)
{
    volatile hw_fifo_t *reg = (volatile hw_fifo_t*)((&USB0->FIFO0_WORD) + ep);
    uintptr_t addr = (uintptr_t)buf;
    while (len >= 4) {
        *(uint32_t *)addr = reg->u32;
        addr += 4;
        len  -= 4;
    }
    if (len >= 2) {
        *(uint16_t *)addr = reg->u16;
        addr += 2;
        len  -= 2;
    }
    if (len) {
        *(uint8_t *)addr = reg->u8;
    }
}

int usbd_read_packet(uint8_t ep_addr, uint8_t *buffer, uint16_t len)
{
    uint16_t old_ep_idx, cnt;
    uint8_t ep_idx = USB_EP_GET_IDX(ep_addr);
    
    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);

    cnt = usb_ep_out_data_avail(ep_idx);
    if (cnt)
    {
		cnt = MIN(cnt,len);
        es_usbd_ep_read_packet_8bit(ep_idx, buffer, cnt);
		if(ep_idx == 0)
		{
			
		}
		else
		{
			USB0->RXCSRL &= ~(USB_RXCSRL1_OVER | USB_RXCSRL1_DATAERR | USB_RXCSRL1_STALL| USB_RXCSRL1_STALLED);
			USB0->RXCSRL &= ~(USB_RXCSRL1_RXRDY);
		}
    }
    
    musb_set_active_ep(old_ep_idx);
    return cnt;
}

int usbd_write_packet(uint8_t ep_addr, uint8_t *buffer, uint16_t len)
{
    uint16_t old_ep_idx, cnt;
    uint8_t ep_idx = USB_EP_GET_IDX(ep_addr);
    
	//printf("@w0_%d\r\n",len);
	
    //if (!g_musb_udc.in_ep[ep_idx].ep_enable)
    //    return -22;
            
    if (!buffer && len){
        return -2;
    }
    
    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
    
    if((USB0->TXCSRL) & USB_TXCSRL1_UNDRN)
        USB0->TXCSRL &= ~USB_TXCSRL1_UNDRN;

    if((USB0->TXCSRL) & USB_TXCSRL1_TXRDY)
    {
        musb_set_active_ep(old_ep_idx);
        return -1;
    }  
    
    if (!len)
    {
        USB0->TXCSRL |= USB_TXCSRL1_TXRDY;                   
        musb_set_active_ep(old_ep_idx);
        return 0;
    }
    
    cnt = usb_ep_in_data_avail(ep_idx);
    if (cnt)
    {
        cnt = MIN(cnt, len);
        es_usbd_ep_write_packet_8bit(ep_idx, buffer, cnt);
    }                                                   
    
    USB0->TXCSRL = USB_TXCSRL1_TXRDY;
    
    musb_set_active_ep(old_ep_idx);
    return cnt;
}

int usbd_ep_start_write(uint8_t busid, const uint8_t ep, const uint8_t *data, uint32_t data_len)
{
    uint8_t ep_idx = USB_EP_GET_IDX(ep);
    uint8_t old_ep_idx;

	//printf("@w1_%d %d",ep_idx,data_len);
	
    if (!data && data_len)
        return -1;
        
    if (!g_musb_udc.in_ep[ep_idx].ep_enable)
        return -2;
            
    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);

    if((USB0->TXCSRL) & USB_TXCSRL1_TXRDY)
    {
        musb_set_active_ep(old_ep_idx);
        return -3;
    }
    
    g_musb_udc.in_ep[ep_idx].xfer_buf = (uint8_t *)data;
    g_musb_udc.in_ep[ep_idx].xfer_len = data_len;
    g_musb_udc.in_ep[ep_idx].actual_xfer_len = 0;
	
	//USB->TXIER = 1U << ep_idx;

	//printf("@\r\n");
	              
    if (data_len == 0) 
    {  
        if (ep_idx == 0x00) 
        {
            if (g_musb_udc.setup.wLength == 0) 
                usb_ep0_state = USB_EP0_STATE_IN_STATUS;
            else
                usb_ep0_state = USB_EP0_STATE_IN_ZLP;
            
            //USB0->CSRL0 = (USB_CSRL0_TXRDY | USB_CSRL0_DATAEND);
            //USB0->CSRL0 = (USB_CSRL0_DATAEND);
        } 
        else 
        {
            USB0->TXCSRL = USB_TXCSRL1_TXRDY;
        }
        musb_set_active_ep(old_ep_idx);
        return 0;
    }
	
    data_len = MIN(data_len, g_musb_udc.in_ep[ep_idx].ep_mps);

    usbd_write_packet(ep_idx, (uint8_t *)data, data_len);

    if (ep_idx == 0x00) 
    {
        usb_ep0_state = USB_EP0_STATE_IN_DATA;
        
        if (data_len < g_musb_udc.in_ep[ep_idx].ep_mps)
            USB0->CSRL0 = (USB_CSRL0_TXRDY | USB_CSRL0_DATAEND);
        else
            USB0->CSRL0 = USB_CSRL0_TXRDY;
    } 
    else
        USB0->TXCSRL = USB_TXCSRL1_TXRDY;
    
    musb_set_active_ep(old_ep_idx);
    
    return 0;
}

int usbd_ep_start_read(uint8_t busid, const uint8_t ep, uint8_t *data, uint32_t data_len)
{
    uint8_t ep_idx = USB_EP_GET_IDX(ep);
    uint8_t old_ep_idx;

	//printf("@r1_%d %d",ep,data_len);
	
    if (!data && data_len) {
        return -1;
    }
    if (!g_musb_udc.out_ep[ep_idx].ep_enable) {
        return -2;
    }

	//printf("@\r\n");
	
    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);

    g_musb_udc.out_ep[ep_idx].xfer_buf = data;
    g_musb_udc.out_ep[ep_idx].xfer_len = data_len;
    g_musb_udc.out_ep[ep_idx].actual_xfer_len = 0;

    if (data_len == 0) {
        if (ep_idx == 0) {
            usb_ep0_state = USB_EP0_STATE_SETUP;
			//USB0->CSRL0 = USB_CSRL0_DATAEND;
            //USB0->CSRL0 = (USB_CSRL0_TXRDY | USB_CSRL0_DATAEND);
        }
        USB0->EPIDX = old_ep_idx;
        return 0;
    }
	
    if (ep_idx == 0) {
        usb_ep0_state = USB_EP0_STATE_OUT_DATA;
    } else {
        //USB0->RXIE = 1U << ep_idx;
    }
    musb_set_active_ep(old_ep_idx);
    return 0;
}

static void handle_ep0(void)
{
    USB0->EPIDX = 0;
	
    uint8_t ep0_status = USB0->CSRL0;
    uint16_t read_count;

	//if(ep0_status)
	//	printf("ep0 CSRL0 %x\r\n",ep0_status);
	//else
	//	printf("&");
	
	//printf("[%x]",ep0_status);

    if (ep0_status & USB_CSRL0_STALLED) {
        USB0->CSRL0 &= ~USB_CSRL0_STALLED;
        usb_ep0_state = USB_EP0_STATE_SETUP;
        return;
    }

    if (ep0_status & USB_CSRL0_SETEND) {
        USB0->CSRL0 = USB_CSRL0_SETENDC;
    }

	if(ep0_status == 0)
	{
		if (g_musb_udc.dev_addr > 0) {
			USB0->FADDR = g_musb_udc.dev_addr;
			g_musb_udc.dev_addr = 0;
    //USB0->CSRL0 = USB_CSRL0_RXRDYC | USB_CSRL0_DATAEND;
		}
	}

    switch (usb_ep0_state) {
        case USB_EP0_STATE_SETUP:
		
	//if(ep0_status)
	//	printf("USB_EP0_STATE_SETUP\r\n");
	//else
		//printf("*");

	
            if (ep0_status & USB_CSRL0_RXRDY) 
			{
                read_count = usb_ep_out_data_avail(0x0);


	//printf("read_count %d\r\n",read_count);

                if (read_count != 8) {
                    return;
                }

                usbd_read_packet(0, (uint8_t *)&g_musb_udc.setup, 8);
                if ((g_musb_udc.setup.wLength)) {
                //if ((g_musb_udc.setup.wLength)&&(!(g_musb_udc.setup.bmRequestType & 0x80))) {
                    USB0->CSRL0 = USB_CSRL0_RXRDYC;
                    //USB0->CSRL0 = (USB_CSRL0_RXRDYC | USB_CSRL0_DATAEND);
                } else {
                    USB0->CSRL0 = (USB_CSRL0_RXRDYC | USB_CSRL0_DATAEND);
                }

                usbd_event_ep0_setup_complete_handler(0, (uint8_t *)&g_musb_udc.setup);
            }
            break;

        case USB_EP0_STATE_IN_DATA:
	//printf("USB_EP0_STATE_IN_DATA\r\n");
			if((ep0_status & USB_CSRL0_TXRDY) == 0)
			{
	
				if (g_musb_udc.in_ep[0].xfer_len > g_musb_udc.in_ep[0].ep_mps) {
					g_musb_udc.in_ep[0].actual_xfer_len += g_musb_udc.in_ep[0].ep_mps;
					g_musb_udc.in_ep[0].xfer_len -= g_musb_udc.in_ep[0].ep_mps;
				} else {
					g_musb_udc.in_ep[0].actual_xfer_len += g_musb_udc.in_ep[0].xfer_len;
					g_musb_udc.in_ep[0].xfer_len = 0;
				}
				
				usbd_event_ep_in_complete_handler(0, 0x80, g_musb_udc.in_ep[0].actual_xfer_len);

//USB0->CSRL0 = (USB_CSRL0_DATAEND);
//USB0->CSRL0 = (USB_CSRL0_RXRDYC | USB_CSRL0_DATAEND);
//USB0->CSRL0 = (USB_CSRL0_RXRDYC);
//	printf("USB_EP0_STATE_IN_DATA %x\r\n",USB0->CSRL0);
			}

            break;
        case USB_EP0_STATE_OUT_DATA:
	//printf("USB_EP0_STATE_OUT_DATA\r\n");
	
            if (ep0_status & USB_CSRL0_RXRDY)
			{
                read_count = usb_ep_out_data_avail(0x0);

                usbd_read_packet(0, g_musb_udc.out_ep[0].xfer_buf, read_count);
                g_musb_udc.out_ep[0].xfer_buf += read_count;
                g_musb_udc.out_ep[0].actual_xfer_len += read_count;

                if (read_count < g_musb_udc.out_ep[0].ep_mps) {
                    usbd_event_ep_out_complete_handler(0, 0x00, g_musb_udc.out_ep[0].actual_xfer_len);
                    USB0->CSRL0 = (USB_CSRL0_RXRDYC | USB_CSRL0_DATAEND);
                    usb_ep0_state = USB_EP0_STATE_IN_STATUS;
                } else {
                    USB0->CSRL0 = USB_CSRL0_RXRDYC;
                }
            }
            break;
        case USB_EP0_STATE_IN_STATUS:
        case USB_EP0_STATE_IN_ZLP:
	//printf("USB_EP0_STATE_IN_ZLP\r\n");
	
			if((ep0_status & USB_CSRL0_TXRDY) == 0)
			{
				usb_ep0_state = USB_EP0_STATE_SETUP;
				usbd_event_ep_in_complete_handler(0, 0x80, 0);
			}
            break;
    }
}

const uint8_t __lowest_bit_bitmap[] =
{
    /* 00 */ 0, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 10 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 20 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 30 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 40 */ 6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 50 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 60 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 70 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 80 */ 7, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 90 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* A0 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* B0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* C0 */ 6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* D0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* E0 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* F0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0
};

//uint32_t tst_int_num = 0;

#include "compile_flag.h"

#ifdef ES_USB_Handler_NOT_IN_IRQ
void USB_Handler(void)
#else
/*void __attribute__((interrupt)) XIP_BANNED_FUNC(USB_Handler)*/
void __attribute__((interrupt)) USB_Handler(void)
#endif/*ES_USBD_RX_TX_NOT_IN_IRQs*/
{
    uint8_t is, txis, rxis,old_ep_idx;
    uint8_t ep_idx;
    uint16_t write_count, read_count;
    
#ifdef ES_USB_Handler_NOT_IN_IRQ
	/**/static unsigned long last_systick_value = 0U;
	if((CORET->MTIME - last_systick_value) < (SDK_HCLK_MHZ << 8))
		return;
	else
		last_systick_value = CORET->MTIME;
#endif/*ES_USB_Handler_NOT_IN_IRQ*/

    is   = USB0->IS;   /* read and clear interrupt status */
    txis = USB0->TXIS; /* read and clear interrupt status */
    rxis = USB0->RXIS; /* read and clear interrupt status */
	
	old_ep_idx = musb_get_active_ep();

    is &= USB0->IE; /* Clear disabled interrupts */
    if (is & USB_IS_DISCON) {
		//printf("int_discon\r\n");
    }
    if (is & USB_IS_SOF) {
		//printf("int_sof\r\n");
    }
    if (is & USB_IS_RESET) {
		//printf("int_resrt\r\n");
        memset(&g_musb_udc, 0, sizeof(struct musb_udc));
        g_musb_udc.fifo_size_offset = USB_CTRL_EP_MPS;
        usbd_event_reset_handler(0);
        
        usb_ep0_state = USB_EP0_STATE_SETUP;
		USB0->FADDR = 0;
    }
    if (is & USB_IS_RESUME) {
		//printf("int_resume\r\n");
    }
    if (is & USB_IS_SUSPEND) {
		//printf("int_suspend\r\n");
    }

    txis &= USB0->TXIE; /* Clear disabled interrupts */
    if (txis & USB_TXIE_EP0)
	{
			do
			{
				handle_ep0();
			}
			/*while(USB0->CSRL0);*/
			while(USB0->CSRL0 & 0x1);
			/*while(usb_ep0_state != USB_EP0_STATE_SETUP);*/
        txis &= 0xFE;
    }
    while (txis) {
        ep_idx = __lowest_bit_bitmap[txis];
        txis &= ~BIT(ep_idx);
        musb_set_active_ep(ep_idx);

            if((USB0->TXCSRL) & USB_TXCSRL1_UNDRN)
                USB0->TXCSRL &= ~USB_TXCSRL1_UNDRN;

            if(g_musb_udc.in_ep[ep_idx].xfer_len)
            {
                if (g_musb_udc.in_ep[ep_idx].xfer_len > g_musb_udc.in_ep[ep_idx].ep_mps) 
                {
                    g_musb_udc.in_ep[ep_idx].xfer_buf += g_musb_udc.in_ep[ep_idx].ep_mps;
                    g_musb_udc.in_ep[ep_idx].actual_xfer_len += g_musb_udc.in_ep[ep_idx].ep_mps;
                    g_musb_udc.in_ep[ep_idx].xfer_len -= g_musb_udc.in_ep[ep_idx].ep_mps;
                } 
                else 
                {
                    g_musb_udc.in_ep[ep_idx].xfer_buf += g_musb_udc.in_ep[ep_idx].xfer_len;
                    g_musb_udc.in_ep[ep_idx].actual_xfer_len += g_musb_udc.in_ep[ep_idx].xfer_len;
                    g_musb_udc.in_ep[ep_idx].xfer_len = 0;
                }
            }

            if (g_musb_udc.in_ep[ep_idx].xfer_len == 0) 
            {   
                usbd_event_ep_in_complete_handler(0, ep_idx | 0x80, g_musb_udc.in_ep[ep_idx].actual_xfer_len);
            } 
            else 
            {
                write_count = MIN(g_musb_udc.in_ep[ep_idx].xfer_len, g_musb_udc.in_ep[ep_idx].ep_mps);

                usbd_write_packet(ep_idx, g_musb_udc.in_ep[ep_idx].xfer_buf, write_count);
            }
    }
    rxis &= USB0->RXIE; /* Clear disabled interrupts */
    while (rxis) {
        ep_idx = __lowest_bit_bitmap[rxis];
        rxis &= ~BIT(ep_idx);
		
        USB0->EPIDX = ep_idx;
        
        if((USB0->RXCSRL) & USB_RXCSRL1_RXRDY)
        {
            read_count = usb_ep_out_data_avail(ep_idx);

            if(read_count <= g_musb_udc.out_ep[ep_idx].xfer_len)
            {
                es_usbd_ep_read_packet_8bit(ep_idx, g_musb_udc.out_ep[ep_idx].xfer_buf, read_count);

                g_musb_udc.out_ep[ep_idx].xfer_buf += read_count;
                g_musb_udc.out_ep[ep_idx].actual_xfer_len += read_count;
                g_musb_udc.out_ep[ep_idx].xfer_len -= read_count;
				
				
				USB0->RXCSRL &= ~(USB_RXCSRL1_DATAERR | USB_RXCSRL1_STALL| USB_RXCSRL1_STALLED);
				USB0->RXCSRL &= ~USB_RXCSRL1_RXRDY;
            }                                    
            /*else
                USB_LOG_ERR("[ep %d]read_count(%d) > xfer_len(%d)\r\n",ep_idx,read_count,g_musb_udc.out_ep[ep_idx].xfer_len);*/
            
            
            if ((read_count < g_musb_udc.out_ep[ep_idx].ep_mps) || (g_musb_udc.out_ep[ep_idx].xfer_len < g_musb_udc.out_ep[ep_idx].ep_mps)) 
            {
                //USB->RXIDR = 1U << ep_idx;
                usbd_event_ep_out_complete_handler(0, ep_idx, g_musb_udc.out_ep[ep_idx].actual_xfer_len);
            }
        }
    }

    musb_set_active_ep(old_ep_idx);
}

