#include "usbd_core.h"
#include "ald_usb.h"
#include "md_utils.h"

#define ES_USB_PERH_EP_MAX_INDEX     (4U)

typedef enum {
    USB_EP0_STATE_SETUP = 0x0,      /**< SETUP DATA */
    USB_EP0_STATE_IN_DATA = 0x1,    /**< IN DATA */
    USB_EP0_STATE_OUT_DATA = 0x3,   /**< OUT DATA */
    USB_EP0_STATE_IN_STATUS = 0x4,  /**< IN status */
    USB_EP0_STATE_OUT_STATUS = 0x5, /**< OUT status */
    USB_EP0_STATE_IN_ZLP = 0x6,     /**< OUT status */
    USB_EP0_STATE_STALL = 0x7,      /**< STALL status */
} ep0_state_t;

/* Endpoint state */
struct musb_ep_state {
    uint16_t ep_mps;    /* Endpoint max packet size */
    uint8_t ep_type;    /* Endpoint type */
    uint8_t ep_stalled; /* Endpoint stall flag */
    uint8_t ep_enable;  /* Endpoint enable */
    uint8_t *xfer_buf;
    uint32_t xfer_len;
    uint32_t actual_xfer_len;
};

/* Driver state */
struct musb_udc {
    volatile uint8_t dev_addr;
    volatile uint32_t fifo_size_offset;
    __attribute__((aligned(32))) struct usb_setup_packet setup;
    struct musb_ep_state in_ep[ES_USB_PERH_EP_MAX_INDEX + 1];  /*!< IN endpoint parameters*/
    struct musb_ep_state out_ep[ES_USB_PERH_EP_MAX_INDEX + 1]; /*!< OUT endpoint parameters */
} g_musb_udc;

static volatile uint8_t usb_ep0_state = USB_EP0_STATE_SETUP;
volatile bool zlp_flag = 0;

void ald_usb_fifo_config_set_no_assert(uint32_t ep_idx, uint32_t addr, uint32_t size, uint32_t flags);

__weak void usb_dc_low_level_init(void)
{
}

__WEAK void usb_dc_low_level_deinit(void)
{
}

__WEAK void es_usb_ep_fifo_set_custom(void)
{
}

static uint32_t es_usbd_get_fifo_size16_log2_min8(uint16_t need_size, uint16_t *used)
{
    uint16_t size;
    uint8_t i;

    for (i = 3; i <= 15; i++) 
    {
        size = (1U << i);
        if (need_size <= size) 
        {
            *used = size;
            return i;
        }
    }

    *used = 0;
    return 0;
}

static void es_usbd_ep_write_packet_8bit(uint8_t ep_idx, uint8_t *buffer, uint16_t len)
{
    uint32_t i;
    uint8_t* buf8 = (uint8_t*)(((uint32_t)(&(USB->EP0FIFO))) + ((ep_idx) << 2));

    for(i = 0;i < len;i++)
        *buf8 = buffer[i];
}

static void es_usbd_ep_read_packet_8bit(uint8_t ep_idx, uint8_t *buffer, uint16_t len)
{
    uint32_t i;
    uint8_t* buf8 = (uint8_t*)(((uint32_t)(&(USB->EP0FIFO))) + ((ep_idx) << 2));

    for(i = 0;i < len;i++)
        buffer[i] = *buf8;
}

/* get current active ep */
static inline uint8_t musb_get_active_ep(void)
{
    return USB->INDEX;
}

/* set the active ep */
static inline void musb_set_active_ep(uint8_t ep_index)
{
    USB->INDEX = ep_index;
}

int usb_dc_init(uint8_t busid)
{
	ald_usb_int_unregister();
    usb_dc_low_level_init();
	
	ald_usb_dev_disconnect();
	
#ifndef ES_USB_FIFO_SET_CUSTOM

	ald_usb_fifo_config_set_no_assert(0 , 0 , 0x3 , ALD_USB_EP_DEV_OUT);
    ald_usb_fifo_config_set_no_assert(0 , 0x40 , 0x3 , ALD_USB_EP_DEV_IN);

#else
	
	es_usb_ep_fifo_set_custom();

#endif
	
	md_delay_1ms(50);


    ald_usb_device_components_init();
    
    ald_usb_int_status_get();
    ald_usb_int_status_ep_get();

	/* Enable software connect */

	ald_usb_dev_connect();
	
    /* Enable USB interrupts */  
    ald_usb_int_enable(ALD_USB_INTCTRL_RESET | ALD_USB_INTCTRL_DISCONNECT | ALD_USB_INTCTRL_RESUME |ALD_USB_INTCTRL_SUSPEND);/*未开启SOF中断*/
	ald_usb_int_enable_ep(ALD_USB_INTEP_ALL);
	ald_usb_int_register();
    return 0;
}

int usb_dc_deinit(uint8_t busid)
{
    return 0;
}

int usbd_set_address(uint8_t busid, const uint8_t addr)
{
    if (addr == 0) {
        ald_usb_dev_set_addr(addr);
    }

    g_musb_udc.dev_addr = addr;
    return 0;
}

uint8_t usbd_get_port_speed(const uint8_t port)
{
    return USB_SPEED_FULL;
}

uint8_t usbd_force_full_speed(const uint8_t port)
{    
    return 0;
}

void ald_usb_fifo_config_set_no_assert(uint32_t ep_idx, uint32_t addr, uint32_t size, uint32_t flags)
{
	USB->INDEX = ep_idx;
	
	if (flags & (ALD_USB_EP_HOST_OUT | ALD_USB_EP_DEV_IN)) {
		USB->TXFIFO1 = (uint8_t)((addr & 0xFFFF) >> 3);
		USB->TXFIFO2 |= (uint8_t)((((addr & 0xFFFF) >> 3) >> 8) & 0x0F);
		USB->TXFIFO2 |= (uint8_t)(size << USB_TXFIFO2_MAXPKTSIZE_POSS);
		
		USB->CSR0L_TXCSRL |= (uint8_t)ALD_USB_TXCSRL_FLUSH;
	}
	else {
		USB->RXFIFO1 = (uint8_t)((addr & 0xFFFF) >> 3);
		USB->RXFIFO2 |= (uint8_t)((((addr & 0xFFFF) >> 3) >> 8) & 0x0F);
		USB->RXFIFO2 |= (uint8_t)(size << USB_RXFIFO2_MAXPKTSIZE_POSS);
		
		USB->RXCSRL |= (uint8_t)ALD_USB_RXCSRL_FLUSH;
	}

	return;
}

int usbd_ep_open(uint8_t busid, const struct usb_endpoint_descriptor *ep_cfg)
{
    uint16_t used = 0;
    uint16_t fifo_index = 0;
    uint8_t ep_idx = USB_EP_GET_IDX(ep_cfg->bEndpointAddress);
    uint8_t old_ep_idx;
    uint32_t ep_conf_flags = 0;

    if (ep_idx == 0) {
        g_musb_udc.out_ep[0].ep_mps = USB_CTRL_EP_MPS;
        g_musb_udc.out_ep[0].ep_type = 0x00;
        g_musb_udc.out_ep[0].ep_enable = true;
        g_musb_udc.in_ep[0].ep_mps = USB_CTRL_EP_MPS;
        g_musb_udc.in_ep[0].ep_type = 0x00;
        g_musb_udc.in_ep[0].ep_enable = true;
        return 0;
    }

    if (ep_idx > (ES_USB_PERH_EP_MAX_INDEX)) {
		
#ifndef ES_USB_DEV_USE_FAKE_EP
        USB_LOG_ERR("Ep addr %d overflow\r\n", ep_cfg->bEndpointAddress);
#endif
		
        return -1;
    }

    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
                 
    if((USB_GET_ENDPOINT_TYPE(ep_cfg->bmAttributes)) == USB_ENDPOINT_TYPE_ISOCHRONOUS) 
        ep_conf_flags |= ALD_USB_EP_MODE_ISOC;          
    if((USB_GET_ENDPOINT_TYPE(ep_cfg->bmAttributes)) == USB_ENDPOINT_TYPE_INTERRUPT) 
        ep_conf_flags |= ALD_USB_EP_MODE_INT;       
    if((USB_GET_ENDPOINT_TYPE(ep_cfg->bmAttributes)) == USB_ENDPOINT_TYPE_CONTROL) 
        ep_conf_flags |= ALD_USB_EP_MODE_CTRL;       
    if((USB_GET_ENDPOINT_TYPE(ep_cfg->bmAttributes)) == USB_ENDPOINT_TYPE_BULK) 
        ep_conf_flags |= ALD_USB_EP_MODE_BULK;
        
    if (USB_EP_DIR_IS_OUT(ep_cfg->bEndpointAddress)) 
    {
        g_musb_udc.out_ep[ep_idx].ep_mps = ep_cfg->wMaxPacketSize;
        g_musb_udc.out_ep[ep_idx].ep_type = (USB_GET_ENDPOINT_TYPE(ep_cfg->bmAttributes));
        g_musb_udc.out_ep[ep_idx].ep_enable = true;
            
        ep_conf_flags |= ALD_USB_EP_DEV_OUT;
    } 
    else 
    {
        g_musb_udc.in_ep[ep_idx].ep_mps = ep_cfg->wMaxPacketSize;
        g_musb_udc.in_ep[ep_idx].ep_type = (USB_GET_ENDPOINT_TYPE(ep_cfg->bmAttributes));
        g_musb_udc.in_ep[ep_idx].ep_enable = true;
                                 
        ep_conf_flags |= ALD_USB_EP_DEV_IN;
    }
    
    ald_usb_dev_ep_config(ep_idx , ep_cfg->wMaxPacketSize , ep_conf_flags);
    
#ifndef ES_USB_FIFO_SET_CUSTOM

    fifo_index = es_usbd_get_fifo_size16_log2_min8(ep_cfg->wMaxPacketSize , &used);
    
	if(g_musb_udc.fifo_size_offset < 0x80)
		g_musb_udc.fifo_size_offset = 0x80;
	
    if((10 >= fifo_index))
    {
        fifo_index -= 3;
        
        if(USB_EP_DIR_IS_OUT(ep_cfg->bEndpointAddress))
            ald_usb_fifo_config_set_no_assert(ep_idx , g_musb_udc.fifo_size_offset , fifo_index , ALD_USB_EP_DEV_OUT);
        else                                                                                              
            ald_usb_fifo_config_set_no_assert(ep_idx , g_musb_udc.fifo_size_offset , fifo_index , ALD_USB_EP_DEV_IN);
    }
    
    g_musb_udc.fifo_size_offset += used;
	
#endif
    
    musb_set_active_ep(old_ep_idx);

    return 0;
}

int usbd_ep_close(uint8_t busid, const uint8_t ep)
{
    return 0;
}

int usbd_ep_set_stall(uint8_t busid, const uint8_t ep)
{
    uint8_t ep_idx = USB_EP_GET_IDX(ep);
    uint8_t old_ep_idx;

    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);

    if (ep_idx == 0x00)
        usb_ep0_state = USB_EP0_STATE_STALL;
   
    if (USB_EP_DIR_IS_OUT(ep))
        ald_usb_dev_ep_stall(ep_idx,ALD_USB_EP_DEV_OUT);
    else
        ald_usb_dev_ep_stall(ep_idx,ALD_USB_EP_DEV_IN);

    musb_set_active_ep(old_ep_idx);
    return 0;
}

int usbd_ep_clear_stall(uint8_t busid, const uint8_t ep)
{
    uint8_t ep_idx = USB_EP_GET_IDX(ep);
    uint8_t old_ep_idx;

    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);

    if (USB_EP_DIR_IS_OUT(ep))
        ald_usb_dev_ep_stall_clear(ep_idx,ALD_USB_EP_DEV_OUT);
    else
        ald_usb_dev_ep_stall_clear(ep_idx,ALD_USB_EP_DEV_IN);

    musb_set_active_ep(old_ep_idx);
    return 0;
}

int usbd_ep_is_stalled(uint8_t busid, const uint8_t ep, uint8_t *stalled)
{
    uint8_t old_ep; 
    
    old_ep = USB->INDEX;
               
    *stalled = 0x0;
    
    if(ep & 0x80)
    {
        USB->INDEX = ep;

        if(ep == 0)
        {
            if((USB->CSR0L_TXCSRL) & (USB_CSR0L_STALL_MSK))
                *stalled = 0x1;
        }
        else
        {
            if((USB->CSR0L_TXCSRL) & (USB_TXCSRL_STALL_MSK))
                *stalled = 0x1;
        }
    }
    else
    {
        USB->INDEX = ep;

        if(ep == 0)
        {
            if((USB->CSR0L_TXCSRL) & (USB_CSR0L_STALL_MSK))
                *stalled = 0x1;
        }
        else
        {
            if((USB->RXCSRL) & (USB_RXCSRL_STALL_MSK))
                *stalled = 0x1;
        }
    }
    
    USB->INDEX = old_ep;
    
    return 0;
}

int es_usb_ep_out_data_avail(uint8_t ep_addr)
{
    return ald_usb_ep_data_avail(USB_EP_GET_IDX(ep_addr));
}

int usb_ep_out_data_avail(uint8_t ep_addr)
{
	uint16_t conut; 
	uint8_t ep_idx = USB_EP_GET_IDX(ep_addr);
    uint8_t old_ep_idx;

    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
    
	conut = ald_usb_ep_data_avail(USB_EP_GET_IDX(ep_idx));
	
    musb_set_active_ep(old_ep_idx);
	
	return conut;
}

static inline uint32_t es_usbd_ep_tx_ready_state(uint8_t ep_addr)
{
    if (ep_addr == 0U) 
    {
		if (USB->CSR0L_TXCSRL & ALD_USB_CSR0L_TXRDY)
			return 1;
	}
	else 
    {
		if (USB->CSR0L_TXCSRL & ALD_USB_TXCSRL_TXRDY)
			return 2;
	}
    
    return 0;
}

int usb_ep_in_data_avail(uint8_t ep_addr)
{
    uint16_t old_ep_idx, length;
    uint8_t ep_idx = USB_EP_GET_IDX(ep_addr);
    
    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
    
    if(es_usbd_ep_tx_ready_state(ep_idx))
        length = 0;
    else
    {
        if(ep_idx == 0)
            length = 64;
        else
            length = USB->TXMAXP;
    }
    
    musb_set_active_ep(old_ep_idx);
    return length;
}

int usb_ep_wait_in_data_avail(uint8_t ep_addr)
{
    uint32_t cnt;
    
    for (cnt = 0; cnt < 3000; cnt++){
        if (usb_ep_in_data_avail(ep_addr))
            return cnt;
    }
    return 0;
}

int usbd_read_packet(uint8_t ep_addr, uint8_t *buffer, uint16_t len)
{
    uint16_t old_ep_idx, cnt;
    uint8_t ep_idx = USB_EP_GET_IDX(ep_addr);
    
    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
    
    cnt = es_usb_ep_out_data_avail(ep_idx);
    if (cnt)
    {
		cnt = MIN(cnt,len);
        es_usbd_ep_read_packet_8bit(ep_idx, buffer, cnt);
		USB->RXCSRL &= ~(ALD_USB_RXCSRL_OVERRUN | ALD_USB_RXCSRL_DATAERR | ALD_USB_RXCSRL_STALL| ALD_USB_RXCSRL_STALLED);
		USB->RXCSRL &= ~(ALD_USB_RXCSRL_RXRDY);
    }
    
    musb_set_active_ep(old_ep_idx);
    return cnt;
}

int usbd_write_packet(uint8_t ep_addr, uint8_t *buffer, uint16_t len)
{
    uint16_t old_ep_idx, cnt;
    uint8_t ep_idx = USB_EP_GET_IDX(ep_addr);
    
    if (!buffer && len){
        return -2;
    }
    
    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);
    
    if((USB->CSR0L_TXCSRL) & USB_TXCSRL_UNDRUN_MSK)
        USB->CSR0L_TXCSRL &= ~USB_TXCSRL_UNDRUN_MSK;

    if((USB->CSR0L_TXCSRL) & USB_TXCSRL_TXRDY_MSK)
    {
        musb_set_active_ep(old_ep_idx);
        return -1;
    }  
    
    if (!len)
    {
        USB->CSR0L_TXCSRL |= USB_TXCSRL_TXRDY_MSK;                   
        musb_set_active_ep(old_ep_idx);
        return 0;
    }
    
    cnt = usb_ep_in_data_avail(ep_idx);
    if (cnt)
    {
        cnt = MIN(cnt, len);
        es_usbd_ep_write_packet_8bit(ep_idx, buffer, cnt);
    }                                                   
    
    USB->CSR0L_TXCSRL |= USB_TXCSRL_TXRDY_MSK;
    
    musb_set_active_ep(old_ep_idx);
    return cnt;
}

int usbd_ep_start_write(uint8_t busid, const uint8_t ep, const uint8_t *data, uint32_t data_len)
{
    uint8_t ep_idx = USB_EP_GET_IDX(ep);
    uint8_t old_ep_idx;

    if (!data && data_len)
        return -1;
    
    if (!g_musb_udc.in_ep[ep_idx].ep_enable)
        return -2;
                              
    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);

    if((USB->CSR0L_TXCSRL) & USB_TXCSRL_TXRDY_MSK)
    {
        musb_set_active_ep(old_ep_idx);
        return -3;
    }
    
    g_musb_udc.in_ep[ep_idx].xfer_buf = (uint8_t *)data;
    g_musb_udc.in_ep[ep_idx].xfer_len = data_len;
    g_musb_udc.in_ep[ep_idx].actual_xfer_len = 0;
	
	USB->TXIER = 1U << ep_idx;

    if (data_len == 0) 
    {  
        if (ep_idx == 0x00) 
        {
            if (g_musb_udc.setup.wLength == 0) 
                usb_ep0_state = USB_EP0_STATE_IN_STATUS;
            else
                usb_ep0_state = USB_EP0_STATE_IN_ZLP;
            
            USB->CSR0L_TXCSRL = (ALD_USB_CSR0L_TXRDY | ALD_USB_CSR0L_DATAEND);
        } 
        else 
        {
            USB->CSR0L_TXCSRL = ALD_USB_TXCSRL_TXRDY;
        }
        musb_set_active_ep(old_ep_idx);
        return 0;
    }
    data_len = MIN(data_len, g_musb_udc.in_ep[ep_idx].ep_mps);

    usbd_write_packet(ep_idx, (uint8_t *)data, data_len);

    if (ep_idx == 0x00) 
    {
        usb_ep0_state = USB_EP0_STATE_IN_DATA;
        
        if (data_len < g_musb_udc.in_ep[ep_idx].ep_mps)
            USB->CSR0L_TXCSRL = (ALD_USB_CSR0L_TXRDY | ALD_USB_CSR0L_DATAEND);
        else
            USB->CSR0L_TXCSRL = ALD_USB_CSR0L_TXRDY;
    } 
    else
        USB->CSR0L_TXCSRL = ALD_USB_TXCSRL_TXRDY;
    
    musb_set_active_ep(old_ep_idx);
    
    return 0;
}

int usbd_ep_start_read(uint8_t busid, const uint8_t ep, uint8_t *data, uint32_t data_len)
{
    uint8_t ep_idx = USB_EP_GET_IDX(ep);
    uint8_t old_ep_idx;

    if (!data && data_len) {
        return -1;
    }
    if (!g_musb_udc.out_ep[ep_idx].ep_enable) {
        return -2;
    }

    old_ep_idx = musb_get_active_ep();
    musb_set_active_ep(ep_idx);

    g_musb_udc.out_ep[ep_idx].xfer_buf = data;
    g_musb_udc.out_ep[ep_idx].xfer_len = data_len;
    g_musb_udc.out_ep[ep_idx].actual_xfer_len = 0;

    if (data_len == 0) {
        if (ep_idx == 0) {
            usb_ep0_state = USB_EP0_STATE_SETUP;
        }
        musb_set_active_ep(old_ep_idx);
        return 0;
    }
    if (ep_idx == 0) {
        usb_ep0_state = USB_EP0_STATE_OUT_DATA;
    } else {
        USB->RXIER = 1U << ep_idx;
    }
    musb_set_active_ep(old_ep_idx);
    return 0;
}

static void handle_ep0(void)
{
    uint8_t ep0_status = USB->CSR0L_TXCSRL;
    uint16_t read_count;

    if (ep0_status & ALD_USB_CSR0L_STALLED) {
        USB->CSR0L_TXCSRL &= ~ALD_USB_CSR0L_STALLED;
        usb_ep0_state = USB_EP0_STATE_SETUP;
        return;
    }

    if (ep0_status & ALD_USB_CSR0L_SETEND) {
        USB->CSR0L_TXCSRL = ALD_USB_CSR0L_SETENDC;
    }

    if (g_musb_udc.dev_addr > 0) {
        USB->FADDR = g_musb_udc.dev_addr;
        g_musb_udc.dev_addr = 0;
    }

    switch (usb_ep0_state) {
        case USB_EP0_STATE_SETUP:
            if (ep0_status & ALD_USB_CSR0L_RXRDY) {
                read_count = es_usb_ep_out_data_avail(0x0);

                if (read_count != 8) {
                    return;
                }

                usbd_read_packet(0, (uint8_t *)&g_musb_udc.setup, 8);
                if (g_musb_udc.setup.wLength) {
                    USB->CSR0L_TXCSRL = ALD_USB_CSR0L_RXRDYC;
                } else {
                    USB->CSR0L_TXCSRL = (ALD_USB_CSR0L_RXRDYC | ALD_USB_CSR0L_DATAEND);
                }

                usbd_event_ep0_setup_complete_handler(0, (uint8_t *)&g_musb_udc.setup);
            }
            break;

        case USB_EP0_STATE_IN_DATA:
            if (g_musb_udc.in_ep[0].xfer_len > g_musb_udc.in_ep[0].ep_mps) {
                g_musb_udc.in_ep[0].actual_xfer_len += g_musb_udc.in_ep[0].ep_mps;
                g_musb_udc.in_ep[0].xfer_len -= g_musb_udc.in_ep[0].ep_mps;
            } else {
                g_musb_udc.in_ep[0].actual_xfer_len += g_musb_udc.in_ep[0].xfer_len;
                g_musb_udc.in_ep[0].xfer_len = 0;
            }

				usbd_event_ep_in_complete_handler(0, 0x80, g_musb_udc.in_ep[0].actual_xfer_len);

            break;
        case USB_EP0_STATE_OUT_DATA:
            if (ep0_status & ALD_USB_CSR0L_RXRDY) {
                read_count = es_usb_ep_out_data_avail(0x0);

                usbd_read_packet(0, g_musb_udc.out_ep[0].xfer_buf, read_count);
                g_musb_udc.out_ep[0].xfer_buf += read_count;
                g_musb_udc.out_ep[0].actual_xfer_len += read_count;

                if (read_count < g_musb_udc.out_ep[0].ep_mps) {
                    usbd_event_ep_out_complete_handler(0, 0x00, g_musb_udc.out_ep[0].actual_xfer_len);
                    USB->CSR0L_TXCSRL = (ALD_USB_CSR0L_RXRDYC | ALD_USB_CSR0L_DATAEND);
                    usb_ep0_state = USB_EP0_STATE_IN_STATUS;
                } else {
                    USB->CSR0L_TXCSRL = ALD_USB_CSR0L_RXRDYC;
                }
            }
            break;
        case USB_EP0_STATE_IN_STATUS:
        case USB_EP0_STATE_IN_ZLP:
            usb_ep0_state = USB_EP0_STATE_SETUP;
				usbd_event_ep_in_complete_handler(0, 0x80, 0);
            break;
    }
}

const uint8_t __lowest_bit_bitmap[] =
{
    /* 00 */ 0, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 10 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 20 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 30 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 40 */ 6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 50 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 60 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 70 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 80 */ 7, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* 90 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* A0 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* B0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* C0 */ 6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* D0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* E0 */ 5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0,
    /* F0 */ 4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0
};

void __attribute__((interrupt)) USB_Handler(void)
{
    uint8_t old_ep_idx;
    uint8_t ep_idx;
    uint16_t write_count, read_count;
    uint8_t status;

	//printf("%x\r\n",USB->IFM);

	//while(USB->IFM)
	{
		status = USB->IFM & 0x7F;
		
		USB->ICR |= status;
		
		old_ep_idx = musb_get_active_ep();

		/* Receive a reset signal from the USB bus */
		if (status & USB_IFM_RESTIFM_MSK) 
		{
			md_delay_1us(20);
			
			memset(&g_musb_udc, 0, sizeof(struct musb_udc));
			g_musb_udc.fifo_size_offset = USB_CTRL_EP_MPS;
			usbd_event_reset_handler(0);
			
			usb_ep0_state = USB_EP0_STATE_SETUP;
		}
	}
	
    while (USB->TXIFM) 
    {                                      
        ep_idx = __lowest_bit_bitmap[USB->TXIFM];
        USB->TXICR = 1U << ep_idx;  
        musb_set_active_ep(ep_idx);
		
        if(ep_idx == 0U)
            handle_ep0();
        else
        {   
            if((USB->CSR0L_TXCSRL) & USB_TXCSRL_UNDRUN_MSK)
                USB->CSR0L_TXCSRL &= ~USB_TXCSRL_UNDRUN_MSK;

            if(g_musb_udc.in_ep[ep_idx].xfer_len)
            {
                if (g_musb_udc.in_ep[ep_idx].xfer_len > g_musb_udc.in_ep[ep_idx].ep_mps) 
                {
                    g_musb_udc.in_ep[ep_idx].xfer_buf += g_musb_udc.in_ep[ep_idx].ep_mps;
                    g_musb_udc.in_ep[ep_idx].actual_xfer_len += g_musb_udc.in_ep[ep_idx].ep_mps;
                    g_musb_udc.in_ep[ep_idx].xfer_len -= g_musb_udc.in_ep[ep_idx].ep_mps;
                } 
                else 
                {
                    g_musb_udc.in_ep[ep_idx].xfer_buf += g_musb_udc.in_ep[ep_idx].xfer_len;
                    g_musb_udc.in_ep[ep_idx].actual_xfer_len += g_musb_udc.in_ep[ep_idx].xfer_len;
                    g_musb_udc.in_ep[ep_idx].xfer_len = 0;
                }
            }

            if (g_musb_udc.in_ep[ep_idx].xfer_len == 0) 
            {   
                USB->TXIDR = 1U << ep_idx; 
                usbd_event_ep_in_complete_handler(0, ep_idx | 0x80, g_musb_udc.in_ep[ep_idx].actual_xfer_len);
            } 
            else 
            {
                write_count = MIN(g_musb_udc.in_ep[ep_idx].xfer_len, g_musb_udc.in_ep[ep_idx].ep_mps);

                usbd_write_packet(ep_idx, g_musb_udc.in_ep[ep_idx].xfer_buf, write_count);
            }
        }
    }

    while (USB->RXIFM) 
    {                                  
        ep_idx = __lowest_bit_bitmap[USB->RXIFM];  
        USB->RXICR = 1U << ep_idx;  
        musb_set_active_ep(ep_idx);
        
        if((USB->RXCSRL) & USB_RXCSRL_RXRDY_MSK)
        {
            read_count = es_usb_ep_out_data_avail(ep_idx);

            if(read_count <= g_musb_udc.out_ep[ep_idx].xfer_len)
            {
                es_usbd_ep_read_packet_8bit(ep_idx, g_musb_udc.out_ep[ep_idx].xfer_buf, read_count);

                g_musb_udc.out_ep[ep_idx].xfer_buf += read_count;
                g_musb_udc.out_ep[ep_idx].actual_xfer_len += read_count;
                g_musb_udc.out_ep[ep_idx].xfer_len -= read_count;
				
				
				USB->RXCSRL &= ~(ALD_USB_RXCSRL_OVERRUN | ALD_USB_RXCSRL_DATAERR | ALD_USB_RXCSRL_STALL| ALD_USB_RXCSRL_STALLED);
				USB->RXCSRL &= ~USB_RXCSRL_RXRDY_MSK;
            }                                    
            /*else
                USB_LOG_ERR("[ep %d]read_count(%d) > xfer_len(%d)\r\n",ep_idx,read_count,g_musb_udc.out_ep[ep_idx].xfer_len);*/
            
            
            if ((read_count < g_musb_udc.out_ep[ep_idx].ep_mps) || (g_musb_udc.out_ep[ep_idx].xfer_len < g_musb_udc.out_ep[ep_idx].ep_mps)) 
            {
                USB->RXIDR = 1U << ep_idx;
                usbd_event_ep_out_complete_handler(0, ep_idx, g_musb_udc.out_ep[ep_idx].actual_xfer_len);
            }
        }
    }

	if(status & ALD_USB_IFM_RESIFM)
	{
		usbd_event_resume_handler(0);
	}

	if(status & ALD_USB_IFM_SUSPDIFM)
	{
		usbd_event_suspend_handler(0);
	}

    musb_set_active_ep(old_ep_idx);
}
