/**
 * @file lv_img.c
 *
 */

/*********************
 *      INCLUDES
 *********************/
#include "lv_img.h"
#if LV_USE_IMG != 0

#include "../core/lv_disp.h"
#include "../misc/lv_assert.h"
#include "../draw/lv_img_decoder.h"
#include "../misc/lv_fs.h"
#include "../misc/lv_txt.h"
#include "../misc/lv_math.h"
#include "../misc/lv_log.h"

/*********************
 *      DEFINES
 *********************/
#define MY_CLASS &lv_img_class

/**********************
 *      TYPEDEFS
 **********************/

/**********************
 *  STATIC PROTOTYPES
 **********************/
void lv_img_constructor(const lv_obj_class_t * class_p, lv_obj_t * obj);
void lv_img_destructor(const lv_obj_class_t * class_p, lv_obj_t * obj);
void lv_img_event(const lv_obj_class_t * class_p, lv_event_t * e);
void draw_img(lv_event_t * e);

/**********************
 *  STATIC VARIABLES
 **********************/
const lv_obj_class_t lv_img_class = {
    .constructor_cb = lv_img_constructor,
    .destructor_cb = lv_img_destructor,
    .event_cb = lv_img_event,
    .width_def = LV_SIZE_CONTENT,
    .height_def = LV_SIZE_CONTENT,
    .instance_size = sizeof(lv_img_t),
    .base_class = &lv_obj_class
};

/**********************
 *      MACROS
 **********************/

/**********************
 *   GLOBAL FUNCTIONS
 **********************/

lv_obj_t * lv_img_create(lv_obj_t * parent)
{
    LV_LOG_INFO("begin");
    lv_obj_t * obj = lv_obj_class_create_obj(MY_CLASS, parent);
    lv_obj_class_init_obj(obj);
    return obj;
}

/*=====================
 * Setter functions
 *====================*/

//#define CF_BUILT_IN_FIRST   LV_IMG_CF_TRUE_COLOR
//#define CF_BUILT_IN_LAST    LV_IMG_CF_RGB565A8


#define CF_BUILT_IN_FIRST   LV_IMG_CF_TRUE_COLOR
#define CF_BUILT_IN_LAST    LV_IMG_CF_TRUE_COLOR_CHROMA_KEYED

/**
* Get information about an image.
* Try the created image decoder one by one. Once one is able to get info that info will be used.
* @param src the image source. E.g. file name or variable.
* @param header the image info will be stored here
* @return LV_RES_OK: success; LV_RES_INV: wasn't able to get info about the image
*/
lv_res_t lv_img_decoder_get_info(const void * src, lv_img_header_t * header)
{
   lv_memset_00(header, sizeof(lv_img_header_t));

   if(src == NULL) return LV_RES_INV;
   
	lv_img_cf_t cf = ((lv_img_dsc_t *)src)->header.cf;
	
	if(cf < CF_BUILT_IN_FIRST || cf > CF_BUILT_IN_LAST) 
	{
		while(1);
	}

	header->w  = ((lv_img_dsc_t *)src)->header.w;
	header->h  = ((lv_img_dsc_t *)src)->header.h;
	header->cf = ((lv_img_dsc_t *)src)->header.cf;
	
   return LV_RES_OK;
}

void lv_img_set_src(lv_obj_t * obj, const void * src)
{
    LV_ASSERT_OBJ(obj, MY_CLASS);

    lv_obj_invalidate(obj);

    lv_img_t * img = (lv_img_t *)obj;
	
	extern void es_spiflash_gui_ext_mem_get_img_info(lv_img_dsc_t * img1);
	es_spiflash_gui_ext_mem_get_img_info((lv_img_dsc_t *)src);
	
    lv_img_header_t header;
    lv_img_decoder_get_info(src, &header);

    img->src = src;
		
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
    img->pivot.x = header.w / 2;
    img->pivot.y = header.h / 2;

    lv_obj_refresh_self_size(obj);

    /*Provide enough room for the rotated corners*/
    if(img->angle || img->zoom != LV_IMG_ZOOM_NONE) lv_obj_refresh_ext_draw_size(obj);
#endif

    lv_obj_invalidate(obj);
}

void lv_img_set_offset_x(lv_obj_t * obj, lv_coord_t x)
{
#ifdef ES_LV_IMG_USE_OFFSET
    LV_ASSERT_OBJ(obj, MY_CLASS);

    lv_img_t * img = (lv_img_t *)obj;

    x = x % ((lv_img_dsc_t *)(img->src))->header.w;

    img->offset.x = x;
#endif
    lv_obj_invalidate(obj);
}

void lv_img_set_offset_y(lv_obj_t * obj, lv_coord_t y)
{
#ifdef ES_LV_IMG_USE_OFFSET
    LV_ASSERT_OBJ(obj, MY_CLASS);

    lv_img_t * img = (lv_img_t *)obj;

    y = y % ((lv_img_dsc_t *)(img->src))->header.h;

    img->offset.y = y;
    lv_obj_invalidate(obj);
#endif
}

void lv_img_set_angle(lv_obj_t * obj, int16_t angle)
{
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
    if(angle < 0 || angle >= 3600) angle = angle % 3600;

    lv_img_t * img = (lv_img_t *)obj;
    if(angle == img->angle) return;

    lv_obj_update_layout(obj);  /*Be sure the object's size is calculated*/
    lv_coord_t w = lv_obj_get_width(obj);
    lv_coord_t h = lv_obj_get_height(obj);
    lv_area_t a;
    _lv_img_buf_get_transformed_area(&a, w, h, img->angle, img->zoom, &img->pivot);
    a.x1 += obj->coords.x1;
    a.y1 += obj->coords.y1;
    a.x2 += obj->coords.x1;
    a.y2 += obj->coords.y1;
    lv_obj_invalidate_area(obj, &a);

    img->angle = angle;

    /* Disable invalidations because lv_obj_refresh_ext_draw_size would invalidate
     * the whole ext draw area */
#ifdef ES_LV_OBJ_USE_SPECIAL_ATTRIBUTES_EXT_DRAW_SIZE
    lv_disp_t * disp = &g_es_disp_def;
    lv_disp_enable_invalidation(disp, false);
    lv_obj_refresh_ext_draw_size(obj);
    lv_disp_enable_invalidation(disp, true);
#endif

    _lv_img_buf_get_transformed_area(&a, w, h, img->angle, img->zoom, &img->pivot);
    a.x1 += obj->coords.x1;
    a.y1 += obj->coords.y1;
    a.x2 += obj->coords.x1;
    a.y2 += obj->coords.y1;
    lv_obj_invalidate_area(obj, &a);
#endif
}

void lv_img_set_pivot(lv_obj_t * obj, lv_coord_t x, lv_coord_t y)
{
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
    lv_img_t * img = (lv_img_t *)obj;
    if(img->pivot.x == x && img->pivot.y == y) return;

    lv_obj_update_layout(obj);  /*Be sure the object's size is calculated*/
    lv_coord_t w = lv_obj_get_width(obj);
    lv_coord_t h = lv_obj_get_height(obj);
    lv_area_t a;
    _lv_img_buf_get_transformed_area(&a, w, h, img->angle, img->zoom, &img->pivot);
    a.x1 += obj->coords.x1;
    a.y1 += obj->coords.y1;
    a.x2 += obj->coords.x1;
    a.y2 += obj->coords.y1;
    lv_obj_invalidate_area(obj, &a);

    img->pivot.x = x;
    img->pivot.y = y;

    /* Disable invalidations because lv_obj_refresh_ext_draw_size would invalidate
     * the whole ext draw area */
#ifdef ES_LV_OBJ_USE_SPECIAL_ATTRIBUTES_EXT_DRAW_SIZE
    lv_disp_t * disp = &g_es_disp_def;
    lv_disp_enable_invalidation(disp, false);
    lv_obj_refresh_ext_draw_size(obj);
    lv_disp_enable_invalidation(disp, true);
#endif

    _lv_img_buf_get_transformed_area(&a, w, h, img->angle, img->zoom, &img->pivot);
    a.x1 += obj->coords.x1;
    a.y1 += obj->coords.y1;
    a.x2 += obj->coords.x1;
    a.y2 += obj->coords.y1;
    lv_obj_invalidate_area(obj, &a);
#endif
}

void lv_img_set_zoom(lv_obj_t * obj, uint16_t zoom)
{
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
    lv_img_t * img = (lv_img_t *)obj;
    if(zoom == img->zoom) return;

    if(zoom == 0) zoom = 1;

    lv_obj_update_layout(obj);  /*Be sure the object's size is calculated*/
    lv_coord_t w = lv_obj_get_width(obj);
    lv_coord_t h = lv_obj_get_height(obj);
    lv_area_t a;
    _lv_img_buf_get_transformed_area(&a, w, h, img->angle, img->zoom >> 8, &img->pivot);
    a.x1 += obj->coords.x1 - 1;
    a.y1 += obj->coords.y1 - 1;
    a.x2 += obj->coords.x1 + 1;
    a.y2 += obj->coords.y1 + 1;
    lv_obj_invalidate_area(obj, &a);

    img->zoom = zoom;

    /* Disable invalidations because lv_obj_refresh_ext_draw_size would invalidate
     * the whole ext draw area */
#ifdef ES_LV_OBJ_USE_SPECIAL_ATTRIBUTES_EXT_DRAW_SIZE
    lv_disp_t * disp = &g_es_disp_def;
    lv_disp_enable_invalidation(disp, false);
    lv_obj_refresh_ext_draw_size(obj);
    lv_disp_enable_invalidation(disp, true);
#endif

    _lv_img_buf_get_transformed_area(&a, w, h, img->angle, img->zoom, &img->pivot);
    a.x1 += obj->coords.x1 - 1;
    a.y1 += obj->coords.y1 - 1;
    a.x2 += obj->coords.x1 + 1;
    a.y2 += obj->coords.y1 + 1;
    lv_obj_invalidate_area(obj, &a);
#endif
}

void lv_img_set_size_mode(lv_obj_t * obj, lv_img_size_mode_t mode)
{
#ifdef ES_LV_IMG_USE_OBJ_SIZE_MODE
    LV_ASSERT_OBJ(obj, MY_CLASS);
    lv_img_t * img = (lv_img_t *)obj;
    if(mode == img->obj_size_mode) return;

    img->obj_size_mode = mode;
    lv_obj_invalidate(obj);
#endif

}

/*=====================
 * Getter functions
 *====================*/

const void * lv_img_get_src(lv_obj_t * obj)
{
    LV_ASSERT_OBJ(obj, MY_CLASS);

    lv_img_t * img = (lv_img_t *)obj;

    return img->src;
}

lv_coord_t lv_img_get_offset_x(lv_obj_t * obj)
{
#ifdef ES_LV_IMG_USE_OFFSET
    LV_ASSERT_OBJ(obj, MY_CLASS);

    lv_img_t * img = (lv_img_t *)obj;

    return img->offset.x;
#else
return 0;
#endif
}

lv_coord_t lv_img_get_offset_y(lv_obj_t * obj)
{
#ifdef ES_LV_IMG_USE_OFFSET
    LV_ASSERT_OBJ(obj, MY_CLASS);

    lv_img_t * img = (lv_img_t *)obj;

    return img->offset.y;
#else
return 0;
#endif
}

uint16_t lv_img_get_angle(lv_obj_t * obj)
{
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
    LV_ASSERT_OBJ(obj, MY_CLASS);

    lv_img_t * img = (lv_img_t *)obj;

    return img->angle;
#else
return 0;
#endif
}

void lv_img_get_pivot(lv_obj_t * obj, lv_point_t * pivot)
{
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
    LV_ASSERT_OBJ(obj, MY_CLASS);

    lv_img_t * img = (lv_img_t *)obj;

    *pivot = img->pivot;
#endif
}

uint16_t lv_img_get_zoom(lv_obj_t * obj)
{
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
    LV_ASSERT_OBJ(obj, MY_CLASS);

    lv_img_t * img = (lv_img_t *)obj;

    return img->zoom;
#else
return LV_IMG_ZOOM_NONE;
#endif
}

lv_img_size_mode_t lv_img_get_size_mode(lv_obj_t * obj)
{
#ifdef ES_LV_IMG_USE_OBJ_SIZE_MODE
    LV_ASSERT_OBJ(obj, MY_CLASS);
    lv_img_t * img = (lv_img_t *)obj;
    return img->obj_size_mode;
#else
return LV_IMG_SIZE_MODE_VIRTUAL;
#endif
}

/**********************
 *   STATIC FUNCTIONS
 **********************/

void lv_img_constructor(const lv_obj_class_t * class_p, lv_obj_t * obj)
{
    LV_UNUSED(class_p);
    LV_TRACE_OBJ_CREATE("begin");

    lv_img_t * img = (lv_img_t *)obj;

    img->src       = NULL;
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
    img->angle = 0;
    img->zoom = LV_IMG_ZOOM_NONE;
    img->pivot.x = 0;
    img->pivot.y = 0;
#endif
#ifdef ES_LV_IMG_USE_OFFSET
    img->offset.x  = 0;
    img->offset.y  = 0;
#endif
#ifdef ES_LV_IMG_USE_OBJ_SIZE_MODE
    img->obj_size_mode = LV_IMG_SIZE_MODE_VIRTUAL;
#endif

    lv_obj_clear_flag(obj, LV_OBJ_FLAG_CLICKABLE);
    lv_obj_add_flag(obj, LV_OBJ_FLAG_ADV_HITTEST);

    LV_TRACE_OBJ_CREATE("finished");
}

void lv_img_destructor(const lv_obj_class_t * class_p, lv_obj_t * obj)
{
    LV_UNUSED(class_p);
}

#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
lv_point_t lv_img_get_transformed_size(lv_obj_t * obj)
{
    lv_img_t * img = (lv_img_t *)obj;


    lv_area_t area_transform;
    _lv_img_buf_get_transformed_area(&area_transform, ((lv_img_dsc_t *)(img->src))->header.w, ((lv_img_dsc_t *)(img->src))->header.h,
                                     img->angle, img->zoom, &img->pivot);

    return (lv_point_t) {
        lv_area_get_width(&area_transform), lv_area_get_height(&area_transform)
    };
}
#endif

void lv_img_event(const lv_obj_class_t * class_p, lv_event_t * e)
{
    LV_UNUSED(class_p);

    lv_event_code_t code = lv_event_get_code(e);

    /*Ancestor events will be called during drawing*/
    if(code != LV_EVENT_DRAW_MAIN && code != LV_EVENT_DRAW_POST) {
        /*Call the ancestor's event handler*/
        lv_res_t res = lv_obj_event_base(MY_CLASS, e);
        if(res != LV_RES_OK) return;
    }

    lv_obj_t * obj = lv_event_get_target(e);
    lv_img_t * img = (lv_img_t *)obj;

    if(code == LV_EVENT_STYLE_CHANGED) {
            lv_obj_refresh_ext_draw_size(obj);
    }
    else if(code == LV_EVENT_REFR_EXT_DRAW_SIZE) {

#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
        lv_coord_t * s = lv_event_get_param(e);

        /*If the image has angle provide enough room for the rotated corners*/
        if(img->angle || img->zoom != LV_IMG_ZOOM_NONE) {
            lv_area_t a;
            lv_coord_t w = lv_obj_get_width(obj);
            lv_coord_t h = lv_obj_get_height(obj);
            _lv_img_buf_get_transformed_area(&a, w, h, img->angle, img->zoom, &img->pivot);
            *s = LV_MAX(*s, -a.x1);
            *s = LV_MAX(*s, -a.y1);
            *s = LV_MAX(*s, a.x2 - w);
            *s = LV_MAX(*s, a.y2 - h);
        }
#endif

    }
    else if(code == LV_EVENT_HIT_TEST) {
        lv_hit_test_info_t * info = lv_event_get_param(e);

#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
        /*If the object is exactly image sized (not cropped, not mosaic) and transformed
         *perform hit test on its transformed area*/
        if(((lv_img_dsc_t *)(img->src))->header.w == lv_obj_get_width(obj) && ((lv_img_dsc_t *)(img->src))->header.h == lv_obj_get_height(obj) &&
           (img->zoom != LV_IMG_ZOOM_NONE || img->angle != 0 || img->pivot.x != ((lv_img_dsc_t *)(img->src))->header.w / 2 || img->pivot.y != ((lv_img_dsc_t *)(img->src))->header.h / 2)) {

            lv_coord_t w = lv_obj_get_width(obj);
            lv_coord_t h = lv_obj_get_height(obj);
            lv_area_t coords;
            _lv_img_buf_get_transformed_area(&coords, w, h, img->angle, img->zoom, &img->pivot);
            coords.x1 += obj->coords.x1;
            coords.y1 += obj->coords.y1;
            coords.x2 += obj->coords.x1;
            coords.y2 += obj->coords.y1;

            info->res = _lv_area_is_point_on(&coords, info->point, 0);
        }
        else {
#endif
            lv_area_t a;
            lv_obj_get_click_area(obj, &a);
            info->res = _lv_area_is_point_on(&a, info->point, 0);
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
        }
#endif
    }
    else if(code == LV_EVENT_GET_SELF_SIZE) {
        lv_point_t * p = lv_event_get_param(e);
#ifdef ES_LV_IMG_USE_OBJ_SIZE_MODE
        if(img->obj_size_mode == LV_IMG_SIZE_MODE_REAL) {
            *p = lv_img_get_transformed_size(obj);
        }
        else {
#endif
            p->x = ((lv_img_dsc_t *)(img->src))->header.w;
            p->y = ((lv_img_dsc_t *)(img->src))->header.h;
#ifdef ES_LV_IMG_USE_OBJ_SIZE_MODE
        }
#endif
    }
    else if(code == LV_EVENT_DRAW_MAIN || code == LV_EVENT_DRAW_POST || code == LV_EVENT_COVER_CHECK) {
        draw_img(e);
    }
}

void draw_img(lv_event_t * e)
{
    lv_event_code_t code = lv_event_get_code(e);
    lv_obj_t * obj = lv_event_get_target(e);
    lv_img_t * img = (lv_img_t *)obj;
    if(code == LV_EVENT_COVER_CHECK) {
        lv_cover_check_info_t * info = lv_event_get_param(e);
        if(info->res == LV_COVER_RES_MASKED) return;

        /*Non true color format might have "holes"*/
        if(((lv_img_dsc_t *)(img->src))->header.cf != LV_IMG_CF_TRUE_COLOR) {
            info->res = LV_COVER_RES_NOT_COVER;
            return;
        }

        /*With not LV_OPA_COVER images can't cover an area */
        if(lv_obj_get_style_img_opa(obj, LV_PART_MAIN) != LV_OPA_COVER) {
            info->res = LV_COVER_RES_NOT_COVER;
            return;
        }

#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
        if(img->angle != 0) {
            info->res = LV_COVER_RES_NOT_COVER;
            return;
        }
#endif

        const lv_area_t * clip_area = lv_event_get_param(e);
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
        if(img->zoom == LV_IMG_ZOOM_NONE) {
#endif
            if(_lv_area_is_in(clip_area, &obj->coords, 0) == false) {
                info->res = LV_COVER_RES_NOT_COVER;
                return;
            }
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
        }
        else {
            lv_area_t a;
            _lv_img_buf_get_transformed_area(&a, lv_obj_get_width(obj), lv_obj_get_height(obj), 0, img->zoom, &img->pivot);
            a.x1 += obj->coords.x1;
            a.y1 += obj->coords.y1;
            a.x2 += obj->coords.x1;
            a.y2 += obj->coords.y1;

            if(_lv_area_is_in(clip_area, &a, 0) == false) {
                info->res = LV_COVER_RES_NOT_COVER;
                return;
            }
        }
#endif
    }
    else if(code == LV_EVENT_DRAW_MAIN || code == LV_EVENT_DRAW_POST) {

        lv_coord_t obj_w = lv_obj_get_width(obj);
        lv_coord_t obj_h = lv_obj_get_height(obj);

#ifdef ES_LV_CONF_OPTIMIZE_USE_BORDER
        lv_coord_t border_width = lv_obj_get_style_border_width(obj, LV_PART_MAIN);
        lv_coord_t pleft = lv_obj_get_style_pad_left(obj, LV_PART_MAIN) + border_width;
        lv_coord_t pright = lv_obj_get_style_pad_right(obj, LV_PART_MAIN) + border_width;
        lv_coord_t ptop = lv_obj_get_style_pad_top(obj, LV_PART_MAIN) + border_width;
        lv_coord_t pbottom = lv_obj_get_style_pad_bottom(obj, LV_PART_MAIN) + border_width;
#else
        lv_coord_t pleft = lv_obj_get_style_pad_left(obj, LV_PART_MAIN);
        lv_coord_t pright = lv_obj_get_style_pad_right(obj, LV_PART_MAIN);
        lv_coord_t ptop = lv_obj_get_style_pad_top(obj, LV_PART_MAIN);
        lv_coord_t pbottom = lv_obj_get_style_pad_bottom(obj, LV_PART_MAIN);
#endif

#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
        lv_point_t bg_pivot;
        bg_pivot.x = img->pivot.x + pleft;
        bg_pivot.y = img->pivot.y + ptop;
#endif
        lv_area_t bg_coords;

#ifdef ES_LV_IMG_USE_OBJ_SIZE_MODE
        if(img->obj_size_mode == LV_IMG_SIZE_MODE_REAL) {
#endif
            /*Object size equals to transformed image size*/
            lv_obj_get_coords(obj, &bg_coords);
#ifdef ES_LV_IMG_USE_OBJ_SIZE_MODE
        }
        else {
            _lv_img_buf_get_transformed_area(&bg_coords, obj_w, obj_h,
                                             img->angle, img->zoom, &bg_pivot);

            /*Modify the coordinates to draw the background for the rotated and scaled coordinates*/
            bg_coords.x1 += obj->coords.x1;
            bg_coords.y1 += obj->coords.y1;
            bg_coords.x2 += obj->coords.x1;
            bg_coords.y2 += obj->coords.y1;
        }
#endif

        lv_area_t ori_coords;
        lv_area_copy(&ori_coords, &obj->coords);
        lv_area_copy(&obj->coords, &bg_coords);

        lv_res_t res = lv_obj_event_base(MY_CLASS, e);
        if(res != LV_RES_OK) return;

        lv_area_copy(&obj->coords, &ori_coords);

        if(code == LV_EVENT_DRAW_MAIN) {
//            if(img->h == 0 || img->w == 0) return;
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
            if(img->zoom == 0) return;
#endif

            lv_draw_ctx_t * draw_ctx = lv_event_get_draw_ctx(e);

            lv_area_t img_max_area;
            lv_area_copy(&img_max_area, &obj->coords);

#ifdef ES_LV_IMG_USE_OBJ_SIZE_MODE
            lv_point_t img_size_final = lv_img_get_transformed_size(obj);

            if(img->obj_size_mode == LV_IMG_SIZE_MODE_REAL) {
                img_max_area.x1 -= ((((lv_img_dsc_t *)(img->src))->header.w - img_size_final.x) + 1) / 2;
                img_max_area.x2 -= ((((lv_img_dsc_t *)(img->src))->header.w - img_size_final.x) + 1) / 2;
                img_max_area.y1 -= ((((lv_img_dsc_t *)(img->src))->header.h - img_size_final.y) + 1) / 2;
                img_max_area.y2 -= ((((lv_img_dsc_t *)(img->src))->header.h - img_size_final.y) + 1) / 2;
            }
            else {
#endif

                img_max_area.x2 = img_max_area.x1 + lv_area_get_width(&bg_coords) - 1;
                img_max_area.y2 = img_max_area.y1 + lv_area_get_height(&bg_coords) - 1;
#ifdef ES_LV_IMG_USE_OBJ_SIZE_MODE
            }
#endif

            img_max_area.x1 += pleft;
            img_max_area.y1 += ptop;
            img_max_area.x2 -= pright;
            img_max_area.y2 -= pbottom;

                lv_draw_img_dsc_t img_dsc;
                lv_draw_img_dsc_init(&img_dsc);
                lv_obj_init_draw_img_dsc(obj, LV_PART_MAIN, &img_dsc);

#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
                img_dsc.zoom = img->zoom;
                img_dsc.angle = img->angle;
                img_dsc.pivot.x = img->pivot.x;
                img_dsc.pivot.y = img->pivot.y;
#endif

                lv_area_t img_clip_area;
                img_clip_area.x1 = bg_coords.x1 + pleft;
                img_clip_area.y1 = bg_coords.y1 + ptop;
                img_clip_area.x2 = bg_coords.x2 - pright;
                img_clip_area.y2 = bg_coords.y2 - pbottom;
                const lv_area_t * clip_area_ori = draw_ctx->clip_area;

                if(!_lv_area_intersect(&img_clip_area, draw_ctx->clip_area, &img_clip_area)) return;
                draw_ctx->clip_area = &img_clip_area;

                lv_area_t coords_tmp;
#ifdef ES_LV_IMG_USE_OFFSET
                coords_tmp.y1 = img_max_area.y1 + img->offset.y;
#else
                coords_tmp.y1 = img_max_area.y1;
#endif
                if(coords_tmp.y1 > img_max_area.y1) coords_tmp.y1 -= ((lv_img_dsc_t *)(img->src))->header.h;
                coords_tmp.y2 = coords_tmp.y1 + ((lv_img_dsc_t *)(img->src))->header.h - 1;

#ifdef ES_LV_IMG_USE_OBJ_SIZE_MODE
                for(; coords_tmp.y1 < img_max_area.y2; coords_tmp.y1 += img_size_final.y, coords_tmp.y2 += img_size_final.y) {
#else
                for(; coords_tmp.y1 < img_max_area.y2; coords_tmp.y1 += ((lv_img_dsc_t *)(img->src))->header.h, coords_tmp.y2 += ((lv_img_dsc_t *)(img->src))->header.h) {
#endif
#ifdef ES_LV_IMG_USE_OFFSET
                    coords_tmp.x1 = img_max_area.x1 + img->offset.x;
#else
                    coords_tmp.x1 = img_max_area.x1;
#endif
                    if(coords_tmp.x1 > img_max_area.x1) coords_tmp.x1 -= ((lv_img_dsc_t *)(img->src))->header.w;
                    coords_tmp.x2 = coords_tmp.x1 + ((lv_img_dsc_t *)(img->src))->header.w - 1;

#ifdef ES_LV_IMG_USE_OBJ_SIZE_MODE
                    for(; coords_tmp.x1 < img_max_area.x2; coords_tmp.x1 += img_size_final.x, coords_tmp.x2 += img_size_final.x) {
#else
                    for(; coords_tmp.x1 < img_max_area.x2; coords_tmp.x1 += ((lv_img_dsc_t *)(img->src))->header.w, coords_tmp.x2 += ((lv_img_dsc_t *)(img->src))->header.w) {
#endif
                        lv_draw_img(draw_ctx, &img_dsc, &coords_tmp, img->src);
                    }
                }
                draw_ctx->clip_area = clip_area_ori;
        }
    }
}

#endif
