/**
 * @file lv_refr.c
 *
 */

/*********************
 *      INCLUDES
 *********************/
#include <stddef.h>
#include "lv_refr.h"
#include "lv_disp.h"
#include "../hal/lv_hal_tick.h"
#include "../hal/lv_hal_disp.h"
#include "../misc/lv_timer.h"
#include "../misc/lv_mem.h"
#include "../misc/lv_math.h"
#include "../misc/lv_gc.h"
#include "../draw/lv_draw.h"
#include "../font/lv_font_fmt_txt.h"
#include "../extra/others/snapshot/lv_snapshot.h"

/*********************
 *      DEFINES
 *********************/

/**********************
 *      TYPEDEFS
 **********************/

/**********************
 *  STATIC PROTOTYPES
 **********************/
static void refr_area_part(lv_draw_ctx_t * draw_ctx);
static lv_obj_t * lv_refr_get_top_obj(const lv_area_t * area_p, lv_obj_t * obj);
static void refr_obj_and_children(lv_draw_ctx_t * draw_ctx, lv_obj_t * top_obj);
static void refr_obj(lv_draw_ctx_t * draw_ctx, lv_obj_t * obj);

/**********************
 *  STATIC VARIABLES
 **********************/
//static uint32_t px_num;

	extern lv_disp_t g_es_disp_def;
/**********************
 *      MACROS
 **********************/
#if LV_LOG_TRACE_DISP_REFR
    #define REFR_TRACE(...) LV_LOG_TRACE(__VA_ARGS__)
#else
    #define REFR_TRACE(...)
#endif

/**********************
 *   GLOBAL FUNCTIONS
 **********************/

/**
 * Initialize the screen refresh subsystem
 */


void lv_refr_now(lv_disp_t * disp)
{
    lv_anim_refr_now();
	
    _lv_disp_refr_timer();
}

void lv_obj_redraw(lv_draw_ctx_t * draw_ctx, lv_obj_t * obj)
{
    const lv_area_t * clip_area_ori = draw_ctx->clip_area;
    lv_area_t clip_coords_for_obj;

    /*Truncate the clip area to `obj size + ext size` area*/
    lv_area_t obj_coords_ext;
    lv_obj_get_coords(obj, &obj_coords_ext);
#ifdef ES_LV_OBJ_USE_SPECIAL_ATTRIBUTES_EXT_DRAW_SIZE
    lv_coord_t ext_draw_size = _lv_obj_get_ext_draw_size(obj);
    lv_area_increase(&obj_coords_ext, ext_draw_size, ext_draw_size);
#endif
    bool com_clip_res = _lv_area_intersect(&clip_coords_for_obj, clip_area_ori, &obj_coords_ext);
    if(com_clip_res) {
        draw_ctx->clip_area = &clip_coords_for_obj;

        lv_event_send(obj, LV_EVENT_DRAW_MAIN, draw_ctx);
    }

    /*With overflow visible keep the previous clip area to let the children visible out of this object too
     *With not overflow visible limit the clip are to the object's coordinates to clip the children*/
    lv_area_t clip_coords_for_children;
    bool refr_children = true;
	
        if(!_lv_area_intersect(&clip_coords_for_children, clip_area_ori, &obj->coords)) {
            refr_children = false;
        }

    if(refr_children) {
        draw_ctx->clip_area = &clip_coords_for_children;
        uint32_t i;
        uint32_t child_cnt = lv_obj_get_child_cnt(obj);
        for(i = 0; i < child_cnt; i++) {
            lv_obj_t * child = obj->spec_attr->children[i];
            refr_obj(draw_ctx, child);
        }
    }

    /*If the object was visible on the clip area call the post draw events too*/
    if(com_clip_res) {
        draw_ctx->clip_area = &clip_coords_for_obj;

        /*If all the children are redrawn make 'post draw' draw*/
        lv_event_send(obj, LV_EVENT_DRAW_POST, draw_ctx);
    }

    draw_ctx->clip_area = clip_area_ori;
}


/**
 * Invalidate an area on display to redraw it
 * @param area_p pointer to area which should be invalidated (NULL: delete the invalidated areas)
 * @param disp pointer to display where the area should be invalidated (NULL can be used if there is
 * only one display)
 */
void _lv_inv_area(const lv_area_t * area_p)
{
    lv_disp_t * disp = &g_es_disp_def;
    
    /*Clear the invalidate buffer if the parameter is NULL*/
    

    lv_area_t scr_area;
    scr_area.x1 = 0;
    scr_area.y1 = 0;
    scr_area.x2 = g_es_disp_def.driver.hor_res - 1;
    scr_area.y2 = g_es_disp_def.driver.ver_res - 1;

    lv_area_t com_area;
    bool suc;

    suc = _lv_area_intersect(&com_area, area_p, &scr_area);
    if(suc == false)  return; /*Out of the screen*/

	es_lv_disable_irq();

    /*Save only if this area is not in one of the saved areas*/
    uint16_t i;
    for(i = 0; i < disp->inv_p; i++) {
        if(_lv_area_is_in(&com_area, &disp->inv_areas[i], 0) != false) return;
    }

    /*Save the area*/
    if(disp->inv_p < LV_INV_BUF_SIZE) {
        lv_area_copy(&disp->inv_areas[disp->inv_p], &com_area);
    }
    else {   /*If no place for the area add the screen*/
        disp->inv_p = 0;
        lv_area_copy(&disp->inv_areas[0], &scr_area);
    }
    disp->inv_p++;
	
	disp->refr_timer_pause = 0;
	
	es_lv_enable_irq();

}

/**
 * Get the display which is being refreshed
 * @return the display being refreshed
 */
lv_disp_t * _lv_refr_get_disp_refreshing(void)
{
	extern lv_disp_t g_es_disp_def;
    return &g_es_disp_def;
}

/**
 * Set the display which is being refreshed.
 * It shouldn't be used directly by the user.
 * It can be used to trick the drawing functions about there is an active display.
 * @param the display being refreshed
 */
void _lv_refr_set_disp_refreshing(lv_disp_t * disp)
{
	
}

/**
 * Called periodically to handle the refreshing
 * @param tmr pointer to the timer itself
 */
void _lv_disp_refr_timer(void)
{
    int32_t i;
    uint32_t join_from;
    uint32_t join_in;
    lv_area_t joined_area;

	g_es_disp_def.refr_timer_pause = 1;

    /*Refresh the screen's layout if required*/
    lv_obj_update_layout(g_es_disp_def.act_scr);
    if(g_es_disp_def.prev_scr) lv_obj_update_layout(g_es_disp_def.prev_scr);

    lv_obj_update_layout(g_es_disp_def.top_layer);
    lv_obj_update_layout(g_es_disp_def.sys_layer);

    /*Do nothing if there is no active screen*/
    if(g_es_disp_def.act_scr == NULL) {
        g_es_disp_def.inv_p = 0;
        return;
    }

    for(join_in = 0; join_in < g_es_disp_def.inv_p; join_in++) {
        if(g_es_disp_def.inv_area_joined[join_in] != 0) continue;

        /*Check all areas to join them in 'join_in'*/
        for(join_from = 0; join_from < g_es_disp_def.inv_p; join_from++) {
            /*Handle only unjoined areas and ignore itself*/
            if(g_es_disp_def.inv_area_joined[join_from] != 0 || join_in == join_from) {
                continue;
            }

            /*Check if the areas are on each other*/
            if(_lv_area_is_on(&g_es_disp_def.inv_areas[join_in], &g_es_disp_def.inv_areas[join_from]) == false) {
                continue;
            }

            _lv_area_join(&joined_area, &g_es_disp_def.inv_areas[join_in], &g_es_disp_def.inv_areas[join_from]);

            /*Join two area only if the joined area size is smaller*/
            if(lv_area_get_size(&joined_area) < (lv_area_get_size(&g_es_disp_def.inv_areas[join_in]) +
                                                 lv_area_get_size(&g_es_disp_def.inv_areas[join_from]))) {
                lv_area_copy(&g_es_disp_def.inv_areas[join_in], &joined_area);

                /*Mark 'join_form' is joined into 'join_in'*/
                g_es_disp_def.inv_area_joined[join_from] = 1;
            }
        }
    }
	
    if(g_es_disp_def.inv_p)
	{

		/*Notify the display driven rendering has started*/
		es_lv_disable_irq();

		for(i = 0; i < g_es_disp_def.inv_p; i++) {
			/*Refresh the unjoined areas*/
			if(g_es_disp_def.inv_area_joined[i] == 0) {
				
				lv_area_t * area_p = &g_es_disp_def.inv_areas[i];
				
				lv_draw_ctx_t * draw_ctx = &g_es_disp_def.driver.draw_ctx;
				draw_ctx->buf = g_es_disp_def.driver.draw_buf->buf_act;

				/*Normal refresh: draw the area in parts*/
				/*Calculate the max row num*/
				lv_coord_t w = lv_area_get_width(area_p);
				lv_coord_t h = lv_area_get_height(area_p);
				lv_coord_t y2 = area_p->y2 >= g_es_disp_def.driver.ver_res ?
								g_es_disp_def.driver.ver_res - 1 : area_p->y2;

				int32_t max_row = (uint32_t)g_es_disp_def.driver.draw_buf->size / w;

				if(max_row > h) max_row = h;

				lv_coord_t row;
				lv_coord_t row_last = 0;
				lv_area_t sub_area;
				for(row = area_p->y1; row + max_row - 1 <= y2; row += max_row) {
					/*Calc. the next y coordinates of draw_buf*/
					sub_area.x1 = area_p->x1;
					sub_area.x2 = area_p->x2;
					sub_area.y1 = row;
					sub_area.y2 = row + max_row - 1;
					draw_ctx->buf_area = &sub_area;
					draw_ctx->clip_area = &sub_area;
					draw_ctx->buf = g_es_disp_def.driver.draw_buf->buf_act;
					if(sub_area.y2 > y2) sub_area.y2 = y2;
					row_last = sub_area.y2;
					refr_area_part(draw_ctx);
				}

				/*If the last y coordinates are not handled yet ...*/
				if(y2 != row_last) {
					/*Calc. the next y coordinates of draw_buf*/
					sub_area.x1 = area_p->x1;
					sub_area.x2 = area_p->x2;
					sub_area.y1 = row;
					sub_area.y2 = y2;
					draw_ctx->buf_area = &sub_area;
					draw_ctx->clip_area = &sub_area;
					draw_ctx->buf = g_es_disp_def.driver.draw_buf->buf_act;
					refr_area_part(draw_ctx);
				}
			}
		}

		es_lv_enable_irq();
		
		/*If refresh happened ...*/
		/*Clean up*/
		lv_memset_00(g_es_disp_def.inv_areas, sizeof(g_es_disp_def.inv_areas));
		lv_memset_00(g_es_disp_def.inv_area_joined, sizeof(g_es_disp_def.inv_area_joined));
		
		g_es_disp_def.inv_p = 0;
	}

    _lv_font_clean_up_fmt_txt();

#if LV_DRAW_COMPLEX
    _lv_draw_mask_cleanup();
#endif

    REFR_TRACE("finished");
}

/**********************
 *   STATIC FUNCTIONS
 **********************/

extern void es_lv_lcd_write_area(lv_disp_drv_t *disp_drv, const lv_area_t *area, lv_color_t *color_p);

static void refr_area_part(lv_draw_ctx_t * draw_ctx)
{
    /* Below the `area_p` area will be redrawn into the draw buffer.
     * In single buffered mode wait here until the buffer is freed.
     * In full double buffered mode wait here while the buffers are swapped and a buffer becomes available*/
    if((g_es_disp_def.driver.draw_buf->buf1 && !g_es_disp_def.driver.draw_buf->buf2)) {
        while(g_es_disp_def.driver.draw_buf->flushing) {
			extern void es_lv_wait_flushing(void);
			es_lv_wait_flushing();
        }

        /*If the screen is transparent initialize it when the flushing is ready*/
#if LV_COLOR_SCREEN_TRANSP
        if(g_es_disp_def.driver.screen_transp) {
                lv_memset_00(g_es_disp_def.driver.draw_buf->buf_act, g_es_disp_def.driver.draw_buf->size * LV_IMG_PX_SIZE_ALPHA_BYTE);
        }
#endif
    }

    lv_obj_t * top_act_scr = NULL;
    lv_obj_t * top_prev_scr = NULL;

    /*Get the most top object which is not covered by others*/
    top_act_scr = lv_refr_get_top_obj(draw_ctx->buf_area, g_es_disp_def.act_scr);
    if(g_es_disp_def.prev_scr) {
        top_prev_scr = lv_refr_get_top_obj(draw_ctx->buf_area, g_es_disp_def.prev_scr);
    }

    /*Draw a display background if there is no top object*/
    if(top_act_scr == NULL && top_prev_scr == NULL) {
		//while(1);
		
          lv_draw_rect_dsc_t dsc;
            lv_draw_rect_dsc_init(&dsc);
//            dsc.bg_color.full = 0xFFFF;
//            dsc.bg_opa = 255;
            lv_draw_rect(draw_ctx, &dsc, draw_ctx->buf_area);
		
    }

        /*Refresh the previous screen if any*/
        if(g_es_disp_def.prev_scr) {
            if(top_prev_scr == NULL) top_prev_scr = g_es_disp_def.prev_scr;
            refr_obj_and_children(draw_ctx, top_prev_scr);
        }

        if(top_act_scr == NULL) top_act_scr = g_es_disp_def.act_scr;
        refr_obj_and_children(draw_ctx, top_act_scr);

    /*Also refresh top and sys layer unconditionally*/
    refr_obj_and_children(draw_ctx, g_es_disp_def.top_layer);
    refr_obj_and_children(draw_ctx, g_es_disp_def.sys_layer);

    /* In partial double buffered mode wait until the other buffer is freed
     * and driver is ready to receive the new buffer */
    if(g_es_disp_def.driver.draw_buf->buf1 && g_es_disp_def.driver.draw_buf->buf2) {
        while(g_es_disp_def.driver.draw_buf->flushing) {
			extern void es_lv_wait_flushing(void);
			es_lv_wait_flushing();
        }
    }

    g_es_disp_def.driver.draw_buf->flushing = 1;

	es_lv_lcd_write_area(&g_es_disp_def.driver, g_es_disp_def.driver.draw_ctx.buf_area, g_es_disp_def.driver.draw_ctx.buf);

    /*If there are 2 buffers swap them. With direct mode swap only on the last area*/
    if(g_es_disp_def.driver.draw_buf->buf1 && g_es_disp_def.driver.draw_buf->buf2) {
        if(g_es_disp_def.driver.draw_buf->buf_act == g_es_disp_def.driver.draw_buf->buf1)
            g_es_disp_def.driver.draw_buf->buf_act = g_es_disp_def.driver.draw_buf->buf2;
        else
            g_es_disp_def.driver.draw_buf->buf_act = g_es_disp_def.driver.draw_buf->buf1;
    }
}

/**
 * Search the most top object which fully covers an area
 * @param area_p pointer to an area
 * @param obj the first object to start the searching (typically a screen)
 * @return
 */
static lv_obj_t * lv_refr_get_top_obj(const lv_area_t * area_p, lv_obj_t * obj)
{
    lv_obj_t * found_p = NULL;

    if(_lv_area_is_in(area_p, &obj->coords, 0) == false) return NULL;
    if(lv_obj_has_flag(obj, LV_OBJ_FLAG_HIDDEN)) return NULL;
    if(_lv_obj_get_layer_type(obj) != LV_LAYER_TYPE_NONE) return NULL;

    /*If this object is fully cover the draw area then check the children too*/
    lv_cover_check_info_t info;
    info.res = LV_COVER_RES_COVER;
    info.area = area_p;
    lv_event_send(obj, LV_EVENT_COVER_CHECK, &info);
    if(info.res == LV_COVER_RES_MASKED) return NULL;

    int32_t i;
    int32_t child_cnt = lv_obj_get_child_cnt(obj);
    for(i = child_cnt - 1; i >= 0; i--) {
        lv_obj_t * child = obj->spec_attr->children[i];
        found_p = lv_refr_get_top_obj(area_p, child);

        /*If a children is ok then break*/
        if(found_p != NULL) {
            break;
        }
    }

    /*If no better children use this object*/
    if(found_p == NULL && info.res == LV_COVER_RES_COVER) {
        found_p = obj;
    }

    return found_p;
}

/**
 * Make the refreshing from an object. Draw all its children and the youngers too.
 * @param top_p pointer to an objects. Start the drawing from it.
 * @param mask_p pointer to an area, the objects will be drawn only here
 */
static void refr_obj_and_children(lv_draw_ctx_t * draw_ctx, lv_obj_t * top_obj)
{
    /*Normally always will be a top_obj (at least the screen)
     *but in special cases (e.g. if the screen has alpha) it won't.
     *In this case use the screen directly*/
    if(top_obj == NULL) top_obj = g_es_disp_def.act_scr;

    /*Refresh the top object and its children*/
    refr_obj(draw_ctx, top_obj);

    /*Draw the 'younger' sibling objects because they can be on top_obj*/
    lv_obj_t * parent;
    lv_obj_t * border_p = top_obj;

    parent = lv_obj_get_parent(top_obj);

    /*Do until not reach the screen*/
    while(parent != NULL) {
        bool go = false;
        uint32_t i;
        uint32_t child_cnt = lv_obj_get_child_cnt(parent);
        for(i = 0; i < child_cnt; i++) {
            lv_obj_t * child = parent->spec_attr->children[i];
            if(!go) {
                if(child == border_p) go = true;
            }
            else {
                /*Refresh the objects*/
                refr_obj(draw_ctx, child);
            }
        }

        /*Call the post draw draw function of the parents of the to object*/
        lv_event_send(parent, LV_EVENT_DRAW_POST, (void *)draw_ctx);

        /*The new border will be the last parents,
         *so the 'younger' brothers of parent will be refreshed*/
        border_p = parent;
        /*Go a level deeper*/
        parent = lv_obj_get_parent(parent);
    }
}


#include "lv_draw_sw.h"

void refr_obj(lv_draw_ctx_t * draw_ctx, lv_obj_t * obj)
{
    /*Do not refresh hidden objects*/
    if(lv_obj_has_flag(obj, LV_OBJ_FLAG_HIDDEN)) return;
    lv_layer_type_t layer_type = _lv_obj_get_layer_type(obj);
    if(layer_type == LV_LAYER_TYPE_NONE) {
        lv_obj_redraw(draw_ctx, obj);
    }
    else {
        lv_opa_t opa = lv_obj_get_style_opa(obj, 0);
        if(opa < LV_OPA_MIN) return;

        lv_area_t layer_area_full;
#ifdef ES_LV_OBJ_USE_SPECIAL_ATTRIBUTES_EXT_DRAW_SIZE
		lv_coord_t ext_draw_size = _lv_obj_get_ext_draw_size(obj);
#endif
		lv_area_t obj_coords_ext;
		lv_obj_get_coords(obj, &obj_coords_ext);
#ifdef ES_LV_OBJ_USE_SPECIAL_ATTRIBUTES_EXT_DRAW_SIZE
		lv_area_increase(&obj_coords_ext, ext_draw_size, ext_draw_size);
#endif

		if(layer_type == LV_LAYER_TYPE_TRANSFORM) {
			/*Get the transformed area and clip it to the current clip area.
			 *This area needs to be updated on the screen.*/
			lv_area_t clip_coords_for_obj;
			lv_area_t tranf_coords = obj_coords_ext;
			lv_obj_get_transformed_area(obj, &tranf_coords, false, false);
			if(!_lv_area_intersect(&clip_coords_for_obj, draw_ctx->clip_area, &tranf_coords)) {
				return;
			}

			/*Transform back (inverse) the transformed area.
			 *It will tell which area of the non-transformed widget needs to be redrawn
			 *in order to cover transformed area after transformation.*/
			lv_area_t inverse_clip_coords_for_obj = clip_coords_for_obj;
			lv_obj_get_transformed_area(obj, &inverse_clip_coords_for_obj, false, true);
			if(!_lv_area_intersect(&inverse_clip_coords_for_obj, &inverse_clip_coords_for_obj, &obj_coords_ext)) {
				return;
			}

			layer_area_full = inverse_clip_coords_for_obj;
		}
		else if(layer_type == LV_LAYER_TYPE_SIMPLE) {
			lv_area_t clip_coords_for_obj;
			if(!_lv_area_intersect(&clip_coords_for_obj, draw_ctx->clip_area, &obj_coords_ext)) {
				return;
			}
			layer_area_full = clip_coords_for_obj;
		}
		else {
			return;
		}

        lv_draw_layer_flags_t flags = LV_DRAW_LAYER_FLAG_HAS_ALPHA;

        if(_lv_area_is_in(&layer_area_full, &obj->coords, 0)) {
            lv_cover_check_info_t info;
            info.res = LV_COVER_RES_COVER;
            info.area = &layer_area_full;
            lv_event_send(obj, LV_EVENT_COVER_CHECK, &info);
            if(info.res == LV_COVER_RES_COVER) flags &= ~LV_DRAW_LAYER_FLAG_HAS_ALPHA;
        }

        if(layer_type == LV_LAYER_TYPE_SIMPLE) flags |= LV_DRAW_LAYER_FLAG_CAN_SUBDIVIDE;

		lv_draw_sw_layer_ctx_t stack_layer_ctx;
	
		lv_draw_layer_ctx_t * layer_ctx = (lv_draw_layer_ctx_t *)(&stack_layer_ctx);
		lv_memset_00(layer_ctx, sizeof(lv_draw_sw_layer_ctx_t));

		layer_ctx->original.buf = draw_ctx->buf;
		layer_ctx->original.buf_area = draw_ctx->buf_area;
		layer_ctx->original.clip_area = draw_ctx->clip_area;
		layer_ctx->original.screen_transp = g_es_disp_def.driver.screen_transp;
		layer_ctx->area_full = layer_area_full;

#if LV_COLOR_SCREEN_TRANSP == 0
		if((flags & LV_DRAW_LAYER_FLAG_HAS_ALPHA)) {
			return;
		}
#endif
		lv_draw_sw_layer_ctx_t * layer_sw_ctx = (lv_draw_sw_layer_ctx_t *) layer_ctx;
		uint32_t px_size = flags & LV_DRAW_LAYER_FLAG_HAS_ALPHA ? LV_IMG_PX_SIZE_ALPHA_BYTE : sizeof(lv_color_t);
		
		if(flags & LV_DRAW_LAYER_FLAG_CAN_SUBDIVIDE) {
			layer_sw_ctx->buf_size_bytes = LV_LAYER_SIMPLE_BUF_SIZE;
			uint32_t full_size = lv_area_get_size(&layer_sw_ctx->base_draw.area_full) * px_size;
			if(layer_sw_ctx->buf_size_bytes > full_size) layer_sw_ctx->buf_size_bytes = full_size;
		}
		else {
	        layer_sw_ctx->base_draw.area_act = layer_sw_ctx->base_draw.area_full;
	        layer_sw_ctx->buf_size_bytes = lv_area_get_size(&layer_sw_ctx->base_draw.area_full) * px_size;
		}

		uint32_t push_size = layer_sw_ctx->buf_size_bytes + 0x3;
		
		if(es_push_stack_check(push_size))
			return;
		
		uint32_t malloced_buf[(push_size) >> 2];
		
		layer_sw_ctx->base_draw.buf = malloced_buf;
		
		if(flags & LV_DRAW_LAYER_FLAG_CAN_SUBDIVIDE) {
			layer_sw_ctx->base_draw.area_act = layer_sw_ctx->base_draw.area_full;
			layer_sw_ctx->base_draw.area_act.y2 = layer_sw_ctx->base_draw.area_full.y1;
			lv_coord_t w = lv_area_get_width(&layer_sw_ctx->base_draw.area_act);
			layer_sw_ctx->base_draw.max_row_with_alpha = layer_sw_ctx->buf_size_bytes / w / LV_IMG_PX_SIZE_ALPHA_BYTE;
			layer_sw_ctx->base_draw.max_row_with_no_alpha = layer_sw_ctx->buf_size_bytes / w / sizeof(lv_color_t);
		}
		else {
	        lv_memset_00(layer_sw_ctx->base_draw.buf, layer_sw_ctx->buf_size_bytes);
	        draw_ctx->buf = layer_sw_ctx->base_draw.buf;
	        draw_ctx->buf_area = &layer_sw_ctx->base_draw.area_act;
	        draw_ctx->clip_area = &layer_sw_ctx->base_draw.area_act;
	
	        g_es_disp_def.driver.screen_transp = flags & LV_DRAW_LAYER_FLAG_HAS_ALPHA ? 1 : 0;
		}

        lv_point_t pivot = {
            .x = lv_obj_get_style_transform_pivot_x(obj, 0),
            .y = lv_obj_get_style_transform_pivot_y(obj, 0)
        };

        lv_draw_img_dsc_t draw_dsc;
        lv_draw_img_dsc_init(&draw_dsc);
        draw_dsc.opa = opa;
#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
        draw_dsc.angle = lv_obj_get_style_transform_angle(obj, 0);
        if(draw_dsc.angle > 3600) draw_dsc.angle -= 3600;
        else if(draw_dsc.angle < 0) draw_dsc.angle += 3600;

        draw_dsc.zoom = lv_obj_get_style_transform_zoom(obj, 0);
#endif
        draw_dsc.blend_mode = lv_obj_get_style_blend_mode(obj, 0);

        if(flags & LV_DRAW_LAYER_FLAG_CAN_SUBDIVIDE) {
            layer_ctx->area_act = layer_ctx->area_full;
            layer_ctx->area_act.y2 = layer_ctx->area_act.y1 + layer_ctx->max_row_with_no_alpha - 1;
            if(layer_ctx->area_act.y2 > layer_ctx->area_full.y2) layer_ctx->area_act.y2 = layer_ctx->area_full.y2;
        }

        while(layer_ctx->area_act.y1 <= layer_area_full.y2) {
            if(flags & LV_DRAW_LAYER_FLAG_CAN_SUBDIVIDE) {
					bool has_alpha;
					/*If globally the layer has alpha maybe this smaller section has not (e.g. not on a rounded corner)
					 *If turns out that this section has no alpha renderer can choose faster algorithms*/
					if(flags & LV_DRAW_LAYER_FLAG_HAS_ALPHA) {
						/*Test for alpha by assuming there is no alpha. If it fails, fall back to rendering with alpha*/
						has_alpha = true;
						if(_lv_area_is_in(&layer_ctx->area_act, &obj->coords, 0)) {
							lv_cover_check_info_t info;
							info.res = LV_COVER_RES_COVER;
							info.area = &layer_ctx->area_act;
							lv_event_send(obj, LV_EVENT_COVER_CHECK, &info);
							if(info.res == LV_COVER_RES_COVER) has_alpha = false;
						}

						if(has_alpha) {
							layer_ctx->area_act.y2 = layer_ctx->area_act.y1 + layer_ctx->max_row_with_alpha - 1;
						}
					}
					else {
						has_alpha = false;
					}

					if(layer_ctx->area_act.y2 > layer_ctx->area_full.y2) layer_ctx->area_act.y2 = layer_ctx->area_full.y2;
					
					lv_draw_sw_layer_ctx_t * layer_sw_ctx = (lv_draw_sw_layer_ctx_t *) layer_ctx;
					if(flags & LV_DRAW_LAYER_FLAG_HAS_ALPHA) {
						lv_memset_00(layer_ctx->buf, layer_sw_ctx->buf_size_bytes);
						layer_sw_ctx->has_alpha = 1;
						g_es_disp_def.driver.screen_transp = 1;
					}
					else {
						layer_sw_ctx->has_alpha = 0;
						g_es_disp_def.driver.screen_transp = 0;
					}

					draw_ctx->buf = layer_ctx->buf;
					draw_ctx->buf_area = &layer_ctx->area_act;
					draw_ctx->clip_area = &layer_ctx->area_act;
            }

            lv_obj_redraw(draw_ctx, obj);

#ifdef ES_LV_IMG_USE_ANGLE_ZOOM
            draw_dsc.pivot.x = obj->coords.x1 + pivot.x - draw_ctx->buf_area->x1;
            draw_dsc.pivot.y = obj->coords.y1 + pivot.y - draw_ctx->buf_area->y1;
#endif

            /*With LV_DRAW_LAYER_FLAG_CAN_SUBDIVIDE it should also go the next chunk*/
			lv_draw_sw_layer_ctx_t * layer_sw_ctx = (lv_draw_sw_layer_ctx_t *) layer_ctx;

			lv_img_dsc_t img;
			img.data = draw_ctx->buf;
			img.header.always_zero = 0;
			img.header.w = lv_area_get_width(draw_ctx->buf_area);
			img.header.h = lv_area_get_height(draw_ctx->buf_area);
			img.header.cf = layer_sw_ctx->has_alpha ? LV_IMG_CF_TRUE_COLOR_ALPHA : LV_IMG_CF_TRUE_COLOR;

			/*Restore the original draw_ctx*/
			draw_ctx->buf = layer_ctx->original.buf;
			draw_ctx->buf_area = layer_ctx->original.buf_area;
			draw_ctx->clip_area = layer_ctx->original.clip_area;
			g_es_disp_def.driver.screen_transp = layer_ctx->original.screen_transp;

			/*Blend the layer*/
			lv_draw_img(draw_ctx, &draw_dsc, &layer_ctx->area_act, &img);
			lv_draw_wait_for_finish(draw_ctx);

				if((flags & LV_DRAW_LAYER_FLAG_CAN_SUBDIVIDE) == 0) break;

				layer_ctx->area_act.y1 = layer_ctx->area_act.y2 + 1;
				layer_ctx->area_act.y2 = layer_ctx->area_act.y1 + layer_ctx->max_row_with_no_alpha - 1;
			}

			lv_draw_wait_for_finish(draw_ctx);
			draw_ctx->buf = layer_ctx->original.buf;
			draw_ctx->buf_area = layer_ctx->original.buf_area;
			draw_ctx->clip_area = layer_ctx->original.clip_area;
			g_es_disp_def.driver.screen_transp = layer_ctx->original.screen_transp;
    }
}


