/**
 * @file lv_indev.c
 *
 */

/*********************
 *      INCLUDES
 ********************/
#include "lv_indev.h"
#include "lv_disp.h"
#include "lv_obj.h"
#include "lv_indev_scroll.h"
#include "lv_group.h"
#include "lv_refr.h"

#include "../hal/lv_hal_tick.h"
#include "../misc/lv_timer.h"
#include "../misc/lv_math.h"

/*********************
 *      DEFINES
 *********************/
#if LV_INDEV_DEF_SCROLL_THROW <= 0
    #warning "LV_INDEV_DRAG_THROW must be greater than 0"
#endif

/**********************
 *      TYPEDEFS
 **********************/

/**********************
 *  STATIC PROTOTYPES
 **********************/
 
/**********************
 *  STATIC VARIABLES
 **********************/
lv_indev_t g_es_indev_act;
static lv_obj_t * indev_obj_act = NULL;

/**********************
 *      MACROS
 **********************/
#if LV_LOG_TRACE_INDEV
    #define INDEV_TRACE(...) LV_LOG_TRACE(__VA_ARGS__)
#else
    #define INDEV_TRACE(...)
#endif

/**********************
 *   GLOBAL FUNCTIONS
 **********************/

extern void es_lv_touch_read(lv_indev_drv_t *s_indev_drv, lv_indev_data_t *data);

void lv_indev_read_timer_cb(void)
{
    lv_indev_data_t data;
	
	_lv_indev_proc_t *proc = &g_es_indev_act.proc;

    /*Handle reset query before processing the point*/
    if(g_es_indev_act.proc.reset_query) {
        g_es_indev_act.proc.types.pointer.act_obj           = NULL;
        g_es_indev_act.proc.types.pointer.last_obj          = NULL;
        g_es_indev_act.proc.types.pointer.scroll_obj          = NULL;
        g_es_indev_act.proc.long_pr_sent                    = 0;
        g_es_indev_act.proc.pr_timestamp                    = 0;
        g_es_indev_act.proc.longpr_rep_timestamp            = 0;
        g_es_indev_act.proc.types.pointer.scroll_sum.x        = 0;
        g_es_indev_act.proc.types.pointer.scroll_sum.y        = 0;
        g_es_indev_act.proc.types.pointer.scroll_dir = LV_DIR_NONE;
        g_es_indev_act.proc.types.pointer.scroll_throw_vect.x = 0;
        g_es_indev_act.proc.types.pointer.scroll_throw_vect.y = 0;
        g_es_indev_act.proc.types.pointer.gesture_sum.x     = 0;
        g_es_indev_act.proc.types.pointer.gesture_sum.y     = 0;
		g_es_indev_act.proc.reset_query                     = 0;
    }

    if(g_es_indev_act.proc.disabled ||
       g_es_disp_def.prev_scr != NULL) return; /*Input disabled or screen animation active*/
	   
        /*Read the data*/
        data.point.x = g_es_indev_act.proc.types.pointer.last_raw_point.x;
        data.point.y = g_es_indev_act.proc.types.pointer.last_raw_point.y;

		es_lv_touch_read(0, &data);

        g_es_indev_act.proc.state = data.state;

		/*Save the raw points so they can be used again in _lv_indev_read*/
		g_es_indev_act.proc.types.pointer.last_raw_point.x = data.point.x;
		g_es_indev_act.proc.types.pointer.last_raw_point.y = data.point.y;

		/*Move the cursor if set and moved*/
		if(g_es_indev_act.cursor != NULL &&
		   (g_es_indev_act.proc.types.pointer.last_point.x != data.point.x || g_es_indev_act.proc.types.pointer.last_point.y != data.point.y)) {
			lv_obj_set_pos(g_es_indev_act.cursor, data.point.x, data.point.y);
		}

		g_es_indev_act.proc.types.pointer.act_point.x = data.point.x;
		g_es_indev_act.proc.types.pointer.act_point.y = data.point.y;

		indev_obj_act = proc->types.pointer.act_obj;

		if(g_es_indev_act.proc.state == LV_INDEV_STATE_PRESSED) {
			
			if(proc->wait_until_release == 0)
			{
				bool new_obj_searched = false;

				/*If there is no last object then search*/
				if(indev_obj_act == NULL) {
					indev_obj_act = lv_indev_search_obj(g_es_disp_def.sys_layer, &proc->types.pointer.act_point);
					if(indev_obj_act == NULL) indev_obj_act = lv_indev_search_obj(g_es_disp_def.top_layer,
																					  &proc->types.pointer.act_point);
					if(indev_obj_act == NULL) indev_obj_act = lv_indev_search_obj(g_es_disp_def.act_scr,
																					  &proc->types.pointer.act_point);
					new_obj_searched = true;
				}
				/*If there is last object but it is not scrolled and not protected also search*/
				else if(proc->types.pointer.scroll_obj == NULL &&
						lv_obj_has_flag(indev_obj_act, LV_OBJ_FLAG_PRESS_LOCK) == false) {
					indev_obj_act = lv_indev_search_obj(g_es_disp_def.sys_layer, &proc->types.pointer.act_point);
					if(indev_obj_act == NULL) indev_obj_act = lv_indev_search_obj(g_es_disp_def.top_layer,
																					  &proc->types.pointer.act_point);
					if(indev_obj_act == NULL) indev_obj_act = lv_indev_search_obj(g_es_disp_def.act_scr,
																					  &proc->types.pointer.act_point);
					new_obj_searched = true;
				}

				/*The last object might have scroll throw. Stop it manually*/
				if(new_obj_searched && proc->types.pointer.last_obj) {
					proc->types.pointer.scroll_throw_vect.x = 0;
					proc->types.pointer.scroll_throw_vect.y = 0;
					_lv_indev_scroll_throw_handler(proc);
				}

				lv_obj_transform_point(indev_obj_act, &proc->types.pointer.act_point, true, true);

				/*If a new object was found reset some variables and send a pressed event handler*/
				if(indev_obj_act != proc->types.pointer.act_obj) {
					proc->types.pointer.last_point.x = proc->types.pointer.act_point.x;
					proc->types.pointer.last_point.y = proc->types.pointer.act_point.y;

					/*If a new object found the previous was lost, so send a Call the ancestor's event handler*/
					if(proc->types.pointer.act_obj != NULL) {
						/*Save the obj because in special cases `act_obj` can change in the Call the ancestor's event handler function*/
						lv_obj_t * last_obj = proc->types.pointer.act_obj;

						lv_event_send(last_obj, LV_EVENT_PRESS_LOST, &g_es_indev_act);
					}

					proc->types.pointer.act_obj  = indev_obj_act; /*Save the pressed object*/
					proc->types.pointer.last_obj = indev_obj_act;

					if(indev_obj_act != NULL) {
						/*Save the time when the obj pressed to count long press time.*/
						proc->pr_timestamp                 = lv_tick_get();
						proc->long_pr_sent                 = 0;
						proc->types.pointer.scroll_sum.x     = 0;
						proc->types.pointer.scroll_sum.y     = 0;
						proc->types.pointer.scroll_dir = LV_DIR_NONE;
						proc->types.pointer.gesture_dir = LV_DIR_NONE;
						proc->types.pointer.gesture_sent   = 0;
						proc->types.pointer.gesture_sum.x  = 0;
						proc->types.pointer.gesture_sum.y  = 0;
						proc->types.pointer.vect.x         = 0;
						proc->types.pointer.vect.y         = 0;

						/*Call the ancestor's event handler about the press*/
						lv_event_send(indev_obj_act, LV_EVENT_PRESSED, &g_es_indev_act);

						if(g_es_indev_act.proc.wait_until_release == 0)
						{
							/*Handle focus*/
							/*Handle click focus*/
							if(lv_obj_has_flag(indev_obj_act, LV_OBJ_FLAG_CLICK_FOCUSABLE) != false &&
							   proc->types.pointer.last_pressed != indev_obj_act) {
								   
								lv_group_t * g_act = lv_obj_get_group(indev_obj_act);
								lv_group_t * g_prev = proc->types.pointer.last_pressed ? lv_obj_get_group(proc->types.pointer.last_pressed) : NULL;

								/*If both the last and act. obj. are in the same group (or have no group)*/
								if(g_act == g_prev) {
									/*The objects are in a group*/
									if(g_act) {
										
									}
									/*The object are not in group*/
									else {
										if(proc->types.pointer.last_pressed) {
											lv_event_send(proc->types.pointer.last_pressed, LV_EVENT_DEFOCUSED, &g_es_indev_act);
										}

										lv_event_send(indev_obj_act, LV_EVENT_FOCUSED, &g_es_indev_act);
									}
								}
								/*The object are not in the same group (in different groups or one has no group)*/
								else {
									/*If the prev. obj. is not in a group then defocus it.*/
									if(g_prev == NULL && proc->types.pointer.last_pressed) {
										lv_event_send(proc->types.pointer.last_pressed, LV_EVENT_DEFOCUSED, &g_es_indev_act);
									}
									/*Focus on a non-group object*/
									else {
										if(proc->types.pointer.last_pressed) {
											/*If the prev. object also wasn't in a group defocus it*/
											if(g_prev == NULL) {
												lv_event_send(proc->types.pointer.last_pressed, LV_EVENT_DEFOCUSED, &g_es_indev_act);
											}
											/*If the prev. object also was in a group at least "LEAVE" it instead of defocus*/
											else {
												lv_event_send(proc->types.pointer.last_pressed, LV_EVENT_LEAVE, &g_es_indev_act);
											}
										}
									}

									/*Focus to the act. in its group*/
									if(g_act) {
									}
									else {
										lv_event_send(indev_obj_act, LV_EVENT_FOCUSED, &g_es_indev_act);
									}
								}
								proc->types.pointer.last_pressed = indev_obj_act;
							}
						}
					}
				}

				/*Calculate the vector and apply a low pass filter: new value = 0.5 * old_value + 0.5 * new_value*/
				proc->types.pointer.vect.x = proc->types.pointer.act_point.x - proc->types.pointer.last_point.x;
				proc->types.pointer.vect.y = proc->types.pointer.act_point.y - proc->types.pointer.last_point.y;

				proc->types.pointer.scroll_throw_vect.x = (proc->types.pointer.scroll_throw_vect.x + proc->types.pointer.vect.x) / 2;
				proc->types.pointer.scroll_throw_vect.y = (proc->types.pointer.scroll_throw_vect.y + proc->types.pointer.vect.y) / 2;

				proc->types.pointer.scroll_throw_vect_ori = proc->types.pointer.scroll_throw_vect;

				if(indev_obj_act) {
					lv_event_send(indev_obj_act, LV_EVENT_PRESSING, &g_es_indev_act);

					if(g_es_indev_act.proc.wait_until_release == 0)
					{
						_lv_indev_scroll_handler(proc);
						
						if((!proc->types.pointer.scroll_obj)&&(!proc->types.pointer.gesture_sent))
						{
							
							lv_obj_t * gesture_obj = proc->types.pointer.act_obj;

							/*If gesture parent is active check recursively the gesture attribute*/
							while(gesture_obj && lv_obj_has_flag(gesture_obj, LV_OBJ_FLAG_GESTURE_BUBBLE)) {
								gesture_obj = lv_obj_get_parent(gesture_obj);
							}

							if(gesture_obj) 
							{
								
								if((LV_ABS(proc->types.pointer.vect.x) < g_es_indev_act.driver.gesture_min_velocity) &&
								   (LV_ABS(proc->types.pointer.vect.y) < g_es_indev_act.driver.gesture_min_velocity)) {
									proc->types.pointer.gesture_sum.x = 0;
									proc->types.pointer.gesture_sum.y = 0;
								}

								/*Count the movement by gesture*/
								proc->types.pointer.gesture_sum.x += proc->types.pointer.vect.x;
								proc->types.pointer.gesture_sum.y += proc->types.pointer.vect.y;

								if((LV_ABS(proc->types.pointer.gesture_sum.x) > g_es_indev_act.driver.gesture_limit) ||
								   (LV_ABS(proc->types.pointer.gesture_sum.y) > g_es_indev_act.driver.gesture_limit)) {

									proc->types.pointer.gesture_sent = 1;

									if(LV_ABS(proc->types.pointer.gesture_sum.x) > LV_ABS(proc->types.pointer.gesture_sum.y)) {
										if(proc->types.pointer.gesture_sum.x > 0)
											proc->types.pointer.gesture_dir = LV_DIR_RIGHT;
										else
											proc->types.pointer.gesture_dir = LV_DIR_LEFT;
									}
									else {
										if(proc->types.pointer.gesture_sum.y > 0)
											proc->types.pointer.gesture_dir = LV_DIR_BOTTOM;
										else
											proc->types.pointer.gesture_dir = LV_DIR_TOP;
									}

									lv_event_send(gesture_obj, LV_EVENT_GESTURE, &g_es_indev_act);
								}

								/*If there is no scrolling then check for long press time*/
								if(proc->types.pointer.scroll_obj == NULL && proc->long_pr_sent == 0) {
									/*Call the ancestor's event handler about the long press if enough time elapsed*/
									if(lv_tick_elaps(proc->pr_timestamp) > g_es_indev_act.driver.long_press_time) {
										lv_event_send(indev_obj_act, LV_EVENT_LONG_PRESSED, &g_es_indev_act);

										/*Mark the Call the ancestor's event handler sending to do not send it again*/
										proc->long_pr_sent = 1;

										/*Save the long press time stamp for the long press repeat handler*/
										proc->longpr_rep_timestamp = lv_tick_get();
									}
								}

								/*Send long press repeated Call the ancestor's event handler*/
								if(proc->types.pointer.scroll_obj == NULL && proc->long_pr_sent == 1) {
									/*Call the ancestor's event handler about the long press repeat if enough time elapsed*/
									if(lv_tick_elaps(proc->longpr_rep_timestamp) > g_es_indev_act.driver.long_press_repeat_time) {
										lv_event_send(indev_obj_act, LV_EVENT_LONG_PRESSED_REPEAT, &g_es_indev_act);
										proc->longpr_rep_timestamp = lv_tick_get();
									}
								}
							}
						}
					}
				}
			}
		}
		else {
			
			if(proc->wait_until_release != 0) {
				lv_event_send(proc->types.pointer.act_obj, LV_EVENT_PRESS_LOST, &g_es_indev_act);

				proc->types.pointer.act_obj  = NULL;
				proc->types.pointer.last_obj = NULL;
				proc->pr_timestamp           = 0;
				proc->longpr_rep_timestamp   = 0;
				proc->wait_until_release     = 0;
			}
			
			lv_obj_t * scroll_obj = proc->types.pointer.scroll_obj;

			/*Forget the act obj and send a released Call the ancestor's event handler*/
			if(indev_obj_act) {

				/*Send RELEASE Call the ancestor's event handler and event*/
				lv_event_send(indev_obj_act, LV_EVENT_RELEASED, &g_es_indev_act);

				/*Send CLICK if no scrolling*/
				if(scroll_obj == NULL) {
					if(proc->long_pr_sent == 0) {
						lv_event_send(indev_obj_act, LV_EVENT_SHORT_CLICKED, &g_es_indev_act);
					}

					lv_event_send(indev_obj_act, LV_EVENT_CLICKED, &g_es_indev_act);
				}

				proc->types.pointer.act_obj = NULL;
				proc->pr_timestamp          = 0;
				proc->longpr_rep_timestamp  = 0;

			}

			/*The reset can be set in the Call the ancestor's event handler function.
			 * In case of reset query ignore the remaining parts.*/
			if(scroll_obj) {
				_lv_indev_scroll_throw_handler(proc);
			}
		}

		g_es_indev_act.proc.types.pointer.last_point.x = g_es_indev_act.proc.types.pointer.act_point.x;
		g_es_indev_act.proc.types.pointer.last_point.y = g_es_indev_act.proc.types.pointer.act_point.y;
	
    /*End of indev processing, so no act indev*/
    indev_obj_act = NULL;
}

void lv_indev_enable(lv_indev_t * indev, bool en)
{
    uint8_t enable = en ? 0 : 1;
	
	g_es_indev_act.proc.disabled = enable;
}

lv_indev_t * lv_indev_get_act(void)
{
    return &g_es_indev_act;
}

void lv_indev_reset(lv_indev_t * indev, lv_obj_t * obj)
{
    g_es_indev_act.proc.reset_query = 1;
			
	if(obj == NULL) {
		if(g_es_indev_act.proc.types.pointer.last_pressed == obj) {
			g_es_indev_act.proc.types.pointer.last_pressed = NULL;
		}
		if(g_es_indev_act.proc.types.pointer.act_obj == obj) {
			g_es_indev_act.proc.types.pointer.act_obj = NULL;
					}
		if(g_es_indev_act.proc.types.pointer.last_obj == obj) {
			g_es_indev_act.proc.types.pointer.last_obj = NULL;
		}
	}
				
    indev_obj_act = NULL;
}

void lv_indev_reset_long_press(lv_indev_t * indev)
{
    indev->proc.long_pr_sent         = 0;
    indev->proc.longpr_rep_timestamp = lv_tick_get();
    indev->proc.pr_timestamp         = lv_tick_get();
}

void lv_indev_set_group(lv_indev_t * indev, lv_group_t * group)
{
	
}

void lv_indev_set_button_points(lv_indev_t * indev, const lv_point_t points[])
{
	
}

void lv_indev_get_point(const lv_indev_t * indev, lv_point_t * point)
{
        point->x = indev->proc.types.pointer.act_point.x;
        point->y = indev->proc.types.pointer.act_point.y;
}

lv_dir_t lv_indev_get_gesture_dir(const lv_indev_t * indev)
{
    return indev->proc.types.pointer.gesture_dir;
}

uint32_t lv_indev_get_key(const lv_indev_t * indev)
{
	return 0;
}

lv_dir_t lv_indev_get_scroll_dir(const lv_indev_t * indev)
{
	return indev->proc.types.pointer.scroll_dir;
}

lv_obj_t * lv_indev_get_scroll_obj(const lv_indev_t * indev)
{
	return indev->proc.types.pointer.scroll_obj;
}

void lv_indev_get_vect(const lv_indev_t * indev, lv_point_t * point)
{
    point->x = 0;
    point->y = 0;

        point->x = indev->proc.types.pointer.vect.x;
        point->y = indev->proc.types.pointer.vect.y;
}

void lv_indev_wait_release(lv_indev_t * indev)
{
    if(indev == NULL)return;
    indev->proc.wait_until_release = 1;
}

lv_obj_t * lv_indev_get_obj_act(void)
{
    return indev_obj_act;
}

lv_obj_t * lv_indev_search_obj(lv_obj_t * obj, lv_point_t * point)
{
    lv_obj_t * found_p = NULL;

    /*If this obj is hidden the children are hidden too so return immediately*/
    if(lv_obj_has_flag(obj, LV_OBJ_FLAG_HIDDEN)) return NULL;

    lv_point_t p_trans = *point;
    lv_obj_transform_point(obj, &p_trans, false, true);

    bool hit_test_ok = lv_obj_hit_test(obj, &p_trans);

    /*If the point is on this object or has overflow visible check its children too*/
    if(_lv_area_is_point_on(&obj->coords, &p_trans, 0)) {
        int32_t i;
        uint32_t child_cnt = lv_obj_get_child_cnt(obj);

        /*If a child matches use it*/
        for(i = child_cnt - 1; i >= 0; i--) {
            lv_obj_t * child = obj->spec_attr->children[i];
            found_p = lv_indev_search_obj(child, &p_trans);
            if(found_p) return found_p;
        }
    }

    /*If not return earlier for a clicked child and this obj's hittest was ok use it
     *else return NULL*/
    if(hit_test_ok) return obj;
    else return NULL;
}

/**********************
 *   STATIC FUNCTIONS
 **********************/


