/**
 * @file lv_disp.c
 *
 */

/*********************
 *      INCLUDES
 *********************/
#include "lv_disp.h"
#include "../misc/lv_math.h"
#include "../core/lv_refr.h"

/*********************
 *      DEFINES
 *********************/

/**********************
 *      TYPEDEFS
 **********************/

/**********************
 *  STATIC PROTOTYPES
 **********************/
 
/**********************
 *  STATIC VARIABLES
 **********************/

/**********************
 *      MACROS
 **********************/

/**********************
 *   GLOBAL FUNCTIONS
 **********************/

extern lv_disp_t g_es_disp_def;

/**
 * Return with a pointer to the active screen
 * @param disp pointer to display which active screen should be get. (NULL to use the default
 * screen)
 * @return pointer to the active screen object (loaded by 'lv_scr_load()')
 */
lv_obj_t * lv_disp_get_scr_act(lv_disp_t * disp)
{
    return g_es_disp_def.act_scr;
}

/**
 * Return with a pointer to the previous screen. Only used during screen transitions.
 * @param disp pointer to display which previous screen should be get. (NULL to use the default
 * screen)
 * @return pointer to the previous screen object or NULL if not used now
 */
lv_obj_t * lv_disp_get_scr_prev(lv_disp_t * disp)
{
    return g_es_disp_def.prev_scr;
}

/**
 * Make a screen active
 * @param scr pointer to a screen
 */
void lv_disp_load_scr(lv_obj_t * scr)
{
    lv_scr_load_anim(scr, LV_SCR_LOAD_ANIM_NONE, 0, 0, false);
}

/**
 * Return with the top layer. (Same on every screen and it is above the normal screen layer)
 * @param disp pointer to display which top layer should be get. (NULL to use the default screen)
 * @return pointer to the top layer object (transparent screen sized lv_obj)
 */
lv_obj_t * lv_disp_get_layer_top(lv_disp_t * disp)
{
    return g_es_disp_def.top_layer;
}

/**
 * Return with the sys. layer. (Same on every screen and it is above the normal screen and the top
 * layer)
 * @param disp pointer to display which sys. layer should be retrieved. (NULL to use the default screen)
 * @return pointer to the sys layer object (transparent screen sized lv_obj)
 */
lv_obj_t * lv_disp_get_layer_sys(lv_disp_t * disp)
{
    return g_es_disp_def.sys_layer;
}

/**
 * Set the background color of a display
 * @param disp pointer to a display
 * @param color color of the background
 */
void lv_disp_set_bg_color(lv_disp_t * disp, lv_color_t color)
{
//    g_es_disp_def.bg_color = color;
//
//    lv_area_t a;
//    lv_area_set(&a, 0, 0, g_es_disp_def.driver->hor_res - 1, g_es_disp_def.driver->ver_res - 1);
//    _lv_inv_area(&a);
}

/**
 * Set the background image of a display
 * @param disp pointer to a display
 * @param img_src path to file or pointer to an `lv_img_dsc_t` variable
 */
void lv_disp_set_bg_image(lv_disp_t * disp, const void  * img_src)
{
//    g_es_disp_def.bg_img = img_src;
//
//    lv_area_t a;
//    lv_area_set(&a, 0, 0, g_es_disp_def.driver->hor_res - 1, g_es_disp_def.driver->ver_res - 1);
//    _lv_inv_area(&a);
}

/**
 * Set opacity of the background
 * @param disp pointer to a display
 * @param opa opacity (0..255)
 */
void lv_disp_set_bg_opa(lv_disp_t * disp, lv_opa_t opa)
{
#ifdef ES_LV_CONF_OPTIMIZE_USE_BG_OPA
    g_es_disp_def.bg_opa = opa;
#endif
    lv_area_t a;
    lv_area_set(&a, 0, 0, g_es_disp_def.driver.hor_res - 1, g_es_disp_def.driver.ver_res - 1);
    _lv_inv_area(&a);
}

/**
 * Switch screen with animation
 * @param scr pointer to the new screen to load
 * @param anim_type type of the animation from `lv_scr_load_anim_t`, e.g. `LV_SCR_LOAD_ANIM_MOVE_LEFT`
 * @param time time of the animation
 * @param delay delay before the transition
 * @param auto_del true: automatically delete the old screen
 */
void lv_scr_load_anim_(lv_obj_t * new_scr, lv_scr_load_anim_t anim_type, uint32_t time, uint32_t delay, bool auto_del)
{
    lv_disp_t * d = &g_es_disp_def;
    lv_obj_t * act_scr = lv_scr_act();

    d->scr_to_load = new_scr;

    /*Be sure there is no other animation on the screens*/
    lv_anim_del(new_scr, NULL);
    lv_anim_del(act_scr, NULL);

    /*Be sure both screens are in a normal position*/
    lv_obj_set_pos(new_scr, 0, 0);
    lv_obj_remove_local_style_prop(new_scr, LV_STYLE_OPA, 0);
	
    lv_obj_set_pos(act_scr, 0, 0);
    lv_obj_remove_local_style_prop(act_scr, LV_STYLE_OPA, 0);


    /*Shortcut for immediate load*/
    lv_obj_t * old_scr = d->act_scr;

    d->act_scr = new_scr;

    if(d->act_scr) lv_event_send(new_scr, LV_EVENT_SCREEN_LOADED, NULL);
    if(d->act_scr) lv_event_send(old_scr, LV_EVENT_SCREEN_UNLOADED, NULL);

    lv_obj_invalidate(new_scr);
}


extern uint8_t change_bg_flag;
extern uint32_t change_bg_img_index;

void lv_scr_load_anim_fast_no_bg(lv_obj_t * new_scr,uint32_t ext_bg_img_type,uint32_t ext_bg_img_id)
{
    lv_disp_t * d = &g_es_disp_def;
    lv_obj_t * act_scr = lv_scr_act();
	lv_obj_t * child1;

    d->scr_to_load = new_scr;

    /*Be sure there is no other animation on the screens*/
    lv_anim_del(new_scr, NULL);
    lv_anim_del(act_scr, NULL);

    /*Be sure both screens are in a normal position*/
    lv_obj_set_pos(new_scr, 0, 0);
    lv_obj_remove_local_style_prop(new_scr, LV_STYLE_OPA, 0);
	
    lv_obj_set_pos(act_scr, 0, 0);
    lv_obj_remove_local_style_prop(act_scr, LV_STYLE_OPA, 0);


    /*Shortcut for immediate load*/
    lv_obj_t * old_scr = d->act_scr;

    d->act_scr = new_scr;

    if(d->act_scr) lv_event_send(new_scr, LV_EVENT_SCREEN_LOADED, NULL);
    if(d->act_scr) lv_event_send(old_scr, LV_EVENT_SCREEN_UNLOADED, NULL);

	//开始改变screen，放弃当前screen的所有待刷新区域
	d->inv_p = 0;

	if(new_scr->spec_attr)
	{
		for(uint32_t i = 0;i < new_scr->spec_attr->child_cnt;i++)
		{
			child1 = new_scr->spec_attr->children[i];
			
			lv_obj_refr_size(child1);	
			lv_obj_refr_pos(child1);			  
			
			lv_obj_invalidate(child1);
		}
	}
	
	if(ext_bg_img_type)
	{
		change_bg_img_index = ext_bg_img_id;
		change_bg_flag = 1;
	}
}

void lv_scr_load_anim(lv_obj_t * new_scr, lv_scr_load_anim_t anim_type, uint32_t time, uint32_t delay, bool auto_del)
{
	const lv_img_dsc_t * bg_img = lv_obj_get_style_bg_img_src(new_scr,LV_PART_MAIN);
	if(bg_img)
	{
		lv_scr_load_anim_fast_no_bg(new_scr,1,bg_img->img_id);
	}
	else
	{
		lv_scr_load_anim_(new_scr, anim_type, time, delay, auto_del);
	}
}

/**
 * Get elapsed time since last user activity on a display (e.g. click)
 * @param disp pointer to a display (NULL to get the overall smallest inactivity)
 * @return elapsed ticks (milliseconds) since the last activity
 */

/**
 * Manually trigger an activity on a display
 * @param disp pointer to a display (NULL to use the default display)
 */

/**********************
 *   STATIC FUNCTIONS
 **********************/
