/**********************************************************************************
 *
 * @file    c_iir.c
 * @brief   IIR implementation in C
 *
 * @date    09 Feb  2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          09 Feb  2023    AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------*/
#include "es32_dsp.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  IIR filter with 4 sections
  * @param y: Output
  * @param x: Input
  * @param IIRCoeff: IIR Filter Coefficients, an array of 20 shorts
  * @param ny: number of output
  * @retval None
  */
void iir_biquad(uint16_t *y,const uint16_t *x, const int16_t *IIRCoeff, uint16_t ny)
{
  uint32_t i;
  uint32_t w1_2 = 0U, w1_1 = 0U, w1;
  uint32_t w2_2 = 0U, w2_1 = 0U, w2;
  uint32_t w3_2 = 0U, w3_1 = 0U, w3;
  uint32_t w4_2 = 0U, w4_1 = 0U, w4;

  /** Canonic form **/
  /* 1st section */
  for (i=0; i<ny-2; i++)
  {
    w1 = x[2+i] - IIRCoeff[0]*w1_1 - IIRCoeff[1]*w1_2;
    y[2+i] = (IIRCoeff[2]*w1 + IIRCoeff[3]*w1_1 + IIRCoeff[4]*w1_2);
    w1_2 = w1_1;
    w1_1 = w1;
  }

  /* 2nd section */
  for (i=0; i<ny-2; i++)
  {
    w2 = y[2+i] - IIRCoeff[5]*w2_1 - IIRCoeff[6]*w2_2;
    y[2+i] = (IIRCoeff[7]*w2 + IIRCoeff[8]*w2_1 + IIRCoeff[9]*w2_2);
    w2_2 = w2_1;
    w2_1 = w2;
  }

  /* 3rd section */
  for (i=0; i<ny-2; i++)
  {
    w3 = y[2+i] - IIRCoeff[10]*w3_1 - IIRCoeff[11]*w3_2;
    y[2+i] = (IIRCoeff[12]*w3 + IIRCoeff[13]*w3_1 + IIRCoeff[14]*w3_2);
    w3_2 = w3_1;
    w3_1 = w3;
  }

  /* 4th section */
  for (i=0; i<ny-2; i++)
  {
    w4 = y[2+i] - IIRCoeff[15]*w4_1 - IIRCoeff[16]*w4_2;
    y[2+i] = (IIRCoeff[17]*w4 + IIRCoeff[18]*w4_1 + IIRCoeff[19]*w4_2);
    w4_2 = w4_1;
    w4_1 = w4;
  }

}

/**
  * @brief  IIR filter with 4 sections
  * @param y: Output
  * @param x: Input
  * @param IIRCoeff: IIR Filter Coefficients, an array of 20 shorts
  * @param ny: number of output
  * @retval None
  */
void iir_biquad_pts(int16_t *y,const int16_t *x, const int16_t *IIRCoeff, uint16_t ny)
{
  int32_t i;
  int32_t w1_2 = 0, w1_1 = 0, w1;
  int32_t w2_2 = 0, w2_1 = 0, w2;
  int32_t w3_2 = 0, w3_1 = 0, w3;
  int32_t w4_2 = 0, w4_1 = 0, w4;

  /** Canonic form **/
  /* 1st section */
  for (i=0; i<ny-2; i++)
  {
    w1 = x[2+i] - ((IIRCoeff[0]*w1_1 + IIRCoeff[1]*w1_2)>>IIR_FIXPTS);
    y[2+i] = (IIRCoeff[2]*w1 + IIRCoeff[3]*w1_1 + IIRCoeff[4]*w1_2)>>IIR_FIXPTS;
    w1_2 = w1_1;
    w1_1 = w1;
  }

  /* 2nd section */
  for (i=0; i<ny-2; i++)
  {
    w2 = y[2+i] - ((IIRCoeff[5]*w2_1 + IIRCoeff[6]*w2_2)>>IIR_FIXPTS);
    y[2+i] = (IIRCoeff[7]*w2 + IIRCoeff[8]*w2_1 + IIRCoeff[9]*w2_2)>>IIR_FIXPTS;
    w2_2 = w2_1;
    w2_1 = w2;
  }

  /* 3rd section */
  for (i=0; i<ny-2; i++)
  {
    w3 = y[2+i] - ((IIRCoeff[10]*w3_1 + IIRCoeff[11]*w3_2)>>IIR_FIXPTS);
    y[2+i] = (IIRCoeff[12]*w3 + IIRCoeff[13]*w3_1 + IIRCoeff[14]*w3_2)>>IIR_FIXPTS;
    w3_2 = w3_1;
    w3_1 = w3;
  }

  /* 4th section */
  for (i=0; i<ny-2; i++)
  {
    w4 = y[2+i] - ((IIRCoeff[15]*w4_1 + IIRCoeff[16]*w4_2)>>IIR_FIXPTS);
    y[2+i] = (IIRCoeff[17]*w4 + IIRCoeff[18]*w4_1 + IIRCoeff[19]*w4_2)>>IIR_FIXPTS;
    w4_2 = w4_1;
    w4_1 = w4;
  }

}
