/*********************************************************

*文件名:  flash.c
*作  者:  AE Team
*版  本:  V1.00
*日  期:  2021/11/10
*描  述:  Flash模块函数
*备  注:
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
**********************************************************/
#include <string.h>
#include "flash.h"
#include "spi.h"
#include "uart.h"

static void delay(int i);

/**
  * @brief  flash sector erase function
  * @param  addr: Specific address which sector to be erase.
  * @retval None.
  */
void flash_sector_erase(uint32_t addr)
{
    uint16_t cnt;
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    cmd_buf[0] = FLASH_ERASE;           /*Flash sector erase command*/
    cmd_buf[1] = (addr >> 16) & 0xff;   /*24 bit Flash address*/
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    CS_RESET;     /*Choose lower, the selected Flash*/

    SPI_SendByte(FLASH_WRITE_ENABLE);  /*First write send enabling instruction*/
    cnt = 4000;
    while ((SPI_GetFlagStatus(SPI_Flag_IDLE) == RESET) && (--cnt));

    CS_SET;    /*Pick up and release of Flash*/

    delay(100);
    CS_RESET;

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the sector erase instructions and Flash address three bytes*/
    {
        SPI_SendByte(cmd_buf[i]);
        cnt = 4000;
        while ((SPI_GetFlagStatus(SPI_Flag_IDLE) == RESET) && (--cnt));
    }

    CS_SET;
}

/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  addr: Specific address which to be write.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval None.
  */
void flash_write(uint32_t addr, char *buf, uint8_t size)
{
    uint16_t cnt;
    uint8_t cmd_buf[4];
    uint16_t i = 0U;

    cmd_buf[0] = FLASH_PROGRAM;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    CS_RESET;     /*Choose lower, the selected Flash*/

    SPI_SendByte(FLASH_WRITE_ENABLE);
    cnt = 4000;
    while ((SPI_GetFlagStatus(SPI_Flag_IDLE) == RESET) && (--cnt));

    CS_SET;    /*Pick up and release of Flash*/

    delay(100);
    CS_RESET;

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the programming instructions and Flash address three bytes*/
    {
        SPI_SendByte(cmd_buf[i]);
        cnt = 4000;
        while ((SPI_GetFlagStatus(SPI_Flag_IDLE) == RESET) && (--cnt));
    }

    for (i = 0; i < size; i++)  /*To write the data sent to the Flash*/
    {
        SPI_SendByte(buf[i]);
        cnt = 4000;
        while ((SPI_GetFlagStatus(SPI_Flag_IDLE) == RESET) && (--cnt));
    }

    CS_SET;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval None.
  */
void flash_read(uint32_t addr, char *buf, uint16_t size)
{
    uint16_t cnt;
    uint8_t cmd_buf[4];
    uint16_t i = 0U;

    cmd_buf[0] = FLASH_READ;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    CS_RESET;     /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(cmd_buf); i++)   /*Send the editor & reader instructions and Flash address three bytes*/
    {
        SPI_SendByte(cmd_buf[i]);
        cnt = 4000;
        while ((SPI_GetFlagStatus(SPI_Flag_IDLE) == RESET) && (--cnt));
    }

    for (i = 0; i < size; i++)  /*Send the programming instructions and Flash address three bytes*/
    {
        SPI_RecEnable();
        cnt = 4000;
        while (SPI_GetFlagStatus(SPI_Flag_RB) && (--cnt));
        SPI_SendByte(0);
        cnt = 4000;
        while ((SPI_GetFlagStatus(SPI_Flag_RB) == RESET) && (--cnt));
        buf[i] = SPI_RecByte();
        SPI_RecDisable();
    }

    CS_SET;
}

/**
  * @brief  wait until flash unbusy.
  * @retval None.
  */
void flash_wait_unbusy(void)
{
    uint16_t cnt;
    uint8_t status;

    CS_RESET; /*Choose lower, the selected Flash*/

    SPI_SendByte((uint8_t)FLASH_STATUS);   /*Send to read status command*/
    cnt = 4000;
    while ((SPI_GetFlagStatus(SPI_Flag_IDLE) == RESET) && (--cnt));

    do
    {
        SPI_RecEnable();
        cnt = 4000;
        while (SPI_GetFlagStatus(SPI_Flag_RB) && (--cnt));
        SPI_SendByte(0);
        cnt = 4000;
        while ((SPI_GetFlagStatus(SPI_Flag_RB) == RESET) && (--cnt));
        status = SPI_RecByte();
        SPI_RecDisable();
    }
    while (status & 0x01);

    CS_SET;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t flash_read_id(void)
{
    uint16_t cnt;
    uint8_t i;
    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    CS_RESET; /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(flash_id); i++)
    {
        SPI_SendByte(flash_id[i]);
        cnt = 4000;
        while ((SPI_GetFlagStatus(SPI_Flag_IDLE) == RESET) && (--cnt));
    }

    for (i = 0; i < 3; i++)
    {
        SPI_RecEnable();
        cnt = 4000;
        while (SPI_GetFlagStatus(SPI_Flag_RB) && (--cnt));
        SPI_SendByte(0);
        cnt = 4000;
        while ((SPI_GetFlagStatus(SPI_Flag_RB) == RESET) && (--cnt));
        flash_id[i] = SPI_RecByte();
        SPI_RecDisable();
    }

    CS_SET;

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /*Manufacturer ID flash_id [0] and device ID flash_id [1]*/
}

/**
  * @brief  delay some time.
  * @retval None.
  */
static void delay(int i)
{
    while (i--) ;
}
