/*********************************************************
*文件名:  lcd.c
*作  者:  AE Team
*版  本:  V1.00
*日  期:  2023/08/15
*描  述:  LCD演示主程序头文件
*备  注:
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
**********************************************************/
#include <string.h>
#include "lcd.h"

const uint8_t dig8[16]  =
{
    0x7B, 0x11, 0x67, 0x57, 0x1D, 0x5E, 0x7E, 0x13, 0x7F, 0x5F
};

const uint32_t lcd_map[8] =
{
    (1 << 8), (1 << 9), (1 << 10), (1 << 11),
    (1 << 12), (1 << 13), (1 << 14), (1 << 15)
};
/***************************************************************
函数名：LCDInit(void)
描  述：LCD GPIO口设置以及相关寄存器设置
输入值：无
输出值：无
返回值：无
***************************************************************/
void LCDInit(void)
{
    GPIO_InitStruType gpio_init;
    LCD_InitStruType lcd_init;
    memset(&gpio_init, 0, sizeof(gpio_init));
    memset(&lcd_init, 0, sizeof(lcd_init));

    gpio_init.GPIO_Signal = GPIO_Pin_Signal_Analog;
    gpio_init.GPIO_Func = GPIO_Func_0;
    gpio_init.GPIO_Direction = GPIO_Dir_Out;
    gpio_init.GPIO_PUEN = GPIO_PUE_Input_Disable;
    gpio_init.GPIO_PDEN = GPIO_PDE_Input_Disable;
    gpio_init.GPIO_OD = GPIO_ODE_Output_Disable;
    gpio_init.GPIO_DS = GPIO_DS_Output_Normal;

    GPIO_Init(GPIOA, GPIO_Pin_17, &gpio_init);
    GPIO_Init(GPIOA, GPIO_Pin_18, &gpio_init);
    GPIO_Init(GPIOA, GPIO_Pin_19, &gpio_init);
    GPIO_Init(GPIOA, GPIO_Pin_20, &gpio_init);
    GPIO_Init(GPIOA, GPIO_Pin_4, &gpio_init);
    GPIO_Init(GPIOB, GPIO_Pin_16, &gpio_init);
    GPIO_Init(GPIOB, GPIO_Pin_6, &gpio_init);
    GPIO_Init(GPIOA, GPIO_Pin_1, &gpio_init);
    GPIO_Init(GPIOA, GPIO_Pin_0, &gpio_init);
    GPIO_Init(GPIOB, GPIO_Pin_13, &gpio_init);
    GPIO_Init(GPIOB, GPIO_Pin_12, &gpio_init);
    GPIO_Init(GPIOB, GPIO_Pin_11, &gpio_init);

    LCD_Reset();

    lcd_init.LCD_CLK = LCD_CLK_LRC;
    lcd_init.LCD_PRS = LCD_PRS_32;
    lcd_init.LCD_DUTY = LCD_COM_DUTY_4;  /* COM3~COM0 */
    lcd_init.LCD_SEG_31_0 = 0x0000FF00;  /* SEG15~SEG8 */
    lcd_init.LCD_SEG_43_32 = 0x000;
    lcd_init.LCD_QTSEL = LCD_QTSEL_1_CLK;
    lcd_init.LCD_VLCD = LCD_VLCD_30_30;
    lcd_init.LCD_BUFEN = DISABLE;
    lcd_init.LCD_TYPE = LCD_TYPE_A;
    lcd_init.LCD_BIAS = LCD_BIAS_1_3;
    lcd_init.LCD_RS = LCD_RS_60K;
    lcd_init.LCD_BLINK = LCD_BLINK_OFF;
    lcd_init.LCD_FRDIV = 3;
    lcd_init.LCD_UPDCTRL = LCD_UPDCTRL_1; /* 在每个帧周期更新 */
    LCD_Init(&lcd_init);
}

/**
  * @brief  LCD fill display buffer.
  * @param  flushbuf:buffer start address.
  * @param  idx: display digital index
  * @param  number: number 0x0 ~ 0x0F
  * @retval None.
  */
void lcd_display(uint32_t *flushbuf, uint8_t idx, uint8_t number)
{
    uint8_t i = 0U;
    uint8_t temp = 0U;
    uint8_t offset = 0x3U;

    if (idx > 4 || idx < 1)
        return;

    for (i = 0; i < 4; i++)
    {
        temp = (dig8[number] & offset) >> 2 * i;

        switch (temp)
        {
            case 1:
                flushbuf[i] |= lcd_map[idx * 2 - 1];
                break;

            case 2:
                flushbuf[i] |= lcd_map[idx * 2 - 2];
                break;

            case 3:
                flushbuf[i] |= lcd_map[idx * 2 - 1] | lcd_map[idx * 2 - 2];
                break;

            default:
                break;
        }

        offset <<= 2;
    }
}

/**
  * @brief  Reflush the used com_seg buffer.
  * @param  srcbuf:the source data start address.
  * @retval None.
  */
void lcd_reflush(uint32_t *flushbuf)
{
    uint8_t i = 0U;
    uint8_t tmp[4][2] = {{0U, 1U}, {2U, 3U}, {4U, 5U}, {6U, 7U}};

    for (i = 0; i < 4; ++i)
        LCD_PixelWrite(tmp[i][0], flushbuf[i]);
}
