/***************************************************************
 *文件名：lib_lcd.h
 *作 者： AE Team
 *版 本： V1.00
 *日 期： 2023/08/14
 *描 述： LCD模块库函数头文件
 *备 注：
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************/
#ifndef __LIBLCD_H__
#define __LIBLCD_H__

#include "es8h018x.h"
#include "type.h"

/* 公共端选择位*/
typedef enum
{
    LCD_COM_DUTY_4   = 0, /* COM0~COM3 */
    LCD_COM_DUTY_6   = 1, /* COM0~COM5 */
    LCD_COM_DUTY_8   = 2, /* COM0~COM7 */
    LCD_COM_DUTY_MAX = 2,
} LCD_TYPE_DUTY;

/* LCD快速充电时间选择 */
typedef enum
{
	  LCD_QTSEL_1_CLK = 0, /* 1 * f_LCD_CLK */
	  LCD_QTSEL_2_CLK = 1, /* 2 * f_LCD_CLK */
	  LCD_QTSEL_3_CLK = 2, /* 3 * f_LCD_CLK */
	  LCD_QTSEL_4_CLK = 3, /* 4 * f_LCD_CLK */
	  LCD_QTSEL_5_CLK = 4, /* 5 * f_LCD_CLK */
	  LCD_QTSEL_6_CLK = 5, /* 6 * f_LCD_CLK */
	  LCD_QTSEL_7_CLK = 6, /* 7 * f_LCD_CLK */
	  LCD_QTSEL_8_CLK = 7, /* 8 * f_LCD_CLK */
	  LCD_QTSEL_MAX   = 7,
} LCD_TYPE_QTSEL;

/* LCD驱动波形类型选择*/
typedef enum
{
    LCD_TYPE_A   = 0, /* A波形(在每一公共端类型内改变相位) */
    LCD_TYPE_B   = 1, /* B波形(在每一帧边界改变相位) */
    LCD_TYPE_MAX = 1,
} LCD_TYPE_TYPE;

/* LCD时钟源选择*/
typedef enum
{
    LCD_CLK_LRC      = 0, /* 内部LRC(约32kHz) */
    LCD_CLK_LOSC     = 1, /* 外部LOSC */
    LCD_CLK_MAX = 2,
} LCD_TYPE_CLK;

/* LCD偏置选择*/
typedef enum
{
    LCD_BIAS_STA = 0, /* 静态 */
    LCD_BIAS_1_2 = 1, /* 1/2 BIAS */
    LCD_BIAS_1_3 = 2, /* 1/3 BIAS */
    LCD_BIAS_1_4 = 3, /* 1/4 BIAS */
    LCD_BIAS_MAX = 3,
} LCD_TYPE_BIAS;

/* LCD内部偏压电阻总和选择*/
typedef enum
{
    LCD_RS_225K     = 0, /* 225KΩ */
    LCD_RS_900K     = 1, /* 900KΩ */
    LCD_RS_60K      = 2, /* 60KΩ */
    LCD_RS_60K_225K = 4, /* 60KΩ与225KΩ自动切换 */
    LCD_RS_60K_900K = 5, /* 60KΩ与900KΩ自动切换 */
    LCD_RS_MAX      = 5,
} LCD_TYPE_RS;

/* LCD时钟源预分频选择 */
typedef enum
{
    LCD_PRS_2   = 0,  /* 2分频 */
    LCD_PRS_4   = 1,  /* 4分频 */
    LCD_PRS_6   = 2,  /* 6分频 */
    LCD_PRS_8   = 3,  /* 8分频 */
    LCD_PRS_10  = 4,  /* 10分频 */
    LCD_PRS_12  = 5,  /* 12分频 */
    LCD_PRS_14  = 6,  /* 14分频 */
    LCD_PRS_16  = 7,  /* 16分频 */
    LCD_PRS_18  = 8,  /* 18分频 */
    LCD_PRS_20  = 9,  /* 20分频 */
    LCD_PRS_22  = 10, /* 22分频 */
    LCD_PRS_24  = 11, /* 24分频 */
    LCD_PRS_26  = 12, /* 26分频 */
    LCD_PRS_28  = 13, /* 28分频 */
    LCD_PRS_30  = 14, /* 30分频 */
    LCD_PRS_32  = 15, /* 32分频 */
    LCD_PRS_MAX = 15,
} LCD_TYPE_PRS;

/* VLCD电压选择 */
typedef enum
{
    LCD_VLCD_15_30 = 0,    /* 1/2VDD*/
    LCD_VLCD_16_30 = 1,    /* 16/30VDD*/
    LCD_VLCD_17_30 = 2,    /* 17/30VDD*/
    LCD_VLCD_18_30 = 3,    /* 18/30VDD*/
    LCD_VLCD_19_30 = 4,    /* 19/30VDD*/
    LCD_VLCD_20_30 = 5,    /* 20/30VDD*/
    LCD_VLCD_21_30 = 6,    /* 21/30VDD*/
    LCD_VLCD_22_30 = 7,    /* 22/30VDD*/
    LCD_VLCD_23_30 = 8,    /* 23/30VDD*/
    LCD_VLCD_24_30 = 9,    /* 24/30VDD*/
    LCD_VLCD_25_30 = 10,   /* 25/30VDD*/
    LCD_VLCD_26_30 = 11,   /* 26/30VDD*/
    LCD_VLCD_27_30 = 12,   /* 27/30VDD*/
    LCD_VLCD_28_30 = 13,   /* 28/30VDD*/
    LCD_VLCD_29_30 = 14,   /* 29/30VDD*/
    LCD_VLCD_30_30 = 15,   /* 30/30VDD*/
    LCD_VLCD_MAX   = 15,
} LCD_TYPE_VLCD;

/* 闪烁模式选择 */
typedef enum
{
    LCD_BLINK_OFF = 0,  /* 闪烁模式关 */
    LCD_BLINK_0   = 1,  /* 闪烁模式0 */
    LCD_BLINK_1   = 2,  /* 闪烁模式1 */
    LCD_BLINK_2   = 3,  /* 闪烁模式2 */
    LCD_BLINK_MAX = 3,
} LCD_TYPE_BLINK;

/* 闪烁模式选择 */
typedef enum
{
    LCD_UPDCTRL_0   = 0,  /* 当SWUPDATE为1时，下个帧周期更新 */
    LCD_UPDCTRL_1   = 1,  /* 在每个帧周期更新 */
    LCD_UPDCTRL_2   = 2,  /* 当帧计数器递减计数至0时的下个帧周期更新 */
    LCD_UPDCTRL_MAX = 2,
} LCD_TYPE_UPDCTRL;

/* LCD初始化结构体*/
typedef struct
{
    LCD_TYPE_DUTY LCD_DUTY;    /* COM占空比选择 */
    uint32_t LCD_SEG_31_0;     /* 段使能 0~0xFFFFFFFF (bit31~0 每1位控制1个SEG  ) */
    uint32_t LCD_SEG_43_32;    /* 段使能 0~0xFFF (bit43~32 每1位控制1个SEG  ) */
    LCD_TYPE_QTSEL LCD_QTSEL;  /* LCD快速充电时间选择 */
    LCD_TYPE_VLCD LCD_VLCD;    /* VLCD电压选择 */
    TYPE_FUNCEN LCD_BUFEN;     /* 输出缓冲器使能 */
    LCD_TYPE_TYPE LCD_TYPE;    /* 驱动波形类型选择 */
    LCD_TYPE_CLK LCD_CLK;      /* 时钟选择 */
    LCD_TYPE_PRS LCD_PRS;      /* 时钟源分频比选择 */
    LCD_TYPE_BIAS LCD_BIAS;    /* 偏置选择*/
    LCD_TYPE_RS LCD_RS;        /* 偏置电阻选择 */
    LCD_TYPE_BLINK LCD_BLINK;  /* 闪烁模式选择 */
	  uint8_t LCD_FRDIV;         /* 帧频率设置 f_LCD_CLK/(2*(FRDIV+1)) */
	  LCD_TYPE_UPDCTRL LCD_UPDCTRL; /* 更新控制选择 */      
} LCD_InitStruType;

#define LCD_ENABLE()      (LCD->CTRL.Word |= 0x00000001)
#define LCD_DISABLE()     (LCD->CTRL.Word &= 0xFFFFFFFE)

/************LCD模块函数声明***********/
ErrorStatus LCD_Init(LCD_InitStruType *LCD_InitStruct);
ErrorStatus LCD_GrayscaleConfig(uint8_t VLCD_Sel);
ErrorStatus LCD_FlickerConfig(LCD_TYPE_BLINK BLINK_Mode, uint8_t BLINK_Value);
ErrorStatus LCD_ResistorConfig(LCD_TYPE_RS Rsel);
void LCD_SoftUpdate(void);
ErrorStatus LCD_PixelWrite(uint8_t bufn, uint32_t LCD_data);
void LCD_Reset(void);

#endif /* __LIBLCD_H__ */
