/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    5 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          5 Jan. 2023     Biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
lin_node_t g_node1;
uint8_t g_tx_buf[8] = {0x88U, 0x77U, 0x66U, 0x55U, 0x44U, 0x33U, 0x22U, 0x11U};
uint8_t g_rx_buf[8];
uint8_t g_master_read = 0U;
uint8_t g_master_write = 0U;

/* Private Constants --------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
void uart_send_byte(uint8_t u8data);
void uart_recv_ctrl(lin_node_t *lin_node);
void lin_breakfield_send(void);
void lin_select_normal_slope(void);
void lin_select_low_slope(void);
void lin_exit_sleep(void);
void lin_enter_sleep(void);

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Init GPIO for exti
  * @retval None
  */
void gpio_init(void)
{
    md_gpio_init_t gpio_init;

    md_gpio_init_struct(&gpio_init);
    gpio_init.mode  = MD_GPIO_MODE_INPUT;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_1;

    md_gpio_init(EXTI_PORT, EXTI_PIN_1, &gpio_init);
    md_gpio_init(EXTI_PORT, EXTI_PIN_2, &gpio_init);

    return;
}

/**
  * @brief  Initializate exti.
  * @retval None
  */
void exti_init(void)
{
    md_gpio_set_interrupt_port(EXTI_PORT, EXTI_PIN_1);      /* set exit interrupt port */
    md_gpio_enable_trailing_edge_trigger(EXTI_PIN_1);       /* enable trailing edge trigger */
    md_gpio_interrupt_filter_enable(EXTI_PIN_1);            /* enable exti filter */
    md_gpio_interrupt_filter_time_set(255);                 /* exti filter time set */
    md_gpio_enable_external_interrupt(EXTI_PIN_1);          /* enable exti */

    md_gpio_set_interrupt_port(EXTI_PORT, EXTI_PIN_2);      /* set exit interrupt port */
    md_gpio_enable_trailing_edge_trigger(EXTI_PIN_2);       /* enable trailing edge trigger */
    md_gpio_interrupt_filter_enable(EXTI_PIN_2);            /* enable exti filter */
    md_gpio_interrupt_filter_time_set(255);                 /* exti filter time set */
    md_gpio_enable_external_interrupt(EXTI_PIN_2);          /* enable exti */

    md_gpio_clear_flag(EXTI_PIN_1);
    md_gpio_clear_flag(EXTI_PIN_2);

    return;
}

/**
  * @brief  Init LIN pin
  * @retval None
  */
void lin_pin_init(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize tx pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_2;
    md_gpio_init(LIN_TX_PORT, LIN_TX_PIN, &gpio_init);

    /* Initialize rx pin */
    gpio_init.mode  = MD_GPIO_MODE_INPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_OUT_DRIVE_NORMAL;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_2;
    md_gpio_init(LIN_RX_PORT, LIN_RX_PIN, &gpio_init);

    return;
}

/**
  * @brief:  Initialize for LIN.
  * @param:  None
  * @retval: None
  */
void lin_init(void)
{
    md_uart_init_t uart_init;
    md_uart_init_struct(&uart_init);

    /* Initialize UART */
    uart_init.baud        = 9600;
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_init.mode        = MD_UART_MODE_LIN;
    md_uart_init(LIN_NODE, &uart_init);

    /* Set the LIN break length as 11 bits */
    md_uart_set_lin_bk11(LIN_NODE);

    /* enable lin break irq and rxth */
    md_uart_enable_it_linbk(LIN_NODE);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_clock_config(MD_CMU_CLOCK_HRC48M, 48000000);

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    memset(&g_node1, 0x00, sizeof(lin_node_t));
    g_node1.id = 0x07;
    g_node1.sync_field = SYNC_SEG;
    g_node1.node_mode = MASTER;
    g_node1.node_send_bk_req = lin_breakfield_send;
    g_node1.node_send_byte = uart_send_byte;
    g_node1.node_recv_ctrl = uart_recv_ctrl;
    g_node1.node_normal_slope_mode = lin_select_normal_slope;
    g_node1.node_low_slope_mode = lin_select_low_slope;
    g_node1.node_enter_sleep = lin_enter_sleep;
    g_node1.node_exit_sleep = lin_exit_sleep;

    /* initialization for exti */
    gpio_init();
    exti_init();
    md_mcu_irq_config(EXTI0_3_IRQn, 0, ENABLE);

    /* LIN config initialization */
    lin_pin_init();
    lin_init();

    /* Enable relative irq */
    md_mcu_irq_config(EUART1_IRQn, 0, ENABLE);
    __enable_irq();

    /* Due to the fact that tx pin has been pushed up in the test environment, only normal slope
        can be selected */
    lin_chstat(&g_node1, NORMAL_SLOPE);

    while (1)
    {
        if (g_master_write == 1)
        {
            /* lin send frame */
            lin_sendframe(&g_node1, g_tx_buf, 8);

            g_master_write = 0;
        }
        else if (g_master_read == 1)
        {
            /* lin receive frame */
            lin_recvframe(&g_node1, 8);

            g_master_read = 0;
        }
		else
        {
			
        }
    }
}

/**
  * @brief  uart send data
  * @retval None
  */
void uart_send_byte(uint8_t u8data)
{
    while (!md_uart_is_active_it_tfempty(LIN_NODE));

    md_uart_set_send_data8(LIN_NODE, u8data);
    md_uart_clear_it_tfempty(LIN_NODE);

    return;
}

/**
  * @brief  uart send data
  * @retval None
  */
void uart_recv_ctrl(lin_node_t *lin_node)
{
    if (lin_node->node_mode == MASTER && lin_node->dir == L_READ)
        md_uart_enable_it_rfnempty(LIN_NODE);
    else if (lin_node->node_mode == SLAVE && lin_node->dir == L_WRITE)
        md_uart_enable_it_rfnempty(LIN_NODE);
    else
        md_uart_disable_it_rfnempty(LIN_NODE);

    return;
}

/**
  * @brief  send break field
  * @retval None
  */
void lin_breakfield_send(void)
{
    md_uart_enable_lin_bkreq(LIN_NODE);

    return;
}

/**
  * @brief  use normal slope mode
  * @retval None
  */
void lin_exit_sleep(void)
{
    md_gpio_write_pin(LIN_SLP_PORT, LIN_SLP_PIN, 1);

    return;
}

/**
  * @brief  use normal slope mode
  * @retval None
  */
void lin_enter_sleep(void)
{
    md_gpio_write_pin(LIN_SLP_PORT, LIN_SLP_PIN, 0);

    return;
}

/**
  * @brief  use normal slope mode
  * @retval None
  */
void lin_select_normal_slope(void)
{
    return;
}

/**
  * @brief  use low slope mode
  * @retval None
  */

void lin_select_low_slope(void)
{
    return;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
