/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    4 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          4 Jan. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initialize TOGGLE IO.
  * @param  None
  * @retval None
  */
static void init_gpio(void)
{
    md_gpio_init_t gpio_init;
    /* Initialize TOGGLE IO as output */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_INPUT;  /* input */
    gpio_init.odos = MD_GPIO_PUSH_PULL;   /* push pull */
    gpio_init.pupd = MD_GPIO_PUSH_DOWN;   /* pull down */
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_1;      /* GPIO function 1 */
    md_gpio_init(TRIGER_PORT, TRIGER_PIN, &gpio_init);

    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_OUTPUT; /* output */
    gpio_init.odos = MD_GPIO_PUSH_PULL;   /* push pull */
    gpio_init.pupd = MD_GPIO_FLOATING;    /* no pull up and pull down */
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_3;      /* GPIO function 3 */
    md_gpio_init(PULSE_PORT, PULSE_PIN, &gpio_init);
}

/**
  * @brief  Initialize PIS for IO triger AD16C4T enable
  * @param  None
  * @retval None
  */
void init_pis(void)
{
    md_pis_init_t pis_init;
    memset(&pis_init, 0, sizeof(md_pis_init_t));
    pis_init.p_src = MD_PIS_GPIO_PIN9;    /* production signal */
    pis_init.p_clk = MD_PIS_CLK_HCLK;     /* production clock zone */
    pis_init.p_edge = MD_PIS_EDGE_UP;     /* production effective edge */
    pis_init.p_output = MD_PIS_OUT_LEVEL; /* production signal type */
    pis_init.c_trig = MD_PIS_CH4_TIMER0_ITR0; /* comsumer signal */
    pis_init.c_clk = MD_PIS_CLK_PCLK;     /* comsumer clock zone */
    md_pis_init(&pis_init);
}

/**
  * @brief:  Initialize AD16C4T to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t ad16c4t_init;
    md_timer_oc_init_t oc_init;

    md_timer_base_struct_init(&ad16c4t_init);   /* initialize the ad16c4t_init  */
    ad16c4t_init.prescaler = 1;                 /* clk_count */
    ad16c4t_init.clk_div = MD_TIMER_CLOCK_DIV1; /* working clock of dead time and filter */
    ad16c4t_init.mode = MD_TIMER_CNT_MODE_UP;   /* count up */
    ad16c4t_init.period = 48000 - 1;            /* period */
    ad16c4t_init.re_cnt = 1 - 1;                /* 0 repeat count */
    md_timer_base_set_config(AD16C4T, &ad16c4t_init);

    md_timer_oc_struct_init(&oc_init);
    oc_init.ocstate = MD_TIMER_OCSTATE_ENABLE;    /* enable CHxO compare output */
    oc_init.oc_mode = MD_TIMER_OC_MODE_PWM2;      /* compare output mode is PWM2 */
    oc_init.oc_polarity = MD_TIMER_OC_POLARITY_HIGH;    /* CHxO compare output polarity is high */
    oc_init.ocn_polarity = MD_TIMER_OCN_POLARITY_HIGH;  /* CHxON compare output polarity is high */
    oc_init.oc_idle = MD_TIMER_OC_IDLE_RESET;       /* CHxO idle is low */
    oc_init.ocn_idle = MD_TIMER_OCN_IDLE_RESET;     /* CHxON idle is low */
    oc_init.pulse = 27000 - 1;                      /* CH1 compare value is 27000 - 1 */
    md_timer_oc_init(AD16C4T, MD_TIMER_CHANNEL_1, &oc_init);

    md_timer_set_slave_mode_smods(AD16C4T, 6);        /* 0x6: Trigger Mode */
    md_timer_set_slave_trigger_tssel(AD16C4T, 0);     /* 0x0: Internal Trigger 0 */

    md_timer_enable_cc1it_interrupt(AD16C4T);   /* enable CC1 interrupt request */
    md_timer_enable_uit_interrupt(AD16C4T);     /* update interrupt request */

    csi_vic_enable_sirq(AD16C4T_UP_TRIG_COM_IRQn);
    csi_vic_enable_sirq(AD16C4T_CC_IRQn);
    __enable_irq();

    /* enable AD16C4T */
    md_timer_enable_onepulse_spmen(AD16C4T);           /* enable single pulse */
    md_timer_set_main_output_enable(AD16C4T, 1);       /* enable output */
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    /* Initialize pis */
    init_pis();
    /* Initialize pin */
    init_gpio();
    /* Initialize timer */
    init_timer();

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
