/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    3 Jan. 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          3 Jan. 2023     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initialize TOGGLE IO.
  * @param  None
  * @retval None
  */
static void init_gpio(void)
{
    md_gpio_init_t gpio_init;

    /* Initialize TOGGLE IO as output */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_OUTPUT; /* output */
    gpio_init.odos = MD_GPIO_PUSH_PULL;   /* push pull */
    gpio_init.pupd = MD_GPIO_FLOATING;    /* no pull up and pull donw */
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_1;      /* GPIO function 1 */
    md_gpio_init(TOGGLE_PORT, TOGGLE_PIN, &gpio_init);

    /* Initialize ETR IO as input */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_INPUT;  /* input */
    gpio_init.odos = MD_GPIO_PUSH_PULL;   /* push pull */
    gpio_init.pupd = MD_GPIO_FLOATING;    /* no pull up and pull donw */
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_4;      /* GPIO function 4 */
    md_gpio_init(ET_PORT, ET_PIN, &gpio_init);
}

/**
  * @brief:  Initialize AD16C4T to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t ad16c4t_init;

    md_timer_base_struct_init(&ad16c4t_init);   /* initialize the ad16c4t_init  */
    ad16c4t_init.prescaler = 1 - 1;             /* clk_count: no prescaler */
    ad16c4t_init.clk_div = MD_TIMER_CLOCK_DIV1; /* working clock of dead time and filter */
    ad16c4t_init.mode = MD_TIMER_CNT_MODE_UP;   /* count up */
    ad16c4t_init.period = 100 - 1;              /* period is 100 count clock */
    ad16c4t_init.re_cnt = 1 - 1;                /* 0 repeat count */
    md_timer_base_set_config(AD16C4T, &ad16c4t_init);

    md_timer_enable_external_clk2mode_ecm2en(AD16C4T); /* enable external clock2 mode */
    md_timer_set_external_trigger_prescaler_etpsel(AD16C4T, 0); /* 0x0: Prescaler OFF */
    md_timer_set_external_trigger_polarity_etpol(AD16C4T, 0); /* 0x0: active at rising edge. */
    md_timer_set_external_trigger_filter_etflt(AD16C4T, 0); /* 0x0:No filter */

    md_timer_enable_uit_interrupt(AD16C4T);    /* enable updata interrupt */
    csi_vic_enable_sirq(AD16C4T_UP_TRIG_COM_IRQn);
    __enable_irq();

    /* enable AD16C4T */
    md_timer_enable_counter_cnten(AD16C4T);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC8M, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);

    /* Enable ALL peripheral */
    MD_SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    MD_SYSCFG_LOCK();

    /* Initialize the gpio */
    init_gpio();

    /* Initialize timer_AD16C4T */
    init_timer();

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
